package com.biz.crm.kms.business.direct.store.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.kms.business.direct.sdk.service.DirectVoService;
import com.biz.crm.kms.business.direct.sdk.vo.DirectVo;
import com.biz.crm.kms.business.direct.store.local.entity.DirectStoreEntity;
import com.biz.crm.kms.business.direct.store.local.repository.DirectStoreRepository;
import com.biz.crm.kms.business.direct.store.local.service.DirectStoreService;
import com.biz.crm.kms.business.direct.store.sdk.constant.StoreConstant;
import com.biz.crm.kms.business.direct.store.sdk.dto.DirectStoreConditionDto;
import com.biz.crm.kms.business.direct.store.sdk.dto.DirectStoreDto;
import com.biz.crm.kms.business.direct.store.sdk.vo.DirectStoreVo;
import com.biz.crm.kms.business.grab.rule.local.model.DirectStoreConditionModel;
import com.biz.crm.kms.business.supermarket.sdk.service.SupermarketVoService;
import com.biz.crm.kms.business.supermarket.sdk.vo.SupermarketVo;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalVo;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.dto.MasterDataMdgBaseDto;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.service.MasterDataMdgService;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.vo.MasterDataMdgKmsDirectStoreVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 直营门店表服务实现类
 *
 * @author ning.zhang
 * @date 2022-09-28 10:28:05
 */
@Slf4j
@Service("directStoreService")
public class DirectStoreServiceImpl implements DirectStoreService {

  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Autowired
  private DirectStoreRepository directStoreRepository;

  @Autowired(required = false)
  private MasterDataMdgService masterDataMdgService;

  @Autowired(required = false)
  private RedisMutexService redisMutexService;

  @Autowired(required = false)
  private TerminalVoService terminalVoService;

  @Autowired(required = false)
  private DirectVoService directVoService;

  @Autowired(required = false)
  private SupermarketVoService supermarketVoService;

  @Autowired(required = false)
  private LoginUserService loginUserService;

  @Autowired(required = false)
  private DirectStoreVoServiceImpl directStoreVoServiceImpl;


  @Override
  @Transactional
  public DirectStoreEntity create(DirectStoreDto dto) {
    this.createValidation(dto);
    DirectStoreEntity entity = this.nebulaToolkitService.copyObjectByBlankList(dto, DirectStoreEntity.class, HashSet.class, ArrayList.class);
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    this.directStoreRepository.save(entity);
    return entity;
  }

  @Override
  @Transactional
  public DirectStoreEntity update(DirectStoreDto dto) {
    this.updateValidation(dto);
    DirectStoreConditionModel model = new DirectStoreConditionModel();
    model.setIds(Sets.newHashSet(dto.getId()));
    model.setTenantCode(dto.getTenantCode());
    List<DirectStoreEntity> entities = this.directStoreRepository.findByDirectStoreConditionModel(model);
    Validate.isTrue(!CollectionUtils.isEmpty(entities), "直营门店信息不存在");
    DirectStoreEntity updateEntity = this.nebulaToolkitService.copyObjectByBlankList(dto, DirectStoreEntity.class, HashSet.class, ArrayList.class);
    this.directStoreRepository.updateById(updateEntity);
    return updateEntity;
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<DirectStoreEntity> entities = this.directStoreRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据删除个数不匹配");
    this.directStoreRepository.updateDelFlagByIds(ids);
  }

  /**
   * 直营门店分页查询接口
   * */
  @Override
  public Page<DirectStoreVo> findByConditions(Pageable pageable, DirectStoreConditionDto dto) {
    if (pageable == null) {
      pageable = PageRequest.of(1, 50);
    }
    if (dto == null) {
      dto = new DirectStoreConditionDto();
    }

    /** 获取租户编号 */
    if (StringUtils.isBlank(dto.getTenantCode())) {
      dto.setTenantCode(TenantUtils.getTenantCode());
    }

    /** 获取删除标志 */
    if (StringUtils.isBlank(dto.getDelFlag())) {
      dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    }

    Page<DirectStoreVo> byConditions = this.directStoreRepository.findByConditions(pageable, dto);
    List<DirectStoreVo> directStoreVos = byConditions.getRecords();
    this.directStoreVoServiceImpl.buildExtendInfo(directStoreVos);
    byConditions.setRecords(directStoreVos);
    return byConditions;
  }

  @DynamicTaskService(cornExpression = "0 30 22 * * ?", taskDesc = "自动拉取MDG直营门店数据")
  public void authPullKmsDirectStoreList(){
    MasterDataMdgBaseDto dto = new MasterDataMdgBaseDto();
    this.pullKmsDirectStoreList(dto);
  }

  /**
   * 批量拉取 MDG 直营门店数据
   */
  @Override
  public void pullKmsDirectStoreList(MasterDataMdgBaseDto dto) {
    loginUserService.refreshAuthentication(null);
    AbstractCrmUserIdentity loginUser = loginUserService.getAbstractLoginUser();
    if (ObjectUtils.isEmpty(dto)) {
      dto = new MasterDataMdgBaseDto();
    }
    if (StringUtils.isEmpty(dto.getPageNum())) {
      dto.setPageNum(StoreConstant.STORE_PAGE_NUM.toString());
    }
    if (StringUtils.isEmpty(dto.getPageSize())) {
      dto.setPageSize(StoreConstant.STORE_PAGE_SIZE.toString());
    }
    if (StringUtils.isEmpty(dto.getDs())) {
      String date = DateUtil.getDate("yyyyMMdd");
      Integer i = Integer.parseInt(date) - 1;
      dto.setDs(i.toString());
    }
    boolean lock = redisMutexService.tryLock(StoreConstant.STORE_PULL_LOCK, TimeUnit.SECONDS, StoreConstant.STORE_REDIS_TIME);
    try {
        List<DirectVo> allDirect = this.directVoService.findAllDirect();
        Map<String,List<DirectVo>> directMap = Maps.newHashMap();
        List<DirectVo> directVos = Lists.newArrayList();
        for(DirectVo directVo : allDirect){
          //如果业务单元是垂直重客，则添加到MAP中，该接口只用于垂直主数据对接
          if("DY00000010".equals(directVo.getBusinessUnitCode()) && StringUtils.isNotEmpty(directVo.getCustomerRetailerName())){
              if(CollectionUtils.isEmpty(directMap.get(directVo.getCustomerRetailerName()))){
                directVos = Lists.newArrayList();
                directVos.add(directVo);
                directMap.put(directVo.getCustomerRetailerName(),directVos);
              }else{
                directVos = directMap.get(directVo.getCustomerRetailerName());
                directVos.add(directVo);
                directMap.put(directVo.getCustomerRetailerName(),directVos);
              }
            }
        }
//        List<SupermarketVo> allSupermarket = this.supermarketVoService.findAllSupermarket();
//        Map<String,SupermarketVo> supermarketMap = Maps.newHashMap();
//        for(SupermarketVo supermarketVo : allSupermarket){
//            supermarketMap.put(supermarketVo.getCustomerRetailerName(),supermarketVo);
//        }
        this.pullKmsDirectStore(dto,directMap);
    }catch (Exception e){
      log.error("拉取直营门店数据错误",e);
    }finally {
      if(lock){
        redisMutexService.unlock(StoreConstant.STORE_PULL_LOCK);
      }
    }
  }

  @Override
  public List<DirectStoreEntity> findSoldToPartyByDirectCode(DirectStoreDto dto) {
    return this.directStoreRepository.findSoldToPartyByDirectCode(dto);
  }

  /**
   * 直营门店数据拉取并转换
   * 1、获取需同步的数据
   * 2、遍历数据，组装门店编码集合，查询对应的门店详情
   * 3、组装门店数据
   * 4、查询现有的直营门店数据，并组装
   * 5、遍历需同步的数据，设置对应的值
   * 6、新增或更新数据
   * 7、递归调用，查询下一页的数据
   */
  private void pullKmsDirectStore(MasterDataMdgBaseDto dto,Map<String,List<DirectVo>> directMap){
    //1
    List<MasterDataMdgKmsDirectStoreVo> directStoreVos = masterDataMdgService.pullKmsDirectStoreList(dto);
    if(!CollectionUtils.isEmpty(directStoreVos)){
      Integer currentCount = directStoreVos.size();
      //2
      List<DirectStoreEntity> storeList = Lists.newArrayList();
      Set<String> storeCodes = Sets.newHashSet();
      List<MasterDataMdgKmsDirectStoreVo> oldStoreVos = Lists.newArrayList();
      MasterDataMdgKmsDirectStoreVo oldStoreVo = new MasterDataMdgKmsDirectStoreVo();
      for(MasterDataMdgKmsDirectStoreVo directStoreVo : directStoreVos){
        if(StringUtils.isNotEmpty(directStoreVo.getOldRetailerStoreCode()) && !StringUtils.equals(directStoreVo.getRetailerStoreCode(),directStoreVo.getOldRetailerStoreCode())){
          storeCodes.add(directStoreVo.getOldRetailerStoreCode());
          oldStoreVo = new MasterDataMdgKmsDirectStoreVo();
          oldStoreVo.setRetailerName(directStoreVo.getRetailerName());
          oldStoreVo.setRetailerRegion(directStoreVo.getRetailerRegion());
          oldStoreVo.setStoreCode(directStoreVo.getStoreCode());
          oldStoreVo.setStoreName(directStoreVo.getStoreName());
          oldStoreVo.setRetailerStoreCode(directStoreVo.getOldRetailerStoreCode());
          oldStoreVos.add(oldStoreVo);
        }
        storeCodes.add(directStoreVo.getStoreCode());
      }
      if(!CollectionUtils.isEmpty(oldStoreVos)){
        directStoreVos.addAll(oldStoreVos);
      }
      List<String> storeCodeList = Lists.newArrayList(storeCodes);
      List<TerminalVo> terminalVos = this.terminalVoService.findMainDetailsByTerminalCodesUsePost(storeCodeList);
      //3
      Map<String,TerminalVo> terminalVoMap = Maps.newHashMap();
      if(!CollectionUtils.isEmpty(terminalVos)){
        for(TerminalVo terminalVo : terminalVos){
          terminalVoMap.put(terminalVo.getTerminalCode(),terminalVo);
        }
      }
      //4
      DirectStoreConditionModel model = new DirectStoreConditionModel();
      model.setTerminalCodes(storeCodes);
      model.setTenantCode(TenantUtils.getTenantCode());
      List<DirectStoreEntity> directStoreEntityList = this.directStoreRepository.findByDirectStoreConditionModel(model);
      Map<String,DirectStoreEntity> directStoreMap = Maps.newHashMap();
      if(!CollectionUtils.isEmpty(directStoreEntityList)){
        for(DirectStoreEntity directStoreEntity : directStoreEntityList){
          directStoreMap.put(directStoreEntity.getDirectCode()+"-"+directStoreEntity.getTerminalCode()+"-"+directStoreEntity.getSupermarketStoreCode()+"-"+directStoreEntity.getRetailerRegion(),directStoreEntity);
        }
      }
      Map<String,String> map = Maps.newHashMap();
      DirectStoreEntity store = new DirectStoreEntity();
      //5
      for(MasterDataMdgKmsDirectStoreVo directStoreVo : directStoreVos){
        if (!CollectionUtils.isEmpty(directMap.get(directStoreVo.getRetailerName()))) {
          List<DirectVo> directVoList =  directMap.get(directStoreVo.getRetailerName());
          for(DirectVo directVo : directVoList){
            store = new DirectStoreEntity();
            if(map.get(directVo.getDirectCode()+"-"+directStoreVo.getStoreCode()+"-"+directStoreVo.getRetailerStoreCode()+"-"+directStoreVo.getRetailerRegion())==null){
              map.put(directVo.getDirectCode()+"-"+directStoreVo.getStoreCode()+"-"+directStoreVo.getRetailerStoreCode()+"-"+directStoreVo.getRetailerRegion(),"1");
            }else{
              continue;
            }
            if(terminalVoMap.get(directStoreVo.getStoreCode())!=null) {
              store.setSoldToPartyCode(terminalVoMap.get(directStoreVo.getStoreCode()).getSellerCode());
              store.setSoldToPartyName(terminalVoMap.get(directStoreVo.getStoreCode()).getSellerName());
              store.setProvinceCode(terminalVoMap.get(directStoreVo.getStoreCode()).getProvinceCode());
              store.setProvinceName(terminalVoMap.get(directStoreVo.getStoreCode()).getProvinceName());
              store.setBusinessArea(terminalVoMap.get(directStoreVo.getStoreCode()).getRegionCode());
            }else{
//          store.setDirectCode("11111111");
              continue;
            }
            store.setDirectCode(directVo.getDirectCode());
            DirectStoreEntity oldStore = directStoreMap.get(store.getDirectCode()+"-"+directStoreVo.getStoreCode()+"-"+directStoreVo.getRetailerStoreCode()+"-"+directStoreVo.getRetailerRegion());
            if(oldStore != null){
              store.setId(oldStore.getId());
            }
            store.setSupermarketStoreCode(directStoreVo.getRetailerStoreCode());
            store.setTerminalCode(directStoreVo.getStoreCode());
            store.setTerminalName(directStoreVo.getStoreName());
            store.setTenantCode(TenantUtils.getTenantCode());
            if(StringUtils.isNotEmpty(directStoreVo.getRetailerRegion())){
              store.setRetailerRegion(directStoreVo.getRetailerRegion());
            }
            storeList.add(store);
          }
        }else{
          continue;
        }
      }
      //6
      this.directStoreRepository.saveOrUpdateBatch(storeList);
      //7
      if(currentCount == 1000){
        Integer pageNum = Integer.parseInt(dto.getPageNum());
        pageNum = pageNum+1;
        dto.setPageNum(pageNum.toString());
        this.pullKmsDirectStore(dto,directMap);
      }else{
        return;
      }
    }
  }

  /**
   * 在创建DirectStore模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(DirectStoreDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getDirectCode(), "缺失系统编码");
    Validate.notBlank(dto.getSupermarketStoreCode(), "缺失商超门店编码");
    Validate.notBlank(dto.getTerminalCode(), "缺失送达方编码");
    Validate.notBlank(dto.getTerminalName(), "缺失送达方名称");
    Validate.isTrue(Objects.nonNull(dto.getTimeThreshold()), "缺失对账时间阈值");
    DirectStoreConditionModel model = new DirectStoreConditionModel();
    model.setTenantCode(dto.getTenantCode());
    model.setDirectCode(dto.getDirectCode());
    model.setSupermarketStoreCode(dto.getSupermarketStoreCode());
    model.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    List<DirectStoreEntity> entities = this.directStoreRepository.findByDirectStoreConditionModel(model);
    Validate.isTrue(CollectionUtils.isEmpty(entities), "客户门店已存在关联企业门店");
  }

  /**
   * 在修改DirectStore模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(DirectStoreDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getId(), "修改信息时，id不能为空！");
    Validate.notBlank(dto.getDirectCode(), "缺失系统编码");
    Validate.notBlank(dto.getSupermarketStoreCode(), "缺失商超门店编码");
    Validate.notBlank(dto.getTerminalCode(), "缺失送达方编码");
    Validate.notBlank(dto.getTerminalName(), "缺失送达方名称");
    Validate.isTrue(Objects.nonNull(dto.getTimeThreshold()), "缺失对账时间阈值");
    DirectStoreConditionModel model = new DirectStoreConditionModel();
    model.setTenantCode(dto.getTenantCode());
    model.setDirectCode(dto.getDirectCode());
    model.setSupermarketStoreCode(dto.getSupermarketStoreCode());
    model.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    List<DirectStoreEntity> entities = this.directStoreRepository.findByDirectStoreConditionModel(model);
    if (!CollectionUtils.isEmpty(entities)) {
      entities = entities.stream().filter(directStoreEntity -> !dto.getId().equals(directStoreEntity.getId())).collect(Collectors.toList());
    }
    Validate.isTrue(CollectionUtils.isEmpty(entities), "客户门店已存在关联企业门店");
  }
}
