package com.biz.crm.kms.business.document.capture.log.local.service.internal;


import com.biz.crm.kms.business.document.capture.log.local.entity.DocumentCaptureLogEntity;
import com.biz.crm.kms.business.document.capture.log.local.repository.DocumentCaptureLogRepository;
import com.biz.crm.kms.business.document.capture.log.local.service.DocumentCaptureLogService;
import com.biz.crm.kms.business.document.capture.log.sdk.enums.OrderTypeEnum;
import com.biz.crm.kms.business.document.capture.log.sdk.constant.DocumentCaptureLogConstant;
import com.biz.crm.kms.business.document.capture.log.sdk.dto.DocumentCaptureLogQueryDto;
import com.biz.crm.kms.business.document.capture.log.sdk.vo.LogNumShowVo;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

/**
 * (documentCaptureLogServiceImpl)表服务实现类
 *
 * @author hd
 * @since 2023-01-10 11:05:25
 */
@Slf4j
@Service("documentCaptureLogService")
public class DocumentCaptureLogServiceImpl implements DocumentCaptureLogService {

    @Autowired(required = false)
    private DocumentCaptureLogRepository documentCaptureLogRepository;

    /**
     * 分页查询所有数据
     *
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据
     */
    @Override
    public LogNumShowVo findByConditions(Pageable pageable, DocumentCaptureLogQueryDto dto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        dto = ObjectUtils.defaultIfNull(dto, new DocumentCaptureLogQueryDto());
        //开始时间
        String queryStartDate = dto.getQueryStartDate();
        //结束时间
        String queryEndDate = dto.getQueryEndDate();
        //new 返回对象
        LogNumShowVo logNumShowVo = new LogNumShowVo();
        dto.setInvoiceType(this.transOrderType(dto.getInvoiceType()));
        //判断操作菜单是否为空
        //如果是空的
        if (StringUtils.isEmpty(dto.getOperateMenu())) {
//            dto = dateData(dto, queryStartDate, queryEndDate);
            Integer grabNum = documentCaptureLogRepository.findGrabNum(dto);
            logNumShowVo.setGrabNum(grabNum);
            logNumShowVo.setTotal(grabNum);
            if (ObjectUtils.isNotEmpty(grabNum) && grabNum != 0) {
                Integer successNum = documentCaptureLogRepository.findSuccessNum(dto);
                logNumShowVo.setGrabSuccessRate(new BigDecimal(successNum).divide((new BigDecimal(grabNum)),2,BigDecimal.ROUND_HALF_UP));
            }
            logNumShowVo.setAbnormalAccount(documentCaptureLogRepository.findAbnormalAccount(dto));
            logNumShowVo.setAbnormalGrabTask(documentCaptureLogRepository.findAbnormalGrabTask(dto));
            logNumShowVo.setMatchFailStore(documentCaptureLogRepository.findAcceptanceCount(dto));
            logNumShowVo.setMatchFailProduct(documentCaptureLogRepository.findAcceptance(dto));
            logNumShowVo.setDocumentCaptureLogVos(documentCaptureLogRepository.findByConditions(pageable, dto).getRecords());
            return logNumShowVo;
        } else {
            String operateMenu = dto.getOperateMenu();
            Validate.notBlank(operateMenu, "操作菜单不能为空");
//            dto = dateData(dto, queryStartDate, queryEndDate);
            if (StringUtils.isNotEmpty(queryStartDate) && StringUtils.isNotEmpty(queryEndDate)) {
                Validate.isTrue(!DateUtil.parseDate(queryStartDate, DateUtil.DEFAULT_YEAR_MONTH_DAY)
                        .after(DateUtil.parseDate(queryEndDate, DateUtil.DEFAULT_YEAR_MONTH_DAY)), "开始时间不能在结束时间之后！");
            }
            String invoiceType = dto.getInvoiceType();
            if (StringUtils.isEmpty(invoiceType)) {
                invoiceType = DocumentCaptureLogConstant.ACCEPTANCE;
            }
            if (DocumentCaptureLogConstant.GRAB_NUM.equals(operateMenu)) {
                Integer grabNum = documentCaptureLogRepository.findGrabNum(dto);
                logNumShowVo.setGrabNum(grabNum);
                logNumShowVo.setTotal(grabNum);
                logNumShowVo.setDocumentCaptureLogVos(documentCaptureLogRepository.findByConditions(pageable, dto).getRecords());
            } else if (DocumentCaptureLogConstant.GRAB_SUCCESS_RATE.equals(operateMenu)) {
                Integer grabNum = documentCaptureLogRepository.findGrabNum(dto);
                logNumShowVo.setTotal(grabNum);
                if (ObjectUtils.isNotEmpty(grabNum) && grabNum != 0) {
                    Integer successNum = documentCaptureLogRepository.findSuccessNum(dto);
                    logNumShowVo.setGrabSuccessRate(new BigDecimal(successNum).divide((new BigDecimal(grabNum)
                            .multiply(new BigDecimal(100))), 1, BigDecimal.ROUND_HALF_UP));
                }
//                logNumShowVo.setDocumentCaptureLogVos(documentCaptureLogRepository.findBySuccess(pageable, dto).getRecords());
                logNumShowVo.setDocumentCaptureLogVos(documentCaptureLogRepository.findByConditions(pageable, dto).getRecords());
            } else if (DocumentCaptureLogConstant.ABNORMAL_ACCOUNT.equals(operateMenu)) {
                Integer abnormalAccount = documentCaptureLogRepository.findAbnormalAccount(dto);
                logNumShowVo.setAbnormalAccount(abnormalAccount);
                logNumShowVo.setTotal(abnormalAccount);
                logNumShowVo.setDocumentCaptureLogVos(documentCaptureLogRepository.findByAbnormal(pageable, dto).getRecords());
            } else if (DocumentCaptureLogConstant.ABNORMAL_GRAB_TASK.equals(operateMenu)) {
                Integer abnormalGrabTask = documentCaptureLogRepository.findAbnormalGrabTask(dto);
                logNumShowVo.setAbnormalGrabTask(abnormalGrabTask);
                logNumShowVo.setTotal(abnormalGrabTask);
                logNumShowVo.setDocumentCaptureLogVos(documentCaptureLogRepository.findByGrabTask(pageable, dto).getRecords());
            } else if (DocumentCaptureLogConstant.MATCH_FAIL_STORE.equals(operateMenu)) {
                logNumShowVo = failStore(pageable, dto, logNumShowVo, invoiceType);
            } else {
                logNumShowVo = failProduct(pageable, dto, logNumShowVo, invoiceType);
            }
        }
        return logNumShowVo;
    }

    private DocumentCaptureLogQueryDto dateData(DocumentCaptureLogQueryDto dto, String queryStartDate, String queryEndDate) {
        if (StringUtils.isEmpty(queryStartDate) && StringUtils.isEmpty(queryEndDate)) {
            String startDate = DateUtil.format(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY).concat(" 00:00:00");
            String date = DateUtil.format(DateUtil.ReturnTheDay(new Date(),1), DateUtil.DEFAULT_YEAR_MONTH_DAY).concat(" 00:00:00");
            dto.setQueryStartDate(startDate);
            dto.setQueryEndDate(date);
        }else if (StringUtils.isNotEmpty(queryEndDate)){
            try {
                Date date = DateUtil.date_yyyy_MM_dd_HH_mm_ss.parse(queryEndDate);
                queryEndDate = DateUtil.format(DateUtil.ReturnTheDay(date,1), DateUtil.DEFAULT_DATE_ALL_PATTERN);
            } catch (ParseException e) {
                e.printStackTrace();
            }
            dto.setQueryEndDate(queryEndDate);
        }
        return dto;
    }

    /**
     * 匹配失败门店
     */
    private LogNumShowVo failStore(Pageable pageable, DocumentCaptureLogQueryDto dto, LogNumShowVo logNumShowVo, String invoiceType) {
        if (OrderTypeEnum.ACCEPTANCE.getDesc().equals(invoiceType)) {
            Integer acceptanceCount = documentCaptureLogRepository.findAcceptanceCount(dto);
            logNumShowVo.setMatchFailStore(acceptanceCount);
            logNumShowVo.setTotal(acceptanceCount);
            logNumShowVo.setMatchLogVos(documentCaptureLogRepository.findAcceptanceCountData(pageable, dto).getRecords());
        } else if (OrderTypeEnum.EXPENSE.getDesc().equals(invoiceType)) {
            Integer expenseCount = documentCaptureLogRepository.findExpenseCount(dto);
            logNumShowVo.setMatchFailStore(expenseCount);
            logNumShowVo.setTotal(expenseCount);
            logNumShowVo.setMatchLogVos(documentCaptureLogRepository.findExpenseCountData(pageable, dto).getRecords());
        } else if (OrderTypeEnum.STATEMENT.getDesc().equals(invoiceType)) {
            Integer statementCount = documentCaptureLogRepository.findStatementCount(dto);
            logNumShowVo.setMatchFailStore(statementCount);
            logNumShowVo.setTotal(statementCount);
            logNumShowVo.setMatchLogVos(documentCaptureLogRepository.findStatementCountData(pageable, dto).getRecords());
        } else if (OrderTypeEnum.STOCK.getDesc().equals(invoiceType)) {
            Integer stockCount = documentCaptureLogRepository.findStockCount(dto);
            logNumShowVo.setMatchFailStore(stockCount);
            logNumShowVo.setTotal(stockCount);
            logNumShowVo.setMatchLogVos(documentCaptureLogRepository.findStockCountData(pageable, dto).getRecords());
        } else if (OrderTypeEnum.RETURN.getDesc().equals(invoiceType)) {
            Integer returnCount = documentCaptureLogRepository.findReturnCount(dto);
            logNumShowVo.setMatchFailStore(returnCount);
            logNumShowVo.setTotal(returnCount);
            logNumShowVo.setMatchLogVos(documentCaptureLogRepository.findReturnCountData(pageable, dto).getRecords());
        } else if (OrderTypeEnum.SALE.getDesc().equals(invoiceType)){
            Integer salesCount = documentCaptureLogRepository.findSalesCount(dto);
            logNumShowVo.setMatchFailStore(salesCount);
            logNumShowVo.setTotal(salesCount);
            logNumShowVo.setMatchLogVos(documentCaptureLogRepository.findSalesCountData(pageable, dto).getRecords());
        }
        return logNumShowVo;
    }

    /**
     * 匹配失败产品
     */
    private LogNumShowVo failProduct(Pageable pageable, DocumentCaptureLogQueryDto dto, LogNumShowVo logNumShowVo, String invoiceType) {
        if (OrderTypeEnum.ACCEPTANCE.getDesc().equals(invoiceType)) {
            Integer acceptance = documentCaptureLogRepository.findAcceptance(dto);
            logNumShowVo.setMatchFailStore(acceptance);
            logNumShowVo.setTotal(acceptance);
            logNumShowVo.setMatchLogVos(documentCaptureLogRepository.findAcceptanceData(pageable, dto).getRecords());
        } else if (OrderTypeEnum.EXPENSE.getDesc().equals(invoiceType)) {
            Integer expense = documentCaptureLogRepository.findExpense(dto);
            logNumShowVo.setMatchFailStore(expense);
            logNumShowVo.setTotal(expense);
            logNumShowVo.setMatchLogVos(documentCaptureLogRepository.findExpenseData(pageable, dto).getRecords());
        } else if (OrderTypeEnum.STATEMENT.getDesc().equals(invoiceType)) {
            Integer statementCount = documentCaptureLogRepository.findStatementCount(dto);
            logNumShowVo.setMatchFailStore(statementCount);
            logNumShowVo.setTotal(statementCount);
            logNumShowVo.setMatchLogVos(documentCaptureLogRepository.findStatementData(pageable, dto).getRecords());
        } else if (OrderTypeEnum.STOCK.getDesc().equals(invoiceType)) {
            Integer stock = documentCaptureLogRepository.findStock(dto);
            logNumShowVo.setMatchFailStore(stock);
            logNumShowVo.setTotal(stock);
            logNumShowVo.setMatchLogVos(documentCaptureLogRepository.findStockData(pageable, dto).getRecords());
        } else if (OrderTypeEnum.RETURN.getDesc().equals(invoiceType)) {
            Integer aReturn = documentCaptureLogRepository.findReturn(dto);
            logNumShowVo.setMatchFailStore(aReturn);
            logNumShowVo.setTotal(aReturn);
            logNumShowVo.setMatchLogVos(documentCaptureLogRepository.findReturnData(pageable, dto).getRecords());
        } else if (OrderTypeEnum.SALE.getDesc().equals(invoiceType)){
            Integer sales = documentCaptureLogRepository.findSales(dto);
            logNumShowVo.setMatchFailStore(sales);
            logNumShowVo.setTotal(sales);
            logNumShowVo.setMatchLogVos(documentCaptureLogRepository.findSalesData(pageable, dto).getRecords());
        }
        return logNumShowVo;
    }

    private String transOrderType(String orderTypeCode){
        if (StringUtils.isEmpty(orderTypeCode)){
            return null;
        }
        String orderTypeDesc = null;
        if (OrderTypeEnum.ACCEPTANCE.getDictCode().equals(orderTypeCode)){
            orderTypeDesc = OrderTypeEnum.ACCEPTANCE.getDesc();
        }else if (OrderTypeEnum.RETURN.getDictCode().equals(orderTypeCode)){
            orderTypeDesc = OrderTypeEnum.RETURN.getDesc();
        }else if (OrderTypeEnum.EXPENSE.getDictCode().equals(orderTypeCode)){
            orderTypeDesc = OrderTypeEnum.EXPENSE.getDesc();
        }else if (OrderTypeEnum.STATEMENT.getDictCode().equals(orderTypeCode)){
            orderTypeDesc = OrderTypeEnum.STATEMENT.getDesc();
        }else if (OrderTypeEnum.STOCK.getDictCode().equals(orderTypeCode)){
            orderTypeDesc = OrderTypeEnum.STOCK.getDesc();
        }else if (OrderTypeEnum.SALE.getDictCode().equals(orderTypeCode)){
            orderTypeDesc = OrderTypeEnum.SALE.getDesc();
        }
        return orderTypeDesc;
    }

    /**
     * 通过日志编号查询单条数据
     *
     * @param logCode 日志编号
     * @return 单条数据
     */
    @Override
    public DocumentCaptureLogEntity findByLogCode(String logCode) {
        if (StringUtils.isBlank(logCode)) {
            return null;
        }
        return this.documentCaptureLogRepository.findByLogCode(logCode);
    }
    /**
     * 每天晚上11点删除三月前的日志的定时任务
     */
    @DynamicTaskService(cornExpression = "0 0 23 * * ?", taskDesc = "删除三月前的日志")
    @Override
    public void deleteLog() {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        Calendar c = Calendar.getInstance();
        c.setTime(new Date());
        c.add(Calendar.MONTH, -3);
        String three = format.format(c.getTime());
        this.documentCaptureLogRepository.deleteLogByDate(three);
    }
}

