package com.biz.crm.kms.business.grab.rule.local.service.internal;

import cn.hutool.core.util.IdUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.kms.business.direct.account.sdk.service.DirectAccountVoService;
import com.biz.crm.kms.business.direct.account.sdk.vo.DirectAccountVo;
import com.biz.crm.kms.business.direct.sdk.service.DirectVoService;
import com.biz.crm.kms.business.direct.sdk.vo.DirectVo;
import com.biz.crm.kms.business.document.capture.log.sdk.dto.GrabRequestSynchronizeDto;
import com.biz.crm.kms.business.document.capture.log.sdk.service.DocumentCaptureLogSynchronizeService;
import com.biz.crm.kms.business.grab.rule.local.entity.GrabRequestEntity;
import com.biz.crm.kms.business.grab.rule.local.entity.GrabRuleEntity;
import com.biz.crm.kms.business.grab.rule.local.entity.GrabRuleStoreEntity;
import com.biz.crm.kms.business.grab.rule.local.repository.GrabRequestRepository;
import com.biz.crm.kms.business.grab.rule.local.service.GrabRequestService;
import com.biz.crm.kms.business.grab.rule.sdk.constant.GrabRuleConstant;
import com.biz.crm.kms.business.grab.rule.sdk.enums.GrabType;
import com.biz.crm.kms.business.grab.rule.sdk.vo.GrabRequestVo;
import com.biz.crm.kms.business.invoice.sdk.service.InvoiceTypeService;
import com.biz.crm.kms.business.invoice.sdk.vo.InvoiceTypeVo;
import com.biz.crm.kms.business.supermarket.parameter.sdk.service.SupermarketParameterGrabModeVoService;
import com.biz.crm.kms.business.supermarket.parameter.sdk.vo.SupermarketParameterGrabModeVo;
import com.biz.crm.kms.business.supermarket.store.sdk.service.SupermarketStoreVoService;
import com.biz.crm.kms.business.supermarket.store.sdk.vo.SupermarketStoreVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 抓单请求表服务实现类
 *
 * @author ning.zhang
 * @date 2022-09-27 14:29:47
 */
@Slf4j
@Service("grabRequestService")
public class GrabRequestServiceImpl implements GrabRequestService {

  @Value("${spring.profiles.active:}")
  private String profilesActive;
  @Autowired
  private GrabRequestRepository grabRequestRepository;
  @Autowired
  private SupermarketStoreVoService supermarketStoreVoService;
  @Autowired
  private DirectAccountVoService directAccountVoService;
  @Autowired
  private DirectVoService directVoService;
  @Autowired
  private InvoiceTypeService invoiceTypeService;
  @Autowired
  private SupermarketParameterGrabModeVoService supermarketParameterGrabModeVoService;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private DocumentCaptureLogSynchronizeService synchronizeService;

  @Override
  @Transactional
  public void createBatch(List<GrabRuleEntity> ruleEntities) {
    Map<String, SupermarketStoreVo> storeMap = this.buildStoreMap(ruleEntities);
    Map<String, DirectAccountVo> accountMap = this.buildAccountMap(ruleEntities);
    Map<String, DirectVo> directMap = this.buildDirectMap(ruleEntities);
    Map<String, InvoiceTypeVo> invoiceTypeMap = this.buildInvoiceTypeMap();
    Map<String, String> methodMap = this.buildMethodMap(directMap);
    List<GrabRequestEntity> requestEntities = Lists.newArrayList();
    Date nowDate = new Date();
    ruleEntities.forEach(ruleEntity -> {
      DirectVo directVo = directMap.getOrDefault(ruleEntity.getDirectCode(), new DirectVo());
      GrabRequestEntity requestEntity = new GrabRequestEntity();
      requestEntity.setTenantCode(ruleEntity.getTenantCode());
      requestEntity.setCreateAccount(ruleEntity.getCreateAccount());
      requestEntity.setCreateName(ruleEntity.getCreateName());
      requestEntity.setModifyAccount(ruleEntity.getModifyAccount());
      requestEntity.setModifyName(ruleEntity.getModifyName());
      requestEntity.setAuto(GrabType.getByDictCode(ruleEntity.getGrabType()).getOrder());
      requestEntity.setFile(0);
      requestEntity.setTime(new Date());
      requestEntity.setResponse("");
      requestEntity.setFail(0);
      requestEntity.setMethod(Integer.valueOf(methodMap.getOrDefault(String.format("%s%s", directVo.getSupermarketCode(), ruleEntity.getOrderType()), "0")));
      JSONObject requestJson = this.buildRequestJson(storeMap, accountMap, directMap, invoiceTypeMap, ruleEntity);
      requestEntity.setRequest(requestJson.toJSONString());
      requestEntity.setUuidLog(IdUtil.fastUUID().replace("-", ""));
      requestEntities.add(requestEntity);
      ruleEntity.setTaskTime(nowDate);
      ruleEntity.setTaskTimestamp(ruleEntity.getTaskTime().getTime());
    });
    this.grabRequestRepository.saveBatch(requestEntities);
    List<GrabRequestSynchronizeDto> requestVos = (List<GrabRequestSynchronizeDto>) nebulaToolkitService.copyCollectionByWhiteList(requestEntities, GrabRequestEntity.class, GrabRequestSynchronizeDto.class, LinkedHashSet.class, ArrayList.class);
    synchronizeService.synchronize(requestVos);
  }

  /**
   * 构建请求商超规则参数json
   *
   * @param storeMap       商超门店映射
   * @param accountMap     账号信息映射
   * @param directMap      系统映射
   * @param invoiceTypeMap 单据类型映射
   * @param ruleEntity     规则信息
   * @return 商超规则参数json
   */
  private JSONObject buildRequestJson(Map<String, SupermarketStoreVo> storeMap, Map<String, DirectAccountVo> accountMap
      , Map<String, DirectVo> directMap, Map<String, InvoiceTypeVo> invoiceTypeMap, GrabRuleEntity ruleEntity) {
    JSONObject inputFileJson = new JSONObject();
    JSONArray jsonArray = new JSONArray();
    jsonArray.add("");
    inputFileJson.put(GrabRuleConstant.FILE_NAMES, jsonArray);
    inputFileJson.put(GrabRuleConstant.FILE_TYPE, "url");

    JSONObject extendArgs = new JSONObject();
    extendArgs.put(GrabRuleConstant.GRPC_REQ_REQ_PARAM, this.buildReqParamJson(storeMap, accountMap, directMap, invoiceTypeMap, ruleEntity));
    extendArgs.put(GrabRuleConstant.GRPC_REQ_EXTEND_PARAM, this.buildExtendParamJson(ruleEntity));

    JSONObject body = new JSONObject();
    body.put(GrabRuleConstant.GRPC_REQ_EXTEND_ARGS, extendArgs);
    body.put(GrabRuleConstant.INPUT_FILE, inputFileJson);

    JSONObject requestJson = new JSONObject();
    requestJson.put(GrabRuleConstant.APP_ID, "ka_grab_order");
    requestJson.put(GrabRuleConstant.GRPC_REQ_BODY, body);
    return requestJson;
  }

  /**
   * 构建请求商超规则参数json
   *
   * @param storeMap       商超门店映射
   * @param accountMap     账号信息映射
   * @param directMap      系统映射
   * @param invoiceTypeMap 单据类型映射
   * @param ruleEntity     规则信息
   * @return 商超规则参数json
   */
  private JSONObject buildReqParamJson(Map<String, SupermarketStoreVo> storeMap, Map<String, DirectAccountVo> accountMap
      , Map<String, DirectVo> directMap, Map<String, InvoiceTypeVo> invoiceTypeMap, GrabRuleEntity ruleEntity) {
    DirectAccountVo accountVo = accountMap.getOrDefault(ruleEntity.getAccount(), new DirectAccountVo());
    DirectVo directVo = directMap.getOrDefault(ruleEntity.getDirectCode(), new DirectVo());
    JSONObject reqParmJson = new JSONObject();
    reqParmJson.put(GrabRuleConstant.KA_NAME, directVo.getSupermarketName());
    reqParmJson.put(GrabRuleConstant.DOC, invoiceTypeMap.get(ruleEntity.getOrderType()).getName());
    reqParmJson.put(GrabRuleConstant.USER_NAME, accountVo.getAccount());
    reqParmJson.put(GrabRuleConstant.PASS_WORD, accountVo.getPassword());
    reqParmJson.put(GrabRuleConstant.KMS_PATH_TYPE, String.format("%s%s", profilesActive, "环境"));
    reqParmJson.put(GrabRuleConstant.KMS_IP, "");
    reqParmJson.put(GrabRuleConstant.PRINCIPAL_ID, ruleEntity.getCreateName());
    // 'force'：'0' 表示会md5去重(自动抓单强制覆盖为md5去重)
    reqParmJson.put(GrabRuleConstant.FORCE, "0");
    // 增加租户ID，AI为了MD5去重
    reqParmJson.put(GrabRuleConstant.TENANTRY_ID, ruleEntity.getTenantCode());
    reqParmJson.put(GrabRuleConstant.RULE_PARAM, JSON.parseArray(ruleEntity.getRuleParam()));
    if (BooleanEnum.TRUE.getCapital().equals(ruleEntity.getAllStoreFlag())) {
      reqParmJson.put(GrabRuleConstant.KMS_ALL_STORE, "true");
    } else {
      reqParmJson.put(GrabRuleConstant.KMS_ALL_STORE, "false");
      reqParmJson.put(GrabRuleConstant.SOLD_TO_PARTY, ruleEntity.getStoreList().stream().map(grabRuleStoreEntity -> {
        JSONObject storeJson = new JSONObject();
        SupermarketStoreVo storeVo = storeMap.getOrDefault(grabRuleStoreEntity.getStoreCode(), new SupermarketStoreVo());
        storeJson.put(GrabRuleConstant.STORE_CODE, grabRuleStoreEntity.getStoreCode());
        storeJson.put(GrabRuleConstant.STORE_NAME, storeVo.getStoreName());
        return storeJson;
      }).collect(Collectors.toList()));
    }
    return reqParmJson;
  }

  /**
   * 构建请求扩展参数json
   *
   * @param ruleEntity 规则信息
   * @return 商超规则参数json
   */
  private JSONObject buildExtendParamJson(GrabRuleEntity ruleEntity) {
    JSONObject extendParmJson = new JSONObject();
    //租户ID
    extendParmJson.put(GrabRuleConstant.TENANTRY_ID, ruleEntity.getTenantCode());
    extendParmJson.put(GrabRuleConstant.RULE_NAME, ruleEntity.getRuleName());
    //系统编码
    extendParmJson.put(GrabRuleConstant.DIRECT_SYSTEM_ID, ruleEntity.getDirectCode());
    //单据类型编码
    extendParmJson.put(GrabRuleConstant.BS_INVOICE_ID, ruleEntity.getOrderType());
    extendParmJson.put(GrabRuleConstant.KA_FLAG, String.format("%s%s", profilesActive, "环境"));
    extendParmJson.put(GrabRuleConstant.TRACK_ID, String.join("<=>", GrabType.getByDictCode(ruleEntity.getGrabType()).getValue()
        , ruleEntity.getTenantCode()
        , ruleEntity.getRuleName()
        , GrabRuleConstant.YYYY_MM_DD_HH_MM_SS_NO_SPLIT.format(LocalDateTime.now())));
    extendParmJson.put(GrabRuleConstant.STYLE, GrabType.getByDictCode(ruleEntity.getGrabType()).getValue());
    extendParmJson.put(GrabRuleConstant.NICK_NAME, "CRM_KMS系统");
    return extendParmJson;
  }

  /**
   * 构建单据类型映射(key:单据类型,value:单据类型信息)
   *
   * @return 单据类型映射
   */
  private Map<String, InvoiceTypeVo> buildInvoiceTypeMap() {
    List<InvoiceTypeVo> invoiceTypeList = this.invoiceTypeService.findAll();
    if (CollectionUtils.isEmpty(invoiceTypeList)) {
      return Maps.newHashMap();
    }
    return invoiceTypeList.stream().collect(Collectors.toMap(InvoiceTypeVo::getType, t -> t, (a, b) -> a));
  }

  /**
   * 构建商超门店映射(key:商超门店编码,value:商超门店信息)
   *
   * @param ruleEntities 抓单规则列表
   * @return 商超门店映射
   */
  private Map<String, SupermarketStoreVo> buildStoreMap(List<GrabRuleEntity> ruleEntities) {
    Set<String> storeCodes = ruleEntities.stream()
        .filter(ruleEntity -> !CollectionUtils.isEmpty(ruleEntity.getStoreList()))
        .flatMap(ruleEntity -> ruleEntity.getStoreList().stream().map(GrabRuleStoreEntity::getStoreCode))
        .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(storeCodes)) {
      return Maps.newHashMap();
    }
    List<SupermarketStoreVo> storeList = this.supermarketStoreVoService.findByStoreCodes(storeCodes);
    if (CollectionUtils.isEmpty(storeList)) {
      return Maps.newHashMap();
    }
    return storeList.stream().collect(Collectors.toMap(SupermarketStoreVo::getStoreCode, t -> t, (a, b) -> a));
  }

  /**
   * 构建客户账号信息映射(key:客户账号,value:客户账号信息)
   *
   * @param ruleEntities 抓单规则列表
   * @return 客户账号信息映射
   */
  private Map<String, DirectAccountVo> buildAccountMap(List<GrabRuleEntity> ruleEntities) {
    Set<String> accounts = ruleEntities.stream().map(GrabRuleEntity::getAccount).collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(accounts)) {
      return Maps.newHashMap();
    }
    List<DirectAccountVo> accountList = this.directAccountVoService.findByAccountList(Lists.newArrayList(accounts));
    if (CollectionUtils.isEmpty(accountList)) {
      return Maps.newHashMap();
    }
    return accountList.stream().collect(Collectors.toMap(DirectAccountVo::getAccount, t -> t, (a, b) -> a));
  }

  /**
   * 构建系统映射(key:系统编码,value:系统)
   *
   * @param ruleEntities 抓单规则列表
   * @return 营体系映射
   */
  private Map<String, DirectVo> buildDirectMap(List<GrabRuleEntity> ruleEntities) {
    Set<String> directCodes = ruleEntities.stream().map(GrabRuleEntity::getDirectCode).collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(directCodes)) {
      return Maps.newHashMap();
    }
    List<DirectVo> directList = this.directVoService.findByDirectCodes(Lists.newArrayList(directCodes));
    if (CollectionUtils.isEmpty(directList)) {
      return Maps.newHashMap();
    }
    return directList.stream().collect(Collectors.toMap(DirectVo::getDirectCode, t -> t, (a, b) -> a));
  }

  /**
   * 构建请求方式映射(key:商超编码+订单类型,value:请求方式)
   *
   * @param directMap 系统映射
   * @return 请求方式映射
   */
  private Map<String, String> buildMethodMap(Map<String, DirectVo> directMap) {
    if (CollectionUtils.isEmpty(directMap.values())) {
      return Maps.newHashMap();
    }
    List<SupermarketParameterGrabModeVo> grabModeList = this.supermarketParameterGrabModeVoService
        .findBySupermarketCodes(Lists.newArrayList(directMap.values().stream().map(DirectVo::getSupermarketCode).collect(Collectors.toList())),null);
    if (CollectionUtils.isEmpty(grabModeList)) {
      return Maps.newHashMap();
    }
    return grabModeList.stream().collect(Collectors.toMap(o -> String.format("%s%s", o.getSupermarketCode(), o.getOrderType())
        , SupermarketParameterGrabModeVo::getDownloadType, (a, b) -> a));
  }
}
