package com.biz.crm.kms.business.grab.rule.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.utils.DateTimeValidateUtil;
import com.biz.crm.kms.business.grab.rule.local.entity.GrabRuleEntity;
import com.biz.crm.kms.business.grab.rule.local.model.GrabRuleConditionModel;
import com.biz.crm.kms.business.grab.rule.local.repository.GrabRuleRepository;
import com.biz.crm.kms.business.grab.rule.local.service.GrabRequestService;
import com.biz.crm.kms.business.grab.rule.local.service.GrabRuleService;
import com.biz.crm.kms.business.grab.rule.local.service.GrabRuleStoreService;
import com.biz.crm.kms.business.grab.rule.sdk.constant.GrabRuleConstant;
import com.biz.crm.kms.business.grab.rule.sdk.dto.GrabRuleDto;
import com.biz.crm.kms.business.grab.rule.sdk.enums.GrabType;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;

/**
 * 抓单规则表服务实现类
 *
 * @author ning.zhang
 * @date 2022-09-26 16:38:28
 */
@Slf4j
@Service("grabRuleService")
public class GrabRuleServiceImpl implements GrabRuleService {

  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private GrabRuleRepository grabRuleRepository;
  @Autowired
  private GrabRuleStoreService grabRuleStoreService;
  @Autowired
  private GrabRequestService grabRequestService;

  @Override
  @Transactional
  public GrabRuleEntity create(GrabRuleDto dto) {
    this.createValidation(dto);
    GrabRuleEntity entity = this.nebulaToolkitService.copyObjectByBlankList(dto, GrabRuleEntity.class, HashSet.class
        , ArrayList.class, "ruleParam");
    entity.setRuleParam(JSON.toJSONString(dto.getRuleParam()));
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    entity.setTaskTime(new Date());
    entity.setTaskTimestamp(entity.getTaskTime().getTime());
    this.grabRuleRepository.save(entity);
    //绑定关联数据
    this.grabRuleStoreService.update(entity);
    return entity;
  }

  @Override
  @Transactional
  public GrabRuleEntity update(GrabRuleDto dto) {
    this.updateValidation(dto);
    GrabRuleConditionModel model = new GrabRuleConditionModel();
    model.setIds(Sets.newHashSet(dto.getId()));
    model.setTenantCode(dto.getTenantCode());
    List<GrabRuleEntity> entities = this.grabRuleRepository.findByGrabRuleConditionModel(model);
    Validate.isTrue(!CollectionUtils.isEmpty(entities), "抓单规则信息不存在");
    GrabRuleEntity updateEntity = this.nebulaToolkitService.copyObjectByBlankList(dto, GrabRuleEntity.class, HashSet.class
        , ArrayList.class, "ruleParam");
    updateEntity.setRuleParam(JSON.toJSONString(dto.getRuleParam()));
    updateEntity.setTaskTime(new Date());
    updateEntity.setTaskTimestamp(updateEntity.getTaskTime().getTime());
    this.grabRuleRepository.updateById(updateEntity);
    //绑定关联数据
    this.grabRuleStoreService.update(updateEntity);
    return updateEntity;
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    List<GrabRuleEntity> entities = this.grabRuleRepository.listByIds(ids);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据删除个数不匹配");
    this.grabRuleRepository.updateDelFlagByIds(ids);
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    GrabRuleConditionModel model = new GrabRuleConditionModel();
    model.setIds(Sets.newHashSet(ids));
    model.setTenantCode(TenantUtils.getTenantCode());
    List<GrabRuleEntity> entities = this.grabRuleRepository.findByGrabRuleConditionModel(model);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据启用个数不匹配");
    this.grabRuleRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    GrabRuleConditionModel model = new GrabRuleConditionModel();
    model.setIds(Sets.newHashSet(ids));
    model.setTenantCode(TenantUtils.getTenantCode());
    List<GrabRuleEntity> entities = this.grabRuleRepository.findByGrabRuleConditionModel(model);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据禁用个数不匹配");
    this.grabRuleRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
  }

  @Override
  @DynamicTaskService(cornExpression = "0 0/10 * * * ?", taskDesc = "自动抓单定时任务")
  @Transactional
  public void executeTask() {
    log.info("---------------------------------进入自动抓单定时任务---------------------------------");
    GrabRuleConditionModel model = new GrabRuleConditionModel();
    model.setTenantCode(TenantUtils.getTenantCode());
    model.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    model.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    model.setAutoGrabFlag(BooleanEnum.TRUE.getCapital());
    model.setNowDate(new Date());
    model.setNowTimestamp(model.getNowDate().getTime());
    model.setNowTime(model.getNowDate().toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime().format(GrabRuleConstant.HH_MM));
    List<GrabRuleEntity> entities = this.grabRuleRepository.findByGrabRuleConditionModel(model);
    if (CollectionUtils.isEmpty(entities)) {
      return;
    }
    entities.forEach(ruleEntity -> ruleEntity.setGrabType(GrabType.AUTO.getDictCode()));
    this.grabRequestService.createBatch(entities);
    this.grabRuleRepository.updateBatchById(entities);
  }

  @Override
  @Transactional
  public void manualGrab(GrabRuleDto dto) {
    this.createToManualValidation(dto);
    GrabRuleEntity entity = this.nebulaToolkitService.copyObjectByBlankList(dto, GrabRuleEntity.class, HashSet.class
        , ArrayList.class, "ruleParam");
    entity.setRuleParam(JSON.toJSONString(dto.getRuleParam()));
    entity.setGrabType(GrabType.MANUAL.getDictCode());
    this.grabRequestService.createBatch(Lists.newArrayList(entity));
  }

  /**
   * 在创建GrabRule模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(GrabRuleDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getAccount(), "缺失客户账号");
//    Validate.notBlank(dto.getAllStoreFlag(), "缺失是否是全部门店");
    Validate.notBlank(dto.getDirectCode(), "缺失系统编码");
    Validate.notBlank(dto.getEndTime(), "缺失结束时间");
    Validate.notBlank(dto.getOrderType(), "缺失单据类型");
    Validate.notBlank(dto.getRuleName(), "缺失规则名称");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getRuleParam()), "缺失规则参数");
    Validate.notBlank(dto.getStartTime(), "缺失开始时间");
    Validate.isTrue(Objects.nonNull(dto.getIntervalTime()) && dto.getIntervalTime() > 0, "时间间隔数据错误");
    Validate.isTrue(DateTimeValidateUtil.validateTime(String.format("%s:%s",dto.getEndTime(),"00")), "非法的结束时间格式,可用的格式:[HH:mm]");
    Validate.isTrue(DateTimeValidateUtil.validateTime(String.format("%s:%s",dto.getStartTime(),"00")), "非法的开始时间格式,可用的格式:[HH:mm]");
    Validate.isTrue(dto.getStartTime().compareTo(dto.getEndTime()) < 0, "开始时间需小于结束时间");
//    if (BooleanEnum.FALSE.getCapital().equals(dto.getAllStoreFlag())) {
////      Validate.isTrue(!CollectionUtils.isEmpty(dto.getStoreList()), "缺失门店信息");
//    }
  }

  /**
   * 在修改GrabRule模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(GrabRuleDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getId(), "修改信息时，id不能为空！");
    Validate.notBlank(dto.getAccount(), "缺失客户账号");
//    Validate.notBlank(dto.getAllStoreFlag(), "缺失是否是全部门店");
    Validate.notBlank(dto.getDirectCode(), "缺失系统编码");
    Validate.notBlank(dto.getEndTime(), "缺失结束时间");
    Validate.notBlank(dto.getOrderType(), "缺失单据类型");
    Validate.notBlank(dto.getRuleName(), "缺失规则名称");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getRuleParam()), "缺失规则参数");
    Validate.notBlank(dto.getStartTime(), "缺失开始时间");
    Validate.isTrue(Objects.nonNull(dto.getIntervalTime()) && dto.getIntervalTime() > 0, "时间间隔数据错误");
    Validate.isTrue(DateTimeValidateUtil.validateTime(String.format("%s:%s",dto.getEndTime(),"00")), "非法的结束时间格式,可用的格式:[HH:mm]");
    Validate.isTrue(DateTimeValidateUtil.validateTime(String.format("%s:%s",dto.getStartTime(),"00")), "非法的开始时间格式,可用的格式:[HH:mm]");
    Validate.isTrue(dto.getStartTime().compareTo(dto.getEndTime()) < 0, "开始时间需小于结束时间");
//    if (BooleanEnum.FALSE.getCapital().equals(dto.getAllStoreFlag())) {
//      Validate.isTrue(!CollectionUtils.isEmpty(dto.getStoreList()), "缺失门店信息");
//    }
  }

  /**
   * 手动抓单创建GrabRule模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createToManualValidation(GrabRuleDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(dto.getAccount(), "缺失客户账号");
//    Validate.notBlank(dto.getAllStoreFlag(), "缺失是否是全部门店");
    Validate.notBlank(dto.getDirectCode(), "缺失系统编码");
//    Validate.notBlank(dto.getEndTime(), "缺失结束时间");
    Validate.notBlank(dto.getOrderType(), "缺失单据类型");
//    Validate.notBlank(dto.getRuleName(), "缺失规则名称");
    Validate.isTrue(!CollectionUtils.isEmpty(dto.getRuleParam()), "缺失规则参数");
//    Validate.notBlank(dto.getStartTime(), "缺失开始时间");
//    Validate.isTrue(Objects.nonNull(dto.getIntervalTime()) && dto.getIntervalTime() > 0, "时间间隔数据错误");
//    Validate.isTrue(DateTimeValidateUtil.validateTime(String.format("%s:%s",dto.getEndTime(),"00")), "非法的结束时间格式,可用的格式:[HH:mm]");
//    Validate.isTrue(DateTimeValidateUtil.validateTime(String.format("%s:%s",dto.getStartTime(),"00")), "非法的开始时间格式,可用的格式:[HH:mm]");
//    Validate.isTrue(dto.getStartTime().compareTo(dto.getEndTime()) < 0, "开始时间需小于结束时间");
//    if (BooleanEnum.FALSE.getCapital().equals(dto.getAllStoreFlag())) {
////      Validate.isTrue(!CollectionUtils.isEmpty(dto.getStoreList()), "缺失门店信息");
//    }
  }
}
