package com.biz.crm.kms.business.invoice.acceptance.local.repository;

import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.kms.business.invoice.acceptance.local.entity.InvoiceAcceptanceGoodsEntity;
import com.biz.crm.kms.business.invoice.acceptance.local.mapper.InvoiceAcceptanceGoodsMapper;
import java.util.Date;
import java.util.List;

import com.biz.crm.kms.business.invoice.acceptance.sdk.vo.InvoiceAcceptanceGoodsVo;
import com.google.common.collect.Lists;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

/**
 * 验收单商品表(repository)
 *
 * @author pengxi
 * @date 2022/10/10
 */
@Component
public class InvoiceAcceptanceGoodsRepository extends ServiceImpl<InvoiceAcceptanceGoodsMapper, InvoiceAcceptanceGoodsEntity> {

  /**
   * 根据验收单号获取验收单集合
   *
   * @param orderNumber 验收单号
   * @param tenantCode  租户编码
   * @return
   */
  public List<InvoiceAcceptanceGoodsEntity> findByOrderNumber(String tenantCode, String orderNumber) {
    return this.lambdaQuery()
        .eq(InvoiceAcceptanceGoodsEntity::getTenantCode, tenantCode)
        .eq(InvoiceAcceptanceGoodsEntity::getOrderNumber, orderNumber)
        .list();
  }

  /**
   * 根据商超验收单号删除
   *
   * @param tenantCode    租户编号
   * @param kaOrderNumber 商超验收单号
   */
  public void deleteByKaOrderNumber(String tenantCode, String kaOrderNumber) {
    this.lambdaUpdate()
        .eq(InvoiceAcceptanceGoodsEntity::getTenantCode, tenantCode)
        .eq(InvoiceAcceptanceGoodsEntity::getKaOrderNumber, kaOrderNumber)
        .remove();
  }

  /**
   * 根据商超验收单号批量删除
   *
   * @param tenantCode    租户编号
   * @param kaOrderNumbers 商超验收单号
   */
  public void deleteByKaOrderNumbers(String tenantCode, List<String> kaOrderNumbers) {
    this.lambdaUpdate()
            .eq(InvoiceAcceptanceGoodsEntity::getTenantCode, tenantCode)
            .in(InvoiceAcceptanceGoodsEntity::getKaOrderNumber, kaOrderNumbers)
            .remove();
  }

  /**
   * 根据多条件查询数据
   *
   * @param invoiceDates   单据日期
   * @param goodsCodes     商品编码
   * @param storeCodes     门店编码
   * @param kaOrderNumbers 商超订单编码
   * @return 数据集合
   */
  public List<InvoiceAcceptanceGoodsEntity> findInvoiceStatistics(List<Date> invoiceDates, List<String> goodsCodes, List<String> storeCodes, List<String> kaOrderNumbers) {
    return this.baseMapper.findInvoiceStatistics(invoiceDates, goodsCodes, storeCodes, kaOrderNumbers);
  }

  /**
   * 多条件查询验收单数据
   *
   * @param goodsCodes
   * @param storeCodes
   * @param orderCodes
   * @param beginDate
   * @param endDate
   * @param soldToPartyCode
   * @param directCodes
   * @return
   */
  public List<InvoiceAcceptanceGoodsEntity> findAcceptanceByConditions(List<String> goodsCodes, List<String> storeCodes, List<String> orderCodes,String beginDate,String endDate, List<String> soldToPartyCode, List<String> directCodes){
    return this.baseMapper.findAcceptanceByConditions(goodsCodes,storeCodes,orderCodes,beginDate,endDate,soldToPartyCode,directCodes);
  }

  /**
   * 根据条件查询单据包含转换失败
   *
   * @param goodsCodes 产品编码
   * @param storeCodes 门店编码
   * @param orderCodes 采购单号
   * @param soldToPartyCodes 售达方
   * @param directCodes 系统编码
   * @return
   */
  public List<InvoiceAcceptanceGoodsVo> findAllByConditions(List<String> goodsCodes, List<String> storeCodes, List<String> orderCodes, String beginDate, String endDate, List<String> soldToPartyCodes, List<String> directCodes){
    if (CollectionUtils.isEmpty(goodsCodes)
            && CollectionUtils.isEmpty(storeCodes)
            && CollectionUtils.isEmpty(orderCodes)
            && StringUtils.isEmpty(beginDate)
            && StringUtils.isEmpty(endDate)
            && CollectionUtils.isEmpty(soldToPartyCodes)
            && CollectionUtils.isEmpty(directCodes)){
      return Lists.newArrayList();
    }else {
      return this.baseMapper.findAllByConditions(goodsCodes,storeCodes,orderCodes,beginDate,endDate,soldToPartyCodes,directCodes);
    }
  }

  /**
   * 根据id集合查询数据
   *
   * @param ids
   * @return
   */
  public List<InvoiceAcceptanceGoodsEntity> findByIds(List<String> ids) {
    return this.lambdaQuery().in(InvoiceAcceptanceGoodsEntity::getId, ids).list();
  }


  /**
   * 根据id集合查询数据(稽核用)
   * @param relationIds
   * @return
   */
  public List<InvoiceAcceptanceGoodsEntity> findByRelationId(List<String> relationIds) {
    return this.baseMapper.findByRelationId(relationIds);
  }


  /**
   * 窜单匹配查验收单
   * @param relationKaOrderCodes
   * @param acceptanceOrderNumber
   * @return
   */
  public List<InvoiceAcceptanceGoodsEntity> findCrossByConditions(List<String> relationKaOrderCodes, List<String> acceptanceOrderNumber) {
    return this.baseMapper.findCrossByConditions(relationKaOrderCodes,acceptanceOrderNumber);
  }


  /**
   * 根据验收单号批量删除
   *
   * @param tenantCode    租户编号
   * @param orderNumbers 商超验收单号
   */
  public void deleteByOrderNumbers(String tenantCode, List<String> orderNumbers) {
    if (CollectionUtils.isEmpty(orderNumbers)){
      return;
    }
    this.lambdaUpdate()
            .eq(InvoiceAcceptanceGoodsEntity::getTenantCode, tenantCode)
            .in(!CollectionUtils.isEmpty(orderNumbers),InvoiceAcceptanceGoodsEntity::getOrderNumber, orderNumbers)
            .remove();
  }

  /**
   * 根据验收单号或采购单号查询数据
   * @param orderNumbers
   * @param relationKaOrderNumber
   * @param tenantCode
   * @return
   */
  public List<InvoiceAcceptanceGoodsVo> findByOrderNumberOrRelationKaOrderNumber(List<String> orderNumbers, List<String> relationKaOrderNumber,String tenantCode){
    if (CollectionUtils.isEmpty(orderNumbers) && CollectionUtils.isEmpty(relationKaOrderNumber)){
      return Lists.newArrayList();
    }else {
      return this.baseMapper.findByOrderNumberOrRelationKaOrderNumber(orderNumbers,relationKaOrderNumber,tenantCode);
    }
  }
}
