package com.biz.crm.kms.business.invoice.acceptance.local.repository;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.kms.business.invoice.acceptance.local.entity.InvoiceAcceptanceEntity;
import com.biz.crm.kms.business.invoice.acceptance.local.mapper.InvoiceAcceptanceMapper;
import java.util.List;
import java.util.Optional;

import com.biz.crm.kms.business.invoice.acceptance.sdk.vo.InvoiceAcceptanceVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

/**
 * 验收单表(repository)
 *
 * @author pengxi
 * @date 2022/10/10
 */
@Component
public class InvoiceAcceptanceRepository extends ServiceImpl<InvoiceAcceptanceMapper, InvoiceAcceptanceEntity> {

  /**
   * 根据ID获取详情
   *
   * @param id
   * @return
   */
  public InvoiceAcceptanceEntity findById(String id) {
    return this.baseMapper.selectById(id);
  }

  /**
   * 结算单查验收单详情
   * @param code
   * @return
   */
  public InvoiceAcceptanceEntity findByDecomentCode(String code,String deliveryPartyCode){
    if (StringUtils.isBlank(code) && StringUtils.isBlank(deliveryPartyCode)){
      return null;
    }
    return this.lambdaQuery()
            .like(StringUtils.isNotBlank(code), InvoiceAcceptanceEntity::getKaOrderNumber,code)
            .eq(StringUtils.isNotBlank(deliveryPartyCode), InvoiceAcceptanceEntity::getDeliveryPartyCode,deliveryPartyCode)
            .one();
  }

  /**
   * 根据id集合获取详情集合
   *
   * @param ids
   * @return
   */
  public List<InvoiceAcceptanceEntity> findByIds(List<String> ids) {
    if (CollectionUtil.isEmpty(ids)){
      return null;
    }
    return this.lambdaQuery()
        .in(CollectionUtil.isNotEmpty(ids),InvoiceAcceptanceEntity::getId, ids)
        .list();
  }

  /**
   * 根据验收单编码查询验收单
   * @param tenantCode
   * @param orderNumber
   * @return
   */
  public InvoiceAcceptanceEntity findByOrderNumber(String tenantCode, String orderNumber){
    return this.lambdaQuery()
            .eq(InvoiceAcceptanceEntity::getTenantCode, tenantCode)
            .eq(InvoiceAcceptanceEntity::getOrderNumber, orderNumber)
            .one();
  }

    /**
     * 根据验收单编码查询验收单
     *
     * @param tenantCode
     * @param orderNumberList
     * @return
     */
    public List<InvoiceAcceptanceEntity> findByOrderNumberList(String tenantCode, List<String> orderNumberList) {
        return this.lambdaQuery()
                .eq(InvoiceAcceptanceEntity::getTenantCode, tenantCode)
                .in(InvoiceAcceptanceEntity::getOrderNumber, orderNumberList)
                .list();
    }

  /**
   * 根据商超验收单编号获取验收单集合
   *
   * @param tenantCode    租户编码
   * @param kaOrderNumber 商超验收单编号
   * @return
   */
  public InvoiceAcceptanceEntity findByKaOrderNumber(String tenantCode, String kaOrderNumber) {
    return this.lambdaQuery()
        .eq(InvoiceAcceptanceEntity::getTenantCode, tenantCode)
        .eq(InvoiceAcceptanceEntity::getKaOrderNumber, kaOrderNumber)
        .one();
  }

  /**
   * 根据商超验收单编码删除数据
   *
   * @param tenantCode    租户编码
   * @param kaOrderNumber 商超验收单编号
   */
  public void deleteByKaOrderNumber(String tenantCode, String kaOrderNumber) {
    this.lambdaUpdate()
        .eq(InvoiceAcceptanceEntity::getTenantCode, tenantCode)
        .eq(InvoiceAcceptanceEntity::getKaOrderNumber, kaOrderNumber)
        .remove();
  }

  /**
   * 根据商超验收单编码批量删除数据
   *
   * @param tenantCode    租户编码
   * @param kaOrderNumbers 商超验收单编号
   */
  public void deleteByKaOrderNumbers(String tenantCode, List<String> kaOrderNumbers) {
    this.lambdaUpdate()
            .eq(InvoiceAcceptanceEntity::getTenantCode, tenantCode)
            .in(InvoiceAcceptanceEntity::getKaOrderNumber, kaOrderNumbers)
            .remove();
  }

  /**
   * 取消捆绑结算单
   * @param statementCode
   */
  public void cancel(String statementCode){
    this.lambdaUpdate()
            .eq(InvoiceAcceptanceEntity::getStatementCode,statementCode)
            .set(InvoiceAcceptanceEntity::getStatementCode,null)
            .update();
  }

  /**
   * 捆绑结算单号
   * @param orderNumbers
   * @param statementCode
   */
  public void match(List<String> orderNumbers,String statementCode){
    this.lambdaUpdate()
            .in(InvoiceAcceptanceEntity::getOrderNumber,orderNumbers)
            .set(InvoiceAcceptanceEntity::getStatementCode,statementCode)
            .update();
  }

  /**
   * 根据系统分页查询结算单号
   * @param pageable
   * @param tenantCode
   * @param directCodes
   * @return
   */
  public Page<InvoiceAcceptanceVo> findAcceptanceByDirectCodes(Pageable pageable, String tenantCode, List<String> directCodes){
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(1, 50));
    Page<InvoiceAcceptanceVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.baseMapper.findAcceptanceByDirectCodes(page, tenantCode, directCodes);
  }

  /**
   * 条件查询验收单
   * @param beginDate
   * @param endDate
   * @param directCodes
   * @return
   */
  public List<InvoiceAcceptanceVo> findAcceptanceOrderVo(String beginDate, String endDate, List<String> directCodes){
    if (CollectionUtil.isNotEmpty(directCodes) && StringUtils.isNotEmpty(beginDate) && StringUtils.isNotEmpty(endDate)){
      return this.baseMapper.findAcceptanceOrderVo(beginDate, endDate, directCodes,TenantUtils.getTenantCode());
    }else {
      return Lists.newArrayList();
    }
  }


  /**
   * 根据验收单号批量删除
   * @param tenantCode
   * @param orderNumbers
   */
  public void deleteByOrderNumbers(String tenantCode,List<String> orderNumbers){
    if (CollectionUtils.isEmpty(orderNumbers)){
      return;
    }
    this.lambdaUpdate()
            .eq(StringUtils.isNotBlank(tenantCode),InvoiceAcceptanceEntity::getTenantCode,tenantCode)
            .in(!CollectionUtils.isEmpty(orderNumbers),InvoiceAcceptanceEntity::getOrderNumber,orderNumbers)
            .remove();
  }
}
