package com.biz.crm.kms.business.invoice.acceptance.local.repository;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.kms.business.invoice.acceptance.local.entity.KmsInvoiceAcceptanceGrabEntity;
import com.biz.crm.kms.business.invoice.acceptance.local.mapper.KmsInvoiceAcceptanceGrabMapper;
import com.biz.crm.kms.business.invoice.sdk.enums.GrabTransStatus;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * 验收单原始数据表(repository)
 *
 * @author xi.peng
 * @since 2022-10-13 17:48:57
 */
@Component
public class KmsInvoiceAcceptanceGrabRepository extends ServiceImpl<KmsInvoiceAcceptanceGrabMapper, KmsInvoiceAcceptanceGrabEntity> {

    /**
     * 根据企业验收单编号集合获取待转换记录
     *
     * @return
     */
    public List<KmsInvoiceAcceptanceGrabEntity> findByOrderNumbers(String tenantCode, List<String> orderNumbers) {
        return this.lambdaQuery()
                .eq(KmsInvoiceAcceptanceGrabEntity::getTenantCode, tenantCode)
                .in(KmsInvoiceAcceptanceGrabEntity::getOrderNumber, orderNumbers)
                .list();
    }

    /**
     * 根据验收单号更新转换状态
     *
     * @param transStatus 转换状态
     */
    public void updateTransStatusById(String transStatus, String id, String transMsg) {
        this.lambdaUpdate()
                .eq(KmsInvoiceAcceptanceGrabEntity::getId, id)
                .set(KmsInvoiceAcceptanceGrabEntity::getTransStatus, transStatus)
                .set(StringUtils.isNotBlank(transMsg), KmsInvoiceAcceptanceGrabEntity::getTransMsg, transMsg)
                .update();
    }


    /**
     * 验收单根据租户编码和转换状态查询验收单带商品
     *
     * @param pageable
     * @param tenantCode
     * @param transStatusList
     * @return
     */
    public Page<String> findAcceptanceOrderGrabPage(Pageable pageable, String tenantCode, List<String> transStatusList,String day) {
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(1, 50));
        Page<String> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.baseMapper.findAcceptanceOrderGrabPage(page, tenantCode, transStatusList,day);
    }

    /**
     * 根据验收单号更新转换状态
     *
     * @param tenantCode  租户编码
     * @param transStatus 转换状态
     * @param orderNumber 验收单号
     */
    public void updateTransStatusByOrderNumber(String tenantCode, GrabTransStatus transStatus, String orderNumber) {
        if (StringUtils.isBlank(tenantCode)
                || Objects.isNull(transStatus)
                || StringUtils.isBlank(orderNumber)) {
            return;
        }
        this.lambdaUpdate()
                .eq(KmsInvoiceAcceptanceGrabEntity::getTenantCode, tenantCode)
                .eq(KmsInvoiceAcceptanceGrabEntity::getOrderNumber, orderNumber)
                .set(KmsInvoiceAcceptanceGrabEntity::getTransStatus, transStatus.getDictCode())
                .update();
    }

    /**
     * 根据验收单号更新转换状态
     *
     * @param tenantCode      租户编码
     * @param transStatus     转换状态
     * @param orderNumberList 验收单号
     */
    public void updateTransStatusByOrderNumberList(String tenantCode, GrabTransStatus transStatus, List<String> orderNumberList) {
        if (StringUtils.isBlank(tenantCode)
                || Objects.isNull(transStatus)
                || CollectionUtil.isEmpty(orderNumberList)) {
            return;
        }
        this.lambdaUpdate()
                .eq(KmsInvoiceAcceptanceGrabEntity::getTenantCode, tenantCode)
                .in(KmsInvoiceAcceptanceGrabEntity::getOrderNumber, orderNumberList)
                .set(KmsInvoiceAcceptanceGrabEntity::getTransStatus, transStatus.getDictCode())
                .update();
    }
}
