package com.biz.crm.kms.business.invoice.acceptance.local.service.internal;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.kms.business.direct.product.sdk.dto.DirectProductDto;
import com.biz.crm.kms.business.direct.product.sdk.service.DirectProductVoService;
import com.biz.crm.kms.business.direct.product.sdk.vo.DirectProductVo;
import com.biz.crm.kms.business.direct.sdk.service.DirectVoService;
import com.biz.crm.kms.business.direct.sdk.vo.DirectVo;
import com.biz.crm.kms.business.direct.store.sdk.dto.DirectStoreConditionDto;
import com.biz.crm.kms.business.direct.store.sdk.service.DirectStoreVoService;
import com.biz.crm.kms.business.direct.store.sdk.vo.DirectStoreVo;
import com.biz.crm.kms.business.invoice.acceptance.local.entity.InvoiceAcceptanceEntity;
import com.biz.crm.kms.business.invoice.acceptance.local.entity.InvoiceAcceptanceGoodsEntity;
import com.biz.crm.kms.business.invoice.acceptance.local.entity.KmsInvoiceAcceptanceGoodsGrabEntity;
import com.biz.crm.kms.business.invoice.acceptance.local.entity.KmsInvoiceAcceptanceGrabEntity;
import com.biz.crm.kms.business.invoice.acceptance.local.register.AcceptanceInvoiceTypeRegister;
import com.biz.crm.kms.business.invoice.acceptance.local.repository.InvoiceAcceptanceRepository;
import com.biz.crm.kms.business.invoice.acceptance.local.repository.KmsInvoiceAcceptanceGoodsGrabRepository;
import com.biz.crm.kms.business.invoice.acceptance.local.repository.KmsInvoiceAcceptanceGrabRepository;
import com.biz.crm.kms.business.invoice.acceptance.local.service.InvoiceAcceptanceGrabService;
import com.biz.crm.kms.business.invoice.acceptance.local.service.InvoiceAcceptanceOrderGrabsThreadService;
import com.biz.crm.kms.business.invoice.acceptance.sdk.constant.AcceptanceOrderConstant;
import com.biz.crm.kms.business.invoice.acceptance.sdk.enums.AcceptanceStatus;
import com.biz.crm.kms.business.invoice.sdk.constant.DirectConstant;
import com.biz.crm.kms.business.invoice.sdk.enums.AutoConvertsEnum;
import com.biz.crm.kms.business.invoice.sdk.enums.ConstantEnums;
import com.biz.crm.kms.business.invoice.sdk.enums.DirectEnum;
import com.biz.crm.kms.business.invoice.sdk.enums.GrabTransStatus;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mn.common.base.service.RedisCrmService;
import com.biz.crm.mn.common.rocketmq.service.RocketMqProducer;
import com.biz.crm.mn.common.rocketmq.util.RocketMqUtil;
import com.biz.crm.mn.common.rocketmq.vo.MqMessageVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;


/**
 * 验收单抓取数据(python)表服务实现类
 */
@Slf4j
@Service
public class InvoiceAcceptanceGrabServiceImpl implements InvoiceAcceptanceGrabService {

    @Autowired(required = false)
    private KmsInvoiceAcceptanceGrabRepository kmsInvoiceAcceptanceGrabRepository;

    @Autowired(required = false)
    private RocketMqProducer rocketMqProducer;

    @Autowired(required = false)
    private DirectStoreVoService directStoreVoService;

    @Autowired(required = false)
    private DirectProductVoService directProductVoService;

    @Autowired(required = false)
    private RedisCrmService redisCrmService;

    @Autowired(required = false)
    private RedisMutexService redisMutexService;

    @Autowired(required = false)
    private InvoiceAcceptanceOrderGrabsThreadService invoiceAcceptanceOrderGrabsThreadService;

    @Autowired(required = false)
    private InvoiceAcceptanceRepository invoiceAcceptanceRepository;

    @Autowired(required = false)
    private KmsInvoiceAcceptanceGoodsGrabRepository kmsInvoiceAcceptanceGoodsGrabRepository;

    @Autowired(required = false)
    private AcceptanceInvoiceTypeRegister acceptanceInvoiceTypeRegister;

    @Autowired(required = false)
    @Qualifier("nebulaToolkitService")
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private DirectVoService directVoService;

    @Autowired(required = false)
    private DictDataVoService dictDataVoService;


    /**
     * 验收单未转换单据自动转换
     */
    @Override
    public void autoNotConvertAcceptanceOrder() {
        //验收单自动转换锁定时间
        log.info("=====>    验收单未转换单据自动转换 start   <=====");
        boolean lock = redisMutexService.tryLock(AcceptanceOrderConstant.ACCEPTANCE_ORDER_AUTO_NOT_CONVERT_LOCK, TimeUnit.SECONDS, AcceptanceOrderConstant.ACCEPTANCE_ORDER_REDIS_TIME);
        if (!lock) {
            log.warn("验收单正在自动转换中,本次自动转换被忽略!");
            log.info("=====>    验收单未转换单据自动转换 end   <=====");
            return;
        }
        Pageable pageable = PageRequest.of(1,AcceptanceOrderConstant.ACCEPTANCE_ORDER_PAGE_SIZE);
        List<String> transStatusList = new ArrayList<>();
        transStatusList.add(GrabTransStatus.NOT_TRANS.getDictCode());
        String day = this.getDays(AutoConvertsEnum.NOT_CONVERTS.getDictCode());
        Set<String> currentOrderNumberSet = new HashSet<>(AcceptanceOrderConstant.ACCEPTANCE_ORDER_PAGE_SIZE);
        try {
            //查询待转换验收单信息
            Page<String> orderNumberPage = this.kmsInvoiceAcceptanceGrabRepository.findAcceptanceOrderGrabPage(pageable, TenantUtils.getTenantCode(), transStatusList,day);
            currentOrderNumberSet.addAll(orderNumberPage.getRecords());
            this.sendMqMessage(orderNumberPage.getRecords());
            while (orderNumberPage.hasNext()
                    && AcceptanceOrderConstant.ACCEPTANCE_ORDER_LOOP_MAX >= pageable.getPageNumber()) {
                pageable = pageable.next();
                orderNumberPage = this.kmsInvoiceAcceptanceGrabRepository.findAcceptanceOrderGrabPage(pageable, TenantUtils.getTenantCode(), transStatusList,day);
                currentOrderNumberSet.clear();
                currentOrderNumberSet.addAll(orderNumberPage.getRecords());
                this.sendMqMessage(orderNumberPage.getRecords());
            }
        } catch (Exception e) {
            log.error("", e);
            if (CollectionUtil.isNotEmpty(currentOrderNumberSet)) {
                //异常时,删除当前加锁redis
                redisCrmService.hdel(AcceptanceOrderConstant.ACCEPTANCE_ORDER_CONVERT_LOCK, currentOrderNumberSet.toArray());
            }
        } finally {
            redisMutexService.unlock(AcceptanceOrderConstant.ACCEPTANCE_ORDER_AUTO_NOT_CONVERT_LOCK);
        }
        log.info("=====>    验收单未转换单据自动转换 end   <=====");
    }

    /**
     * 验收单转换失败单据自动转换
     */
    @Override
    public void autoFailConvertAcceptanceOrder() {
        //验收单转换失败单据自动转换锁定时间
        log.info("=====>    验收单转换失败单据自动转换 start   <=====");
        boolean lock = redisMutexService.tryLock(AcceptanceOrderConstant.ACCEPTANCE_ORDER_AUTO_FAIL_CONVERT_LOCK, TimeUnit.SECONDS, AcceptanceOrderConstant.ACCEPTANCE_ORDER_REDIS_TIME);
        if (!lock) {
            log.warn("验收单正在自动转换中,本次自动转换被忽略!");
            log.info("=====>    验收单转换失败单据自动转换 end   <=====");
            return;
        }
        Pageable pageable = Pageable.ofSize(AcceptanceOrderConstant.ACCEPTANCE_ORDER_PAGE_SIZE);
        List<String> transStatusList = new ArrayList<>();
        transStatusList.add(GrabTransStatus.TRANS_FAIL.getDictCode());
        String day = this.getDays(AutoConvertsEnum.FAIL_CONVERTS.getDictCode());
        Set<String> currentOrderNumberSet = new HashSet<>(AcceptanceOrderConstant.ACCEPTANCE_ORDER_PAGE_SIZE);
        try {
            //查询待转换验收单信息
            Page<String> orderNumberPage = this.kmsInvoiceAcceptanceGrabRepository.findAcceptanceOrderGrabPage(pageable, TenantUtils.getTenantCode(), transStatusList,day);
            currentOrderNumberSet.addAll(orderNumberPage.getRecords());
            this.sendMqMessage(orderNumberPage.getRecords());
            while (orderNumberPage.hasNext()
                    && AcceptanceOrderConstant.ACCEPTANCE_ORDER_LOOP_MAX >= pageable.getPageNumber()) {
                pageable = pageable.next();
                orderNumberPage = this.kmsInvoiceAcceptanceGrabRepository.findAcceptanceOrderGrabPage(pageable, TenantUtils.getTenantCode(), transStatusList,day);
                currentOrderNumberSet.clear();
                currentOrderNumberSet.addAll(orderNumberPage.getRecords());
                this.sendMqMessage(orderNumberPage.getRecords());
            }
        } catch (Exception e) {
            log.error("", e);
            if (CollectionUtil.isNotEmpty(currentOrderNumberSet)) {
                //异常时,删除当前加锁redis
                redisCrmService.hdel(AcceptanceOrderConstant.ACCEPTANCE_ORDER_CONVERT_LOCK, currentOrderNumberSet.toArray());
            }
        } finally {
            redisMutexService.unlock(AcceptanceOrderConstant.ACCEPTANCE_ORDER_AUTO_FAIL_CONVERT_LOCK);
        }
        log.info("=====>    验收单转换失败单据自动转换 end   <=====");
    }

    /**
     * 自动转单验收单发送 MQ消息
     *
     * @param orderNumberList 验收单集合
     */
    private void sendMqMessage(List<String> orderNumberList) {
        if (CollectionUtils.isEmpty(orderNumberList)) {
            return;
        }
        //过滤单号为空的数据
        orderNumberList = orderNumberList.stream()
                .filter(StringUtils::isNotBlank)
                .distinct()
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(orderNumberList)) {
            log.error("验收单原始单据有空单号[order_number]存在!");
            return;
        }
        //过滤正在转换的验收单号
        this.filterSwitchIngOrderNumberList(orderNumberList);
        if (CollectionUtil.isEmpty(orderNumberList)) {
            return;
        }

        //发送MQ消息开始转换单据
        MqMessageVo message = new MqMessageVo();
        message.setMsgBody(JSON.toJSONString(orderNumberList));
        message.setTopic(AcceptanceOrderConstant.KMS_ACCEPTANCE_ORDER_TOPIC + RocketMqUtil.mqEnvironment());
        message.setTag(AcceptanceOrderConstant.ACCEPTANCE_ORDER_MESSAGE_TAG);
        rocketMqProducer.sendMqMsg(message);
        try {
            //单位：毫秒 防止MQ消息发送过于频繁
            Thread.sleep(200);
        } catch (Exception e) {
            log.error("", e);
        }
    }


    /**
     * 根据单据号 转换单据信息
     *
     * @param orderNumberList 单据号
     * @return void
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.15 13:42
     */
    @Override
    public void manualSwitch(List<String> orderNumberList) {
        if (CollectionUtil.isEmpty(orderNumberList)) {
            log.warn("未接收到需转换验收单号！");
            return;
        }
        //组装单据 根据单据号  组装单据头和单据明细
        List<InvoiceAcceptanceEntity> orderList = this.buildOrder(orderNumberList);
        //转换单据新
        this.convertOrder(orderList);
    }

    /**
     * 组装单据
     *
     * @param orderNumberList
     * @return
     */
    private List<InvoiceAcceptanceEntity> buildOrder(List<String> orderNumberList) {
        log.info("--  组装单据步骤1  --");
        if (CollectionUtil.isEmpty(orderNumberList)) {
            return Collections.emptyList();
        }
        List<KmsInvoiceAcceptanceGrabEntity> entityGrabList = this.kmsInvoiceAcceptanceGrabRepository.findByOrderNumbers(TenantUtils.getTenantCode(), orderNumberList);
        if (CollectionUtil.isEmpty(entityGrabList)) {
            log.warn("不存在或已删除！");
            return Collections.emptyList();
        }
        //默认版本号  很重要  很重要  很重要
        int defaultVersionNumber = 1;

        // 设置基础字段和版本号
        entityGrabList.forEach(entity -> {
            entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            if (Objects.isNull(entity.getVersionNumber())) {
                entity.setVersionNumber(defaultVersionNumber);
            }
        });
        //单据转换
        List<InvoiceAcceptanceEntity> entityList = (List<InvoiceAcceptanceEntity>) this.nebulaToolkitService.copyCollectionByBlankList(entityGrabList, KmsInvoiceAcceptanceGrabEntity.class,
                InvoiceAcceptanceEntity.class, HashSet.class, ArrayList.class);

        //获取最新大版本号数据
        Map<String, InvoiceAcceptanceEntity> entityMap = entityList.stream().collect(
                Collectors.toMap(InvoiceAcceptanceEntity::getOrderNumber,
                        Function.identity(),
                        (a, b) -> a.getVersionNumber() > b.getVersionNumber() ? a : b));
        //拼接商品数据
        List<InvoiceAcceptanceEntity> convertData = new ArrayList<>(entityMap.values());
        List<KmsInvoiceAcceptanceGoodsGrabEntity> goodsGrabEntities = this.kmsInvoiceAcceptanceGoodsGrabRepository.findByOrderNumbers(orderNumberList, TenantUtils.getTenantCode());
        if (CollectionUtil.isNotEmpty(goodsGrabEntities)) {
            List<InvoiceAcceptanceGoodsEntity> goodsEntities = (List<InvoiceAcceptanceGoodsEntity>) this.nebulaToolkitService.copyCollectionByWhiteList(goodsGrabEntities, KmsInvoiceAcceptanceGoodsGrabEntity.class, InvoiceAcceptanceGoodsEntity.class, HashSet.class, ArrayList.class);
            //设置基础字段和版本号
            goodsEntities.forEach(entity -> {
                entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                if (Objects.isNull(entity.getVersionNumber())) {
                    entity.setVersionNumber(defaultVersionNumber);
                }
            });
            Map<String, List<InvoiceAcceptanceGoodsEntity>> goodsMap = goodsEntities.stream()
                    .collect(Collectors.groupingBy(InvoiceAcceptanceGoodsEntity::getOrderNumber));
            convertData.forEach(entity -> {
                List<InvoiceAcceptanceGoodsEntity> goodsEntityList = goodsMap.get(entity.getOrderNumber());
                if (CollectionUtil.isNotEmpty(goodsEntityList)) {
                    List<InvoiceAcceptanceGoodsEntity> goodsInfo = goodsEntityList
                            .stream().filter(goods -> entity.getVersionNumber()
                                    .compareTo(goods.getVersionNumber()) == 0)
                            .collect(Collectors.toList());
                    entity.setGoodsList(goodsInfo);
                }
            });
        }
        return convertData;
    }
    /**
     * 验收单商超门店和商超产品转换
     *
     * @param entityList
     * @return void
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.14 14:58
     */
    @Override
    public void convertOrder(List<InvoiceAcceptanceEntity> entityList) {
        log.info("--  单据转换所需数据查询  --");
        if (CollectionUtils.isEmpty(entityList)) {
            return;
        }
        entityList = entityList.stream()
                .filter(k -> StringUtils.isNotBlank(k.getOrderNumber())).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(entityList)) {
            return;
        }
        //系统编码集合
        Set<String> directCodeSet = Sets.newHashSet();
        //需转换的单据编码集合
        List<String> orderNumberList = Lists.newArrayList();
        //需转换的门店编码集合
        Set<String> kaStoreCodes = Sets.newHashSet();
        //需转换的商品编码集合
        Set<String> kaGoodsCodes = Sets.newHashSet();

        //需转换的终端编码集合（终端业务员）和产品集合
        entityList.forEach(entity -> {
            //初始化单据状态
            entity.setOrderStatus(AcceptanceStatus.S101.getDictCode());

            if (StringUtils.isNotEmpty(entity.getDirectCode())) {
                directCodeSet.add(entity.getDirectCode());
            }
            if (StringUtils.isNotEmpty(entity.getKaStoreCode())) {
                if(DirectEnum.GAOXIN_TJ.getDictCode().equals(entity.getDirectCode())||DirectEnum.GAOXIN_BJ.getDictCode().equals(entity.getDirectCode())) {
                    //区域编码
                    entity.setAreaCode(entity.getKaStoreCode().substring(0,1));
                    String kaStoreCode = entity.getKaStoreCode().substring(1,4);
                    Integer aa = new Integer(kaStoreCode);
                    String bb = String.valueOf(aa);
                    //门店编码分割
                    entity.setKaStoreCode(bb);
                    kaStoreCodes.add(bb);
                }else{
                    kaStoreCodes.add(entity.getKaStoreCode());
                }
            }
            orderNumberList.add(entity.getOrderNumber());
            if (CollectionUtil.isNotEmpty(entity.getGoodsList())) {
                Set<String> goodsCodeSet = entity.getGoodsList().stream()
                        .filter(k -> StringUtils.isNotBlank(k.getKaGoodsCode()))
                        .map(InvoiceAcceptanceGoodsEntity::getKaGoodsCode).collect(Collectors.toSet());
                if (CollectionUtil.isNotEmpty(goodsCodeSet)) {
                    kaGoodsCodes.addAll(goodsCodeSet);
                }
            }
        });
        //已转换过的单据信息更新到最新的单据上
        entityList = this.updateOldDataToNewData(orderNumberList, entityList);
        if (CollectionUtils.isEmpty(entityList)) {
            return;
        }
        List<DirectVo> directVos = this.directVoService.findByDirectCodes(directCodeSet.stream().collect(Collectors.toList()));
        //获取系统数据
        Map<String, List<DirectVo>> directMap = Maps.newHashMap();
        if (!CollectionUtils.isEmpty(directVos)){
            directMap.putAll(directVos.stream()
                    .filter(k ->StringUtils.isNotBlank(k.getBusinessUnitCode()) && StringUtils.isNotBlank(k.getBusinessFormatCode()))
                    .collect(Collectors.groupingBy(DirectVo::getDirectCode)));
        }
        //门店map 系统编码+商超门店编码 构成唯一
        Map<String, List<DirectStoreVo>> directStoreMap = this.buildStoreInfo(directCodeSet, kaStoreCodes);
        Set<String> deliveryPartyCodeSet = Sets.newHashSet();
        if (CollectionUtil.isNotEmpty(directStoreMap)) {
            directStoreMap.values().stream()
                    .filter(CollectionUtil::isNotEmpty)
                    .forEach(directStoreVoList -> {
                        directStoreVoList.forEach(entity -> {
                            deliveryPartyCodeSet.add(entity.getTerminalCode());
                            if (StringUtils.isNotBlank(entity.getRetailerTerminalCode())){
                                deliveryPartyCodeSet.add(entity.getRetailerTerminalCode());
                            }
                        });
                    });
        }
        //商品map 系统编码+送达方编码+商超产品编码 构成唯一
        Map<String, List<DirectProductVo>> directProductMap = this.buildProductInfo(directCodeSet, deliveryPartyCodeSet, kaGoodsCodes);

        //异步线程进行组装数据，并且保存
        //注：异步线程的方法不能和调用方在同一个类中，否则不会走线程池
        List<List<InvoiceAcceptanceEntity>> entityGroupList = Lists.partition(entityList, ConstantEnums.THREAD_SIZE.getValue());

        entityGroupList.stream()
                .filter(CollectionUtil::isNotEmpty)
                .forEach(asyncList -> {
                    invoiceAcceptanceOrderGrabsThreadService.saveAcceptanceOrderData(asyncList,directMap, directStoreMap, directProductMap);
                });

    }

    /**
     * 构建门店信息
     * <p>
     * 门店map 系统编码+商超门店编码 构成唯一
     *
     * @param directCodeSet 系统编码集合
     * @param kaStoreCodes  商超门店编码集合
     */
    private Map<String, List<DirectStoreVo>> buildStoreInfo(Set<String> directCodeSet, Set<String> kaStoreCodes) {
        Map<String, List<DirectStoreVo>> directStoreMap = Maps.newHashMap();
        if (CollectionUtil.isEmpty(kaStoreCodes)
                || CollectionUtil.isEmpty(directCodeSet)) {
            return directStoreMap;
        }
        DirectStoreConditionDto conditionDto = new DirectStoreConditionDto();
        conditionDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        conditionDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        conditionDto.setSupermarketStoreCodes(kaStoreCodes);
        conditionDto.setDirectCodes(directCodeSet);
        List<DirectStoreVo> directStoreList = this.directStoreVoService.findByDirectStoreConditionDto(conditionDto);
        if (CollectionUtil.isNotEmpty(directStoreList)) {
            directStoreMap.putAll(directStoreList.stream()
                    .filter(k -> StringUtils.isNotBlank(k.getDirectCode()))
                    .filter(k -> StringUtils.isNotBlank(k.getSupermarketStoreCode()))
                    .collect(Collectors.groupingBy(k -> k.getDirectCode()
                            + "_" + k.getSupermarketStoreCode())));
        }
        return directStoreMap;
    }

    /**
     * 构建商品信息
     * <p>
     * 门店map 系统编码+商超门店编码 构成唯一
     *
     * @param directCodeSet        系统编码集合
     * @param deliveryPartyCodeSet 商超门店编码集合
     * @param kaGoodsCodes         商超商品编码集合
     */
    private Map<String, List<DirectProductVo>> buildProductInfo(Set<String> directCodeSet,
                                                                Set<String> deliveryPartyCodeSet,
                                                                Set<String> kaGoodsCodes) {
        Map<String, List<DirectProductVo>> directProductMap = Maps.newHashMap();
        if (CollectionUtil.isEmpty(directCodeSet)
                || CollectionUtil.isEmpty(deliveryPartyCodeSet)
                || CollectionUtil.isEmpty(kaGoodsCodes)) {
            return directProductMap;
        }
        /**
         * deliveryPartyCode 送达方编码
         * terminalCode 企业门店编码
         * storeCode 企业门店编码
         *
         * 三个字段含义一样 值一样
         */

        List<List<String>> kaProductCodeGroupList = Lists.partition(Lists.newArrayList(kaGoodsCodes), ConstantEnums.PRODUCT_SIZE.getValue());
        DirectProductDto directProductDto = new DirectProductDto();
        directProductDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        directProductDto.setTenantCode(TenantUtils.getTenantCode());
        directProductDto.setOnShelfStatus(BooleanEnum.TRUE.getCapital());
        directProductDto.setDirectCodes(new ArrayList<>(directCodeSet));
        directProductDto.setDeliveryPartyCodes(new ArrayList<>(deliveryPartyCodeSet));
        directProductDto.setOrderType(acceptanceInvoiceTypeRegister.type());
        kaProductCodeGroupList.forEach(kaProductCodeList -> {
            directProductDto.setKaProductCodes(kaProductCodeList);
            List<DirectProductVo> productList = this.directProductVoService.findByDirectProductDto(directProductDto);
            if (CollectionUtil.isNotEmpty(productList)) {
                directProductMap.putAll(productList.stream()
                        .filter(k -> BooleanEnum.TRUE.getCapital().equals(k.getOnShelfStatus()))
                        .filter(k -> StringUtils.isNotBlank(k.getDirectCode()))
                        .filter(k -> StringUtils.isNotBlank(k.getDeliveryPartyCode()))
                        .filter(k -> StringUtils.isNotBlank(k.getKaProductCode()))
                        .collect(Collectors.groupingBy(k -> k.getDirectCode()
                                + "_" + k.getDeliveryPartyCode()
                                + "_" + k.getKaProductCode())));
            }
        });
        return directProductMap;
    }


    /**
     * 已转换过的单据信息更新到最新的单据上
     * 并过滤当前状态不可转换的单据
     *
     * @param orderNumberList
     * @param entityList
     */
    private List<InvoiceAcceptanceEntity> updateOldDataToNewData(List<String> orderNumberList, List<InvoiceAcceptanceEntity> entityList) {
        if (CollectionUtil.isEmpty(orderNumberList)
                || CollectionUtil.isEmpty(entityList)) {
            return entityList;
        }
        //已转换的验收单数据
        List<InvoiceAcceptanceEntity> oldEntityList = this.invoiceAcceptanceRepository.findByOrderNumberList(TenantUtils.getTenantCode(), orderNumberList);
        Map<String, InvoiceAcceptanceEntity> oldMap = oldEntityList.stream()
                .collect(Collectors.toMap(InvoiceAcceptanceEntity::getOrderNumber, v -> v, (oldValue, newValue) -> newValue));
        //已转换的单据信息 单据ID和单据状态
        entityList.forEach(entity -> {
            InvoiceAcceptanceEntity oldEntity = oldMap.get(entity.getOrderNumber());
            if (oldEntity != null) {
                entity.setId(oldEntity.getId());
                entity.setOrderStatus(oldEntity.getOrderStatus());
            } else {
                entity.setId(null);
            }
        });

        return entityList;
    }

    /**
     * 过滤正在转换的验收单号
     *
     * @param orderNumberList 验收单号集合
     * @return void
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.14 1:45
     */
    @Override
    public void filterSwitchIngOrderNumberList(List<String> orderNumberList) {
        if (CollectionUtil.isEmpty(orderNumberList)) {
            return;
        }
        //获取redis已存在key
        List<Object> updateList = redisCrmService.hmget(AcceptanceOrderConstant.ACCEPTANCE_ORDER_CONVERT_LOCK, new HashSet<>(orderNumberList));

        if (CollectionUtil.isNotEmpty(updateList)) {
            orderNumberList.removeAll(updateList.stream()
                    .filter(Objects::nonNull)
                    .filter(k -> StringUtils.isNotEmpty(k.toString()))
                    .map(Object::toString)
                    .collect(Collectors.toList()));
        }

        Map<String, String> redisMap = orderNumberList.stream()
                .collect(Collectors.toMap(key -> key, key -> key, (oldValue, newValue) -> newValue));

        //当前执行效率很高,锁定30分钟即可
        redisCrmService.hmset(AcceptanceOrderConstant.ACCEPTANCE_ORDER_CONVERT_LOCK, redisMap, TimeUnit.SECONDS, AcceptanceOrderConstant.ACCEPTANCE_ORDER_REDIS_TIME);
    }

    /**
     * 零售商验收单导入保存
     * @param kmsInvoiceAcceptanceGrabEntities
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void importSave(List<KmsInvoiceAcceptanceGrabEntity> kmsInvoiceAcceptanceGrabEntities) {
        if (CollectionUtils.isEmpty(kmsInvoiceAcceptanceGrabEntities)){
            return;
        }
        kmsInvoiceAcceptanceGrabEntities.forEach(aa ->{
            aa.setTenantCode(TenantUtils.getTenantCode());
            aa.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            aa.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            aa.setInvoicesSource("手动导入");
            this.kmsInvoiceAcceptanceGrabRepository.save(aa);
            this.kmsInvoiceAcceptanceGoodsGrabRepository.saveBatch(aa.getGoodsList());
        });
    }

    /**
     * 获取数据字典中设置的转换时间
     * @param type
     * @return
     */
    private String getDays(String type){
        Map<String, List<DictDataVo>> fomatMap = this.dictDataVoService.findByDictTypeCodeList(Lists.newArrayList(DirectConstant.KMS_AUTO_CONVERTS));
        final List<DictDataVo> vos = fomatMap.get(DirectConstant.KMS_AUTO_CONVERTS);
        final Optional<String> first =
                vos.stream()
                        .filter(a -> a.getDictCode().equals(type))
                        .map(DictDataVo::getDictValue)
                        .findFirst();
      int past = Integer.parseInt(first.orElse(StringUtils.EMPTY));
        //获取数据字典中设置的时间
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.DATE, calendar.get(Calendar.DATE) - past);
        String day = sdf.format(calendar.getTime());
        return day;
    }
}
