package com.biz.crm.kms.business.invoice.acceptance.local.service.internal;

import cn.hutool.core.collection.CollectionUtil;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.kms.business.direct.product.sdk.vo.DirectProductUnitVo;
import com.biz.crm.kms.business.direct.product.sdk.vo.DirectProductVo;
import com.biz.crm.kms.business.direct.sdk.vo.DirectVo;
import com.biz.crm.kms.business.direct.store.sdk.vo.DirectStoreVo;
import com.biz.crm.kms.business.invoice.acceptance.local.entity.InvoiceAcceptanceEntity;
import com.biz.crm.kms.business.invoice.acceptance.local.entity.InvoiceAcceptanceGoodsEntity;
import com.biz.crm.kms.business.invoice.acceptance.local.repository.InvoiceAcceptanceGoodsRepository;
import com.biz.crm.kms.business.invoice.acceptance.local.repository.InvoiceAcceptanceRepository;
import com.biz.crm.kms.business.invoice.acceptance.local.repository.KmsInvoiceAcceptanceGrabRepository;
import com.biz.crm.kms.business.invoice.acceptance.local.service.InvoiceAcceptanceOrderGrabsThreadService;
import com.biz.crm.kms.business.invoice.acceptance.sdk.constant.AcceptanceOrderConstant;
import com.biz.crm.kms.business.invoice.acceptance.sdk.enums.AcceptanceStatus;
import com.biz.crm.kms.business.invoice.sdk.enums.DirectEnum;
import com.biz.crm.kms.business.invoice.sdk.enums.GrabTransStatus;
import com.biz.crm.mn.common.base.service.RedisCrmService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

/**
 * 单据转换异步方法
 *
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2022.12.15 13:53
 */
@Slf4j
@Service
public class InvoiceAcceptanceOrderGrabsThreadServiceImpl implements InvoiceAcceptanceOrderGrabsThreadService {

    @Autowired
    private InvoiceAcceptanceRepository invoiceAcceptanceRepository;

    @Autowired
    private InvoiceAcceptanceGoodsRepository invoiceAcceptanceGoodsRepository;

    @Autowired
    private KmsInvoiceAcceptanceGrabRepository kmsInvoiceAcceptanceGrabRepository;

    @Autowired
    private RedisCrmService redisCrmService;

    @Autowired(required = false)
    private RedisTemplate redisTemplate;

    @Async("grabThread")
    @Override
    @Transactional(propagation = Propagation.NOT_SUPPORTED, rollbackFor = Exception.class)
    public void saveAcceptanceOrderData(List<InvoiceAcceptanceEntity> entityList, Map<String, List<DirectVo>> directMap, Map<String, List<DirectStoreVo>> directStoreMap, Map<String, List<DirectProductVo>> directProductMap) {
        log.info("--  进入线程池存储数据  --");
        if (CollectionUtil.isEmpty(entityList)) {
            return;
        }
        List<String> orderNumbers = entityList.stream().map(InvoiceAcceptanceEntity::getOrderNumber).collect(Collectors.toList());
        Set<String> orderNumberSet = new HashSet<>();
        Set<String> orderNumberSuccessSet = new HashSet<>();
        entityList.stream()
                .filter(Objects::nonNull)
                .filter(k -> StringUtils.isNotBlank(k.getOrderNumber()))
                .forEach(entity -> {
                    try {
                        //清除数据中的售达方，企业门店，区域等数据 第一步
                        this.clearTransformInfoOne(entity,directMap);
                        //转换验收单头数据  第二步
                        String deliveryCode = this.buildOrderInfoTwo(entity, directStoreMap);
                        //转换验收单商品行数据  第三步
                        this.convertAcceptanceOrderItemThree(entity, directProductMap,deliveryCode);
                        //设置状态  第四步
                        this.setOrderStatusAndMsgFour(entity);

                        this.invoiceAcceptanceRepository.saveOrUpdate(entity);
                        if (Objects.equals(AcceptanceStatus.S200.getDictCode(), entity.getOrderStatus())){
                            orderNumberSuccessSet.add(entity.getOrderNumber());
                        }else {
                            orderNumberSet.add(entity.getOrderNumber());
                        }
                    } catch (Exception e) {
                        log.error("保存验收单数据失败", e);
                        entity.setOrderStatus(AcceptanceStatus.S100.getDictCode());
                        entity.setTransFlag(BooleanEnum.TRUE.getCapital());
                        entity.setOrderStatusMsg("转换时异常!");
                        try {
                            this.invoiceAcceptanceRepository.saveOrUpdate(entity);
                            orderNumberSet.add(entity.getOrderNumber());
                        } catch (Exception ex) {
                            log.error("", ex);
                        }
                    }
                });

        if (CollectionUtil.isEmpty(orderNumberSet) && CollectionUtils.isEmpty(orderNumberSuccessSet)) {
            return;
        }
        //处理数据前先进行解锁
        redisCrmService.hdel(AcceptanceOrderConstant.ACCEPTANCE_ORDER_CONVERT_LOCK, orderNumbers);
        try {
            if (!CollectionUtils.isEmpty(orderNumberSet)){
                kmsInvoiceAcceptanceGrabRepository.updateTransStatusByOrderNumberList(TenantUtils.getTenantCode(),
                        GrabTransStatus.TRANS_FAIL, new ArrayList<>(orderNumberSet));
            }

            if (!CollectionUtils.isEmpty(orderNumberSuccessSet)){
                kmsInvoiceAcceptanceGrabRepository.updateTransStatusByOrderNumberList(TenantUtils.getTenantCode(),
                        GrabTransStatus.HAVE_TRANS, new ArrayList<>(orderNumberSuccessSet));
            }


        } catch (Exception e) {
            log.error("", e);
        }
    }

    /**
     * 清除数据中的售达方，企业门店，区域等数据 第一步
     *
     * @param entity
     */
    private void clearTransformInfoOne(InvoiceAcceptanceEntity entity,Map<String, List<DirectVo>> directMap) {
        entity.setSoldToPartyName(null);
        entity.setSoldToPartyCode(null);
        entity.setStoreCode(null);
        entity.setStoreName(null);
        entity.setAreaName(null);
        //验收时间转换
        if (StringUtils.isNotBlank(entity.getAcceptanceDate())){
            try {
                entity.setAcceptanceTime(new SimpleDateFormat("yyyy-MM-dd").parse(entity.getAcceptanceDate()));
            } catch (ParseException e) {
                e.printStackTrace();
            }
        }
        //业态业务单元赋值
        List<DirectVo> directVos = directMap.get(entity.getDirectCode());
        if (CollectionUtils.isEmpty(directVos)){
            entity.setOrderStatusMsg("不存在该系统");
        }else {
            DirectVo directVo = directVos.get(0);
            entity.setBusinessFormatCode(directVo.getBusinessFormatCode());
            entity.setBusinessUnitCode(directVo.getBusinessUnitCode());
            entity.setDirectCode(directVo.getDirectCode());
            entity.setDirectName(directVo.getSupermarketName());
            entity.setKaCode(directVo.getSupermarketCode());
            entity.setKaName(directVo.getSupermarketName());
        }
    }

    /**
     * 转换验收单头数据  第二步
     *
     * @param entity
     * @param directStoreMap
     */
    private String buildOrderInfoTwo(InvoiceAcceptanceEntity entity, Map<String, List<DirectStoreVo>> directStoreMap) {
        entity.setOrderStatus(AcceptanceStatus.S100.getDictCode());
        entity.setTransFlag(BooleanEnum.TRUE.getCapital());

        if (StringUtils.isBlank(entity.getDirectCode())) {
            entity.setOrderStatusMsg("单据系统编码为空");
            return null;
        }
        if (StringUtils.isBlank(entity.getKaStoreCode())) {
            entity.setOrderStatusMsg("单据商超门店编码为空");
            return null;
        }
        List<DirectStoreVo> directStoreVoList = directStoreMap.get(entity.getDirectCode() + "_" + entity.getKaStoreCode());
        DirectStoreVo directStoreVo = null;
        if(DirectEnum.GAOXIN_TJ.getDictCode().equals(entity.getDirectCode())||DirectEnum.GAOXIN_BJ.getDictCode().equals(entity.getDirectCode())){
            //区域名称查找
            if (DirectEnum.GAOXIN_QY1.getDictCode().equals(entity.getAreaCode())) {
                entity.setAreaName(DirectEnum.GAOXIN_QY1.getDesc());
            } else if (DirectEnum.GAOXIN_QY2.getDictCode().equals(entity.getAreaCode())) {
                entity.setAreaName(DirectEnum.GAOXIN_QY2.getDesc());
            } else if (DirectEnum.GAOXIN_QY3.getDictCode().equals(entity.getAreaCode())) {
                entity.setAreaName(DirectEnum.GAOXIN_QY3.getDesc());
            } else if (DirectEnum.GAOXIN_QY4.getDictCode().equals(entity.getAreaCode())) {
                entity.setAreaName(DirectEnum.GAOXIN_QY4.getDesc());
            } else if (DirectEnum.GAOXIN_QY5.getDictCode().equals(entity.getAreaCode())) {
                entity.setAreaName(DirectEnum.GAOXIN_QY5.getDesc());
            } else {
                entity.setOrderStatusMsg("该单据不存在区域");
            }
            List<DirectStoreVo> directStoreVos = directStoreVoList.stream().filter(directStoreVo1 -> entity.getAreaName().equals(directStoreVo1.getRetailerRegion())).collect(Collectors.toList());
            if (CollectionUtils.isEmpty(directStoreVos)) {
                entity.setOrderStatus(AcceptanceStatus.S100.getDictCode());
                entity.setOrderStatusMsg("该系统下的门店中未找到匹配的区域");
            }
            directStoreVo = directStoreVos.get(0);
            entity.setSoldToPartyCode(directStoreVo.getSoldToPartyCode());
            entity.setSoldToPartyName(directStoreVo.getSoldToPartyName());
            entity.setDeliveryPartyCode(directStoreVo.getTerminalCode());
            entity.setDeliveryPartyName(directStoreVo.getTerminalName());
            entity.setBusinessArea(directStoreVo.getBusinessArea());
            entity.setProvinceCode(directStoreVo.getProvinceCode());
            entity.setProvinceName(directStoreVo.getProvinceName());
            entity.setOrderStatus(AcceptanceStatus.S200.getDictCode());
            return directStoreVo.getRetailerTerminalCode();
        }else {
            if (CollectionUtil.isEmpty(directStoreVoList)) {
                entity.setOrderStatusMsg("未查询到与企业门店对应关系");
                return null;
            } else if (directStoreVoList.size() > 1) {
                entity.setOrderStatusMsg("商超门店与企业门店对应关系存在[" + directStoreVoList.size() + "]条");
                return null;
            } else {
                directStoreVo = directStoreVoList.get(0);
            }
            if (StringUtils.isEmpty(entity.getKaStoreName())) {
                entity.setKaStoreName(directStoreVo.getSupermarketStoreName());
            }
            entity.setSoldToPartyCode(directStoreVo.getSoldToPartyCode());
            entity.setSoldToPartyName(directStoreVo.getSoldToPartyName());
            entity.setDeliveryPartyCode(directStoreVo.getTerminalCode());
            entity.setDeliveryPartyName(directStoreVo.getTerminalName());
            entity.setAreaName(directStoreVo.getRetailerRegion());
            entity.setBusinessArea(directStoreVo.getBusinessArea());
            entity.setProvinceCode(directStoreVo.getProvinceCode());
            entity.setProvinceName(directStoreVo.getProvinceName());
            entity.setOrderStatus(AcceptanceStatus.S200.getDictCode());
            return directStoreVo.getRetailerTerminalCode();
        }
    }

    /**
     * 转换验收单商品行数据  第三步
     *
     * @param entity
     * @param directProductMap
     * @return void
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.14 4:24
     */
    private void convertAcceptanceOrderItemThree(InvoiceAcceptanceEntity entity,
                                                 Map<String, List<DirectProductVo>> directProductMap ,String deliveryCode) {
        if (CollectionUtils.isEmpty(entity.getGoodsList())) {
            entity.setOrderStatus(AcceptanceStatus.S100.getDictCode());
            entity.setOrderStatusMsg("商品行信息数据为空");
            return;
        }
        //订货单总金额（含税）
        AtomicReference<BigDecimal> finalOrderAmount = new AtomicReference<>(BigDecimal.ZERO);
        //订货单总金额（不含税）
        AtomicReference<BigDecimal> finalOrderAmountNot = new AtomicReference<>(BigDecimal.ZERO);
        //验收单总金额（含税）
        AtomicReference<BigDecimal> finalAcceptanceAmount = new AtomicReference<>(BigDecimal.ZERO);
        //验收单总金额（不含税）
        AtomicReference<BigDecimal> finalAcceptanceAmountNot = new AtomicReference<>(BigDecimal.ZERO);
        String createAccount = entity.getCreateAccount();
        String createName = entity.getCreateName();
        String modifyAccount = entity.getModifyAccount();
        String modifyName = entity.getModifyName();
        BigDecimal decimal = new BigDecimal(1);
        Map<String, BigDecimal> map = this.redisTemplate.hasKey(AcceptanceOrderConstant.CACHE_PRODUCT_TAX_RATE) ? (Map<String, BigDecimal>) this.redisTemplate.opsForValue().get(AcceptanceOrderConstant.CACHE_PRODUCT_TAX_RATE) : null;
        entity.getGoodsList().forEach(orderGoodsEntity -> {
            this.clearTransformItem(orderGoodsEntity);
            orderGoodsEntity.setCreateAccount(createAccount);
            orderGoodsEntity.setCreateName(createName);
            orderGoodsEntity.setModifyAccount(modifyAccount);
            orderGoodsEntity.setModifyName(modifyName);
            orderGoodsEntity.setCurCompanyUnitOrderQuantity(BigDecimal.ZERO);
            DirectProductVo productVo = this.getGoodsInfo(orderGoodsEntity, entity, directProductMap,deliveryCode);
            this.buildGoodsInfo(orderGoodsEntity, entity, productVo);
            // 设置含税未税
            orderGoodsEntity = setAccount(orderGoodsEntity,decimal,map);
            if (Objects.nonNull(orderGoodsEntity.getOrderAmount())) {
                finalOrderAmount.set(finalOrderAmount.get().add(orderGoodsEntity.getOrderAmount()));
            }
            if (Objects.nonNull(orderGoodsEntity.getOrderAmountNot())) {
                finalOrderAmountNot.set(finalOrderAmountNot.get().add(orderGoodsEntity.getOrderAmountNot()));
            }
            if (Objects.nonNull(orderGoodsEntity.getAcceptanceAmount())) {
                finalAcceptanceAmount.set(finalAcceptanceAmount.get().add(orderGoodsEntity.getAcceptanceAmount()));
            }
            if (Objects.nonNull(orderGoodsEntity.getAcceptanceAmountNot())) {
                finalAcceptanceAmountNot.set(finalAcceptanceAmountNot.get().add(orderGoodsEntity.getAcceptanceAmountNot()));
            }
        });
        Boolean falg = !entity.getGoodsList().stream().map(InvoiceAcceptanceGoodsEntity::getOrderStatus).collect(Collectors.toList()).contains(AcceptanceStatus.S100.getDictCode());
        if (entity.getFinalOrderAmount() == null && falg){
            entity.setFinalOrderAmount(finalOrderAmount.get());
        }
        if (entity.getFinalOrderAmountNot()==null && falg){
            entity.setFinalOrderAmountNot(finalOrderAmountNot.get());
        }
        BigDecimal acceptanceAmount = entity.getFinalAcceptanceAmount();
        if (falg && (Objects.isNull(acceptanceAmount) || acceptanceAmount.compareTo(BigDecimal.ZERO) == 0)){
            entity.setFinalAcceptanceAmount(finalAcceptanceAmount.get());
        }
        BigDecimal acceptanceAmountNot = entity.getFinalAcceptanceAmountNot();
        if (falg && (Objects.isNull(acceptanceAmountNot) || acceptanceAmountNot.compareTo(BigDecimal.ZERO) == 0)){
            entity.setFinalAcceptanceAmountNot(finalAcceptanceAmountNot.get());
        }
        //删除原单号下的商品行信息
        this.invoiceAcceptanceGoodsRepository.deleteByKaOrderNumber(entity.getTenantCode(), entity.getKaOrderNumber());
        this.invoiceAcceptanceGoodsRepository.saveBatch(entity.getGoodsList());
    }
    /**
     * 设置含税未税
     */
    private InvoiceAcceptanceGoodsEntity setAccount(InvoiceAcceptanceGoodsEntity orderGoodsEntity, BigDecimal decimal, Map<String, BigDecimal> map) {
        String goodsCode = orderGoodsEntity.getGoodsCode();
        if (!StringUtils.isEmpty(goodsCode)) {
            if (Objects.isNull(map)) {
                return orderGoodsEntity;
            }
            BigDecimal bigDecimal = map.get(goodsCode);
            if (Objects.isNull(bigDecimal)){
                return orderGoodsEntity;
            }
            //如果未税金额为0  则用含税金额反算未税金额
            if(Objects.nonNull(orderGoodsEntity.getAcceptanceAmount()) &&
                    !orderGoodsEntity.getAcceptanceAmount().equals(BigDecimal.ZERO) &&
                    Objects.isNull(orderGoodsEntity.getAcceptanceAmountNot())){
                BigDecimal amount = orderGoodsEntity.getAcceptanceAmount().divide(BigDecimal.ONE.add(bigDecimal),4, BigDecimal.ROUND_HALF_UP);
                orderGoodsEntity.setAcceptanceAmountNot(amount);
            }
            BigDecimal acceptanceAmount = orderGoodsEntity.getAcceptanceAmount();
            BigDecimal acceptanceAmountNot = orderGoodsEntity.getAcceptanceAmountNot();
            if (Objects.nonNull(acceptanceAmount) &&Objects.isNull(acceptanceAmountNot)) {
                if (acceptanceAmount.compareTo(BigDecimal.ZERO) != 0) {
                    BigDecimal multiply = acceptanceAmount.divide(decimal.add(bigDecimal).setScale(4, BigDecimal.ROUND_HALF_UP), 4, BigDecimal.ROUND_HALF_UP);
                    orderGoodsEntity.setAcceptanceAmountNot(multiply);
                }
            }
            if (Objects.nonNull(acceptanceAmountNot) && Objects.isNull(acceptanceAmount)) {
                if (acceptanceAmountNot.compareTo(BigDecimal.ZERO) != 0) {
                    BigDecimal multiply = acceptanceAmountNot.multiply(decimal.add(bigDecimal).setScale(4, BigDecimal.ROUND_HALF_UP));
                    orderGoodsEntity.setAcceptanceAmount(multiply.setScale(4, BigDecimal.ROUND_HALF_UP));
                }
            }
            //含税单价
            if(Objects.isNull(orderGoodsEntity.getAcceptanceOneAmount()) && Objects.nonNull(orderGoodsEntity.getAcceptanceAmount())&& Objects.nonNull(orderGoodsEntity.getCurUnitAcceptanceQuantity())){
                if (orderGoodsEntity.getCurUnitAcceptanceQuantity().compareTo(BigDecimal.ZERO) != 0 && orderGoodsEntity.getAcceptanceAmount().compareTo(BigDecimal.ZERO) != 0 ){
                    BigDecimal oneAmount = orderGoodsEntity.getAcceptanceAmount().divide(orderGoodsEntity.getCurUnitAcceptanceQuantity(),4,BigDecimal.ROUND_HALF_UP);
                    orderGoodsEntity.setAcceptanceOneAmount(oneAmount);
                }
            }
            //未税单价
            if(Objects.isNull(orderGoodsEntity.getAcceptanceOneAmountNot()) && Objects.nonNull(orderGoodsEntity.getAcceptanceAmountNot())&& Objects.nonNull(orderGoodsEntity.getCurUnitAcceptanceQuantity())){
                if (orderGoodsEntity.getCurUnitAcceptanceQuantity().compareTo(BigDecimal.ZERO) != 0 && orderGoodsEntity.getAcceptanceAmountNot().compareTo(BigDecimal.ZERO) != 0 ){
                    BigDecimal oneAmount = orderGoodsEntity.getAcceptanceAmountNot().divide(orderGoodsEntity.getCurUnitAcceptanceQuantity(),4,BigDecimal.ROUND_HALF_UP);
                    orderGoodsEntity.setAcceptanceOneAmountNot(oneAmount);
                }
            }
        }
        return orderGoodsEntity;
    }

    /**
     * 清理转换信息
     *
     * @param entity
     */
    private void clearTransformItem(InvoiceAcceptanceGoodsEntity entity) {
        entity.setGoodsCode(null);
        entity.setGoodsName(null);
        entity.setOrderStatus(AcceptanceStatus.S100.getDictCode());
        entity.setId(null);
    }

    /**
     * 构建商品转换明细信息
     *
     * @param orderGoodsEntity
     * @param entity
     * @param directProductMap
     */
    private DirectProductVo getGoodsInfo(InvoiceAcceptanceGoodsEntity orderGoodsEntity, InvoiceAcceptanceEntity entity,
                                         Map<String, List<DirectProductVo>> directProductMap,String deliveryCode) {
        DirectProductVo productVo = null;
        orderGoodsEntity.setDeliveryPartyCode(entity.getDeliveryPartyCode());
        orderGoodsEntity.setDeliveryPartyName(entity.getDeliveryPartyName());
        orderGoodsEntity.setDirectCode(entity.getDirectCode());
        if (StringUtils.isBlank(orderGoodsEntity.getKaGoodsCode())) {
            orderGoodsEntity.setOrderStatusMsg("商超商品编码为空");
            return null;
        }
        List<DirectProductVo> productVoList = new ArrayList<>();
        List<DirectProductVo> productVoLists = directProductMap.get(entity.getDirectCode() + "_" +
                entity.getDeliveryPartyCode() + "_" + orderGoodsEntity.getKaGoodsCode());
        if (CollectionUtil.isNotEmpty(productVoLists)){
            productVoList.addAll(productVoLists);
        }
        //如果送达方编码没有找到，那么会通过关联的送达方编码去查
        if (CollectionUtil.isEmpty(productVoList) && StringUtils.isNotBlank(deliveryCode)){
            List<DirectProductVo> productVos = directProductMap.get(entity.getDirectCode() + "_" +
                    deliveryCode + "_" + orderGoodsEntity.getKaGoodsCode());
            if (CollectionUtil.isNotEmpty(productVos)){
                productVoList.addAll(productVos);
            }
        }
        if (CollectionUtil.isEmpty(productVoList)) {
            orderGoodsEntity.setOrderStatusMsg("未查询到商品对应关系");
        } else if (productVoList.size() == 1) {
            productVo = productVoList.get(0);
        } else {
            int allSize = productVoList.size();
            //匹配到多条数据  用是否默认标记为是的数据
            productVoList = productVoList.stream()
                    .filter(k -> BooleanEnum.TRUE.getCapital().equals(k.getIsDefault()))
                    .collect(Collectors.toList());
            if (CollectionUtil.isEmpty(productVoList)) {
                orderGoodsEntity.setOrderStatusMsg("共[" + allSize + "]条商品对应关系,无默认对应关系");
            } else if (productVoList.size() > 1) {
                orderGoodsEntity.setOrderStatusMsg("[" + productVoList.size() + "]条默认商品对应关系");
            } else {
                productVo = productVoList.get(0);
            }
        }
        return productVo;

    }


    /**
     * 构建商品转换明细信息
     *
     * @param orderGoodsEntity
     * @param entity
     * @param productVo
     */
    private void buildGoodsInfo(InvoiceAcceptanceGoodsEntity orderGoodsEntity, InvoiceAcceptanceEntity entity,
                                DirectProductVo productVo) {
        if (Objects.isNull(productVo)) {
            return;
        }
        orderGoodsEntity.setGoodsCode(productVo.getProductCode());
        orderGoodsEntity.setGoodsName(productVo.getProductName());
        if (Objects.isNull(orderGoodsEntity.getCurUnitAcceptanceQuantity())) {
            orderGoodsEntity.setOrderStatusMsg("无验收数量");
            return;
        }
        List<DirectProductUnitVo> unitList = productVo.getUnitList();
        if (CollectionUtils.isEmpty(unitList)) {
            orderGoodsEntity.setOrderStatusMsg("未匹配企业商品单位转换关系");
            return;
        }

        if (StringUtils.isNotBlank(orderGoodsEntity.getCurUnit())){
            //商超单位不为空
            unitList = unitList.stream()
                    .filter(vo -> Objects.nonNull(vo.getKaUnitQuantity())
                            && Objects.nonNull(vo.getUnitQuantity()))
                    .filter(vo -> Objects.equals(orderGoodsEntity.getCurUnit(), vo.getKaUnitCode()))
                    .collect(Collectors.toList());
        }
        //当商超单位找不到对应的单位关系时，这种情况就找默认上架产品+相应单据+单位为空的第一个转换关系
        if (CollectionUtils.isEmpty(unitList)){
            //商超单位为空
            unitList = productVo.getUnitList().stream()
                    .filter(vo -> Objects.nonNull(vo.getKaUnitQuantity())
                            && Objects.nonNull(vo.getUnitQuantity()))
                    .filter(vo -> StringUtils.isBlank(vo.getKaUnitCode()))
                    .collect(Collectors.toList());
        }

        if (CollectionUtils.isEmpty(unitList)) {
            orderGoodsEntity.setOrderStatusMsg("未匹配企业商品单位转换关系");
            return;
        }

        DirectProductUnitVo unitVo = unitList.get(0);
        if (Objects.isNull(orderGoodsEntity.getAcceptanceAmount()) || Objects.isNull(orderGoodsEntity.getCurUnitAcceptanceQuantity())){
            orderGoodsEntity.setOrderStatusMsg("该验收单明细的金额或数量为空");
        }
        //转换后验收数量
        if (ObjectUtils.isNotEmpty(orderGoodsEntity.getCurUnitAcceptanceQuantity()) &&
                Objects.nonNull(unitVo.getUnitQuantity()) &&
                Objects.nonNull(unitVo.getKaUnitQuantity()) &&
                orderGoodsEntity.getCurUnitAcceptanceQuantity().compareTo(BigDecimal.ZERO) != 0 &&
                unitVo.getUnitQuantity().compareTo(BigDecimal.ZERO) != 0 &&
                unitVo.getKaUnitQuantity().compareTo(BigDecimal.ZERO) != 0){
            BigDecimal curCompanyUnitOrderQuantityOne = orderGoodsEntity.getCurUnitAcceptanceQuantity().multiply(unitVo.getUnitQuantity());
            BigDecimal curCompanyUnitOrderQuantitytwo = curCompanyUnitOrderQuantityOne.divide(unitVo.getKaUnitQuantity(),4,BigDecimal.ROUND_HALF_UP);
            orderGoodsEntity.setCurCompanyUnitOrderQuantity(curCompanyUnitOrderQuantitytwo);
        }
        orderGoodsEntity.setCurCompanyUnitCode(unitVo.getUnitCode());
        orderGoodsEntity.setCurCompanyUnitName(unitVo.getUnitName());
        orderGoodsEntity.setOrderStatus(AcceptanceStatus.S200.getDictCode());
        orderGoodsEntity.setOrderStatusMsg(AcceptanceOrderConstant.ACCEPTANCE_ORDER_CONVERT_SUCCESS_MSG);
    }

    /**
     * 设置状态  第四步
     *
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.14 4:24
     */
    private void setOrderStatusAndMsgFour(InvoiceAcceptanceEntity entity) {
        long errorCount = 0;
        if (CollectionUtil.isNotEmpty(entity.getGoodsList())) {
            errorCount = entity.getGoodsList().stream()
                    .filter(k -> Objects.equals(AcceptanceStatus.S100.getDictCode(), k.getOrderStatus()))
                    .count();
        }
        if (errorCount > 0) {
            entity.setOrderStatus(AcceptanceStatus.S100.getDictCode());
            String orderStatusMsg = entity.getOrderStatusMsg();
            if (StringUtils.isNotBlank(orderStatusMsg)) {
                orderStatusMsg = String.format("%s||%s", orderStatusMsg, "商品行信息存在转换失败数据");
            } else {
                orderStatusMsg = "商品行信息存在转换失败数据";
            }
            entity.setOrderStatusMsg(orderStatusMsg);
        } else if (Objects.equals(AcceptanceStatus.S200.getDictCode(), entity.getOrderStatus())) {
            entity.setOrderStatusMsg(AcceptanceOrderConstant.ACCEPTANCE_ORDER_CONVERT_SUCCESS_MSG);

        }
    }
}
