package com.biz.crm.kms.business.invoice.acceptance.local.service.internal;

import com.biz.crm.kms.business.invoice.acceptance.local.entity.InvoiceAcceptanceEntity;
import com.biz.crm.kms.business.invoice.acceptance.local.entity.InvoiceAcceptanceGoodsEntity;
import com.biz.crm.kms.business.invoice.acceptance.local.repository.InvoiceAcceptanceGoodsRepository;
import com.biz.crm.kms.business.invoice.acceptance.local.repository.InvoiceAcceptanceRepository;
import com.biz.crm.kms.business.invoice.acceptance.local.service.InvoiceAcceptanceGoodsService;
import com.biz.crm.kms.business.invoice.acceptance.local.service.InvoiceAcceptanceService;
import com.biz.crm.kms.business.invoice.acceptance.sdk.enums.AcceptanceStatus;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import java.util.List;
import java.util.stream.Collectors;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

/**
 * 验收单表(InvoiceAcceptance)表服务实现类
 *
 * @author pengxi
 * @date 2022/10/10
 */
@Slf4j
@Service
public class InvoiceAcceptanceServiceImpl implements InvoiceAcceptanceService {

  @Autowired(required = false)
  private InvoiceAcceptanceRepository invoiceAcceptanceRepository;

  @Autowired(required = false)
  private InvoiceAcceptanceGoodsService invoiceAcceptanceGoodsService;

  @Autowired(required = false)
  private InvoiceAcceptanceGoodsRepository invoiceAcceptanceGoodsRepository;

  @Override
  public InvoiceAcceptanceEntity findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    InvoiceAcceptanceEntity entity = this.invoiceAcceptanceRepository.findById(id);
    if (entity == null) {
      return null;
    }
    List<InvoiceAcceptanceGoodsEntity> goodsEntities = this.invoiceAcceptanceGoodsService.findByOrderNumber(entity.getOrderNumber());
    entity.setGoodsList(goodsEntities);
    return entity;
  }

  @Override
  public void transformation(InvoiceAcceptanceEntity entity) {
    InvoiceAcceptanceEntity oldEntity = this.invoiceAcceptanceRepository.findByKaOrderNumber(TenantUtils.getTenantCode(), entity.getKaOrderNumber());
    if (oldEntity != null) {
      Validate.isTrue(entity.getVersionNumber() > oldEntity.getVersionNumber(), "已存在更大版本号转换记录！");
      Validate.isTrue(
          AcceptanceStatus.S100.getDictCode().equals(oldEntity.getOrderStatus())
              || AcceptanceStatus.S101.getDictCode().equals(oldEntity.getOrderStatus())
              || AcceptanceStatus.S200.getDictCode().equals(oldEntity.getOrderStatus())
          , String.format("当前验收单状态[%s]不能继续转换！", oldEntity.getOrderStatusMsg())
      );
      // 删除原记录
      this.invoiceAcceptanceRepository.deleteByKaOrderNumber(TenantUtils.getTenantCode(), entity.getKaOrderNumber());
      this.invoiceAcceptanceGoodsService.deleteByKaOrderNumber(entity.getKaOrderNumber());
    }
    // 创建新记录
    this.invoiceAcceptanceRepository.save(entity);
    // 保存商品详情
    if (CollectionUtils.isNotEmpty(entity.getGoodsList())) {
      this.invoiceAcceptanceGoodsService.createBatch(entity.getGoodsList());
    }
  }


  @Transactional(rollbackFor = Exception.class)
  @Override
  public void deleteByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids),"请选择数据!");
    List<InvoiceAcceptanceEntity> entities = invoiceAcceptanceRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities),"数据不存在或已删除!");
    String tenantCode = TenantUtils.getTenantCode();
    List<String> kaOrderNumbers = entities.stream().map(InvoiceAcceptanceEntity::getKaOrderNumber).distinct().collect(Collectors.toList());
    Validate.isTrue(CollectionUtils.isNotEmpty(kaOrderNumbers),"商超验收单编号不能为空!");
    this.invoiceAcceptanceGoodsRepository.deleteByKaOrderNumbers(tenantCode,kaOrderNumbers);
    this.invoiceAcceptanceRepository.deleteByKaOrderNumbers(tenantCode,kaOrderNumbers);
  }

  @Transactional(rollbackFor = Exception.class)
  @Override
  public InvoiceAcceptanceEntity create(InvoiceAcceptanceEntity invoiceAcceptanceEntity) {
    this.createValidate(invoiceAcceptanceEntity);
    this.invoiceAcceptanceRepository.saveOrUpdate(invoiceAcceptanceEntity);
    if (CollectionUtils.isNotEmpty(invoiceAcceptanceEntity.getGoodsList())){
      invoiceAcceptanceEntity.getGoodsList().forEach(aa ->{
        this.invoiceAcceptanceGoodsRepository.saveOrUpdate(aa);
      });
    }
    return invoiceAcceptanceEntity;
  }

  /**
   * 创建验证
   * @param invoiceAcceptanceEntity
   */
  private void createValidate(InvoiceAcceptanceEntity invoiceAcceptanceEntity) {
    Validate.notNull(invoiceAcceptanceEntity, "新增时，对象信息不能为空！");
    Validate.notBlank(invoiceAcceptanceEntity.getBusinessFormatCode(),"业态不能为空");
    Validate.notBlank(invoiceAcceptanceEntity.getBusinessUnitCode(),"业务单元不能为空");
    Validate.notBlank(invoiceAcceptanceEntity.getKaName(),"系统名称不能为空");
    Validate.notBlank(invoiceAcceptanceEntity.getInvoicesSource(),"单据来源不能为空");
    InvoiceAcceptanceEntity entity = this.invoiceAcceptanceRepository.findByKaOrderNumber(TenantUtils.getTenantCode(),invoiceAcceptanceEntity.getKaOrderNumber());
    Validate.isTrue(ObjectUtils.isEmpty(entity),"该验收单存在");
    invoiceAcceptanceEntity.setId(null);
  }
}
