package com.biz.crm.kms.business.invoice.acceptance.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.kms.business.invoice.acceptance.local.entity.InvoiceAcceptanceEntity;
import com.biz.crm.kms.business.invoice.acceptance.local.entity.InvoiceAcceptanceGoodsEntity;
import com.biz.crm.kms.business.invoice.acceptance.local.repository.InvoiceAcceptanceGoodsRepository;
import com.biz.crm.kms.business.invoice.acceptance.local.repository.InvoiceAcceptanceRepository;
import com.biz.crm.kms.business.invoice.acceptance.local.service.InvoiceAcceptanceGrabService;
import com.biz.crm.kms.business.invoice.acceptance.local.service.InvoiceAcceptanceService;
import com.biz.crm.kms.business.invoice.acceptance.sdk.constant.AcceptanceOrderConstant;
import com.biz.crm.kms.business.invoice.acceptance.sdk.enums.AcceptanceStatus;
import com.biz.crm.kms.business.invoice.acceptance.sdk.service.InvoiceAcceptanceVoService;
import com.biz.crm.kms.business.invoice.acceptance.sdk.vo.InvoiceAcceptanceGoodsVo;
import com.biz.crm.kms.business.invoice.acceptance.sdk.vo.InvoiceAcceptanceVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 验收单表(InvoiceAcceptance)表服务实现类
 *
 * @author pengxi
 * @date 2022/10/10
 */
@Slf4j
@Service
public class InvoiceAcceptanceVoServiceImpl implements InvoiceAcceptanceVoService {

    @Autowired(required = false)
    private InvoiceAcceptanceService invoiceAcceptanceService;

    @Autowired(required = false)
    private InvoiceAcceptanceRepository invoiceAcceptanceRepository;

    @Autowired(required = false)
    private InvoiceAcceptanceGrabService invoiceAcceptanceGrabService;

    @Autowired(required = false)
    private InvoiceAcceptanceGoodsRepository invoiceAcceptanceGoodsRepository;

    @Autowired(required = false)
    @Qualifier("nebulaToolkitService")
    private NebulaToolkitService nebulaToolkitService;

    @Override
    public InvoiceAcceptanceVo findDetailById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        InvoiceAcceptanceEntity entity = this.invoiceAcceptanceService.findDetailById(id);
        if (entity == null) {
            return null;
        }
        InvoiceAcceptanceVo invoiceAcceptanceVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, InvoiceAcceptanceVo.class, HashSet.class, ArrayList.class);
        if (CollectionUtils.isEmpty(entity.getGoodsList())) {
            return invoiceAcceptanceVo;
        }
        try {
            invoiceAcceptanceVo.setAcceptanceTime(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").parse(invoiceAcceptanceVo.getAcceptanceDate()));
        } catch (ParseException e) {
            e.printStackTrace();
        }
        List<InvoiceAcceptanceGoodsVo> goodsList = (List<InvoiceAcceptanceGoodsVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entity.getGoodsList(), InvoiceAcceptanceGoodsEntity.class, InvoiceAcceptanceGoodsVo.class, HashSet.class, ArrayList.class);
        invoiceAcceptanceVo.setGoodsList(goodsList);
        return invoiceAcceptanceVo;
    }

    /**
     * 通过编码查详情
     * @param code
     * @return
     */
    @Override
    public InvoiceAcceptanceVo findByDecomentCode(String code , String deliveryPartyCode){
        if (StringUtils.isBlank(code) || StringUtils.isBlank(deliveryPartyCode)){
            return null;
        }
        InvoiceAcceptanceEntity entity = this.invoiceAcceptanceRepository.findByDecomentCode(code, deliveryPartyCode);
        if (ObjectUtils.isEmpty(entity)){
            return null;
        }
        InvoiceAcceptanceVo invoiceAcceptanceVo = this.findDetailById(entity.getId());
        return invoiceAcceptanceVo;
    }

    @Override
    public Page<InvoiceAcceptanceVo> findByDirectCodes(Pageable pageable, String tenantCode,List<String> directCodes) {
        return invoiceAcceptanceRepository.findAcceptanceByDirectCodes(pageable,tenantCode,directCodes);
    }

    @Override
    public void manualSwitch(List<String> ids) {
        Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空！");
        List<InvoiceAcceptanceEntity> entities = this.invoiceAcceptanceRepository.findByIds(ids);
        Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
        Validate.isTrue(!CollectionUtils.isEmpty(entities), "未查询到数据，请刷新重试");
        Validate.isTrue(!ObjectUtils.notEqual(entities.size(),ids.size()), "数据转换个数不匹配");
        entities.forEach(entity -> Validate.isTrue(Lists.newArrayList(AcceptanceStatus.S200.getDictCode(), AcceptanceStatus.S101.getDictCode(), AcceptanceStatus.S100.getDictCode()).contains(entity.getOrderStatus())
                , String.format("验收单[%s]已确认,无法继续匹配", entity.getOrderNumber())));
        List<String> orderNumbers = entities.stream()
                .filter(k -> StringUtils.isNotBlank(k.getOrderNumber()))
                .map(InvoiceAcceptanceEntity::getOrderNumber).distinct().collect(Collectors.toList());
        //过滤正在转换的验收单号
        this.invoiceAcceptanceGrabService.filterSwitchIngOrderNumberList(orderNumbers);
        log.info("===== 验收单手动转换开始 ======");
        this.invoiceAcceptanceGrabService.manualSwitch(orderNumbers);
        log.info("===== 验收单手动转换完成 ======");
    }

    /**
     * 手动关联结算单
     * @param orderNumbers
     * @param statementCode
     */
    @Override
    public void match(List<String> orderNumbers, String statementCode) {
        Validate.isTrue(!CollectionUtils.isEmpty(orderNumbers)&&StringUtils.isNotBlank(statementCode),"验收单号和结算单号不能为空");
        //取消捆绑结算单
        this.invoiceAcceptanceRepository.cancel(statementCode);
        //捆绑结算单
        this.invoiceAcceptanceRepository.match(orderNumbers,statementCode);
    }

    /**
     * 根据条件查询单据包含转换失败(稽核汇总用)
     *
     * @param goodsCodes 产品编码
     * @param storeCodes 门店编码
     * @param orderCodes 采购单号
     * @param beginDate
     * @param endDate
     * @param soldToPartyCodes 售达方
     * @param directCodes 系统编码
     * @return
     */
    @Override
    public List<InvoiceAcceptanceGoodsVo> findAllByConditions(List<String> goodsCodes, List<String> storeCodes, List<String> orderCodes, String beginDate, String endDate, List<String> soldToPartyCodes, List<String> directCodes) {
        return this.invoiceAcceptanceGoodsRepository.findAllByConditions(goodsCodes,storeCodes,orderCodes,beginDate,endDate,soldToPartyCodes,directCodes);
    }

    @Override
    public List<InvoiceAcceptanceVo> findAcceptanceOrderVo(String beginDate, String endDate, List<String> directCodes) {
        return this.invoiceAcceptanceRepository.findAcceptanceOrderVo(beginDate,endDate,directCodes);
    }

    @Override
    public List<InvoiceAcceptanceGoodsVo> findByOrderNumberOrRelationKaOrderNumber(List<String> orderNumbers, List<String> relationKaOrderNumber) {
        return this.invoiceAcceptanceGoodsRepository.findByOrderNumberOrRelationKaOrderNumber(orderNumbers,relationKaOrderNumber, TenantUtils.getTenantCode());
    }


    /**
     * 拆单逻辑：
     * 1.校验是否转换成功
     * 2.是否已经拆过单
     * 3.校验子单中产品的金额和数量
     * 4.存储：改变原单状态，并且不能够继续拆单
     * 5.将新生成的子验收单保持到原验收单中
     */

    @Override
    public void splitAcceptance(List<InvoiceAcceptanceVo> invoiceAcceptanceVos) {
        if (CollectionUtils.isEmpty(invoiceAcceptanceVos)){
            return;
        }
        Validate.isTrue(CollectionUtils.isEmpty(invoiceAcceptanceVos.stream().filter(f ->StringUtils.isEmpty(f.getSplitBeforeId())).collect(Collectors.toList())),"拆分单据中原始单据ID不能为空");
        //查询原验收单
        InvoiceAcceptanceEntity entity = this.invoiceAcceptanceService.findDetailById(invoiceAcceptanceVos.get(0).getSplitBeforeId());
        if (Objects.isNull(entity)){
            return;
        }
        Validate.isTrue(AcceptanceStatus.S200.getDictCode().equals(entity.getOrderStatus()),"该单据未转换成功");
        Validate.isTrue(!BooleanEnum.TRUE.getCapital().equals(entity.getIsSplit()),"已拆分过的单据不能再次拆分");
        //原验收单中的产品数量
        Set<String> productNums = entity.getGoodsList().stream().map(InvoiceAcceptanceGoodsEntity::getGoodsCode).collect(Collectors.toSet());
        if (CollectionUtils.isEmpty(productNums)){
            return;
        }
        //原验收单中的产品与对应的数量Map
        Map<String, BigDecimal> productsMapTotal = Maps.newHashMap();
        Map<String, BigDecimal> productsUnitMapTotal = Maps.newHashMap();
        for (String product : productNums) {
            BigDecimal quentitys = entity.getGoodsList().stream().filter(f ->product.equals(f.getGoodsCode())).map(InvoiceAcceptanceGoodsEntity::getCurUnitAcceptanceQuantity).reduce(BigDecimal.ZERO, BigDecimal::add);
            BigDecimal nowQunantitys = entity.getGoodsList().stream().filter(f ->product.equals(f.getGoodsCode())).map(InvoiceAcceptanceGoodsEntity::getCurCompanyUnitOrderQuantity).reduce(BigDecimal.ZERO, BigDecimal::add);
            productsMapTotal.put(product, quentitys);
            productsUnitMapTotal.put(product, nowQunantitys.divide(quentitys,10,BigDecimal.ROUND_HALF_UP));

        }
        //子验收单中的产品数量
        Set<String> goods = Sets.newHashSet();
        //子验收单中的产品与对应的数量Map
        Map<String, BigDecimal> productsMap = Maps.newHashMap();
        //记录子验收单中的数量，用来改变自验收单的验收单号
        Integer integer = 1;
        String acceptanceSplitOrders ="";
        List<InvoiceAcceptanceEntity> entities = new ArrayList<>();
        List<InvoiceAcceptanceGoodsEntity> invoiceAcceptanceGoodsEntityList = new ArrayList<>();
        for (InvoiceAcceptanceVo acceptanceVo : invoiceAcceptanceVos) {
            acceptanceVo.setId(null);
            acceptanceVo.setAcceptanceSplitOrders(null);
            acceptanceVo.setIsSlips(BooleanEnum.TRUE.getCapital());
            acceptanceVo.setTenantCode(TenantUtils.getTenantCode());
            acceptanceVo.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            acceptanceVo.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            acceptanceVo.setOrderNumber(entity.getOrderNumber()+"-"+integer);
            acceptanceVo.setKaOrderNumber(entity.getKaOrderNumber()+"-"+integer);
            if (StringUtils.isNotBlank(entity.getRelateOrderNumber())){
                acceptanceVo.setRelateOrderNumber(entity.getRelateOrderNumber()+"-"+integer);
            }
            if (StringUtils.isNotBlank(entity.getRelateKaOrderNumber())) {
                acceptanceVo.setRelateKaOrderNumber(entity.getRelateKaOrderNumber() + "-" + integer);
            }
            acceptanceVo.setFinalAcceptanceAmount(BigDecimal.ZERO);
            acceptanceVo.setFinalAcceptanceAmountNot(BigDecimal.ZERO);
            //添加子验收单中的产品编码
            goods.addAll(acceptanceVo.getGoodsList().stream().map(InvoiceAcceptanceGoodsVo::getGoodsCode).collect(Collectors.toSet()));
            BigDecimal quentitys =BigDecimal.ZERO;
            for (String good : goods) {
                //判断子验收单的详情中是否有该产品
                if (CollectionUtils.isEmpty(acceptanceVo.getGoodsList().stream().filter(f ->good.equals(f.getGoodsCode())).collect(Collectors.toSet()))){
                    return;
                }
                BigDecimal nums = acceptanceVo.getGoodsList().stream().filter(f ->good.equals(f.getGoodsCode())).map(InvoiceAcceptanceGoodsVo::getCurUnitAcceptanceQuantity).reduce(BigDecimal.ZERO, BigDecimal::add);
                quentitys =  quentitys.add(nums);
                //先判断 子验收单产品Map 中是否有该产品，有的话就做加法再放进Map中，没有则直接放进Map
                if (Objects.nonNull(productsMap.get(good))){
                    BigDecimal quentity = productsMap.get(good);
                    quentity = quentity.add(quentitys);
                    productsMap.put(good, quentity);
                }else {
                    productsMap.put(good, quentitys);
                }
                quentitys = BigDecimal.ZERO;
            }
            acceptanceVo.getGoodsList().forEach(aa ->{
                aa.setId(null);
                aa.setOrderNumber(acceptanceVo.getOrderNumber());
                aa.setKaOrderNumber(acceptanceVo.getKaOrderNumber());
                if (Objects.nonNull(productsUnitMapTotal.get(aa.getGoodsCode()))){
                    aa.setCurCompanyUnitOrderQuantity(aa.getCurUnitAcceptanceQuantity().multiply(productsUnitMapTotal.get(aa.getGoodsCode())).setScale(4,BigDecimal.ROUND_HALF_UP));
                }
                aa.setOrderQuantity(aa.getCurUnitAcceptanceQuantity());
                Validate.isTrue(!Objects.isNull(aa.getCurUnitAcceptanceQuantity()),"该验收单的数量不能为空");
                Validate.isTrue(!Objects.isNull(aa.getAcceptanceOneAmount()),"该验收单的单价（含税）不能为空");
                Validate.isTrue(!Objects.isNull(aa.getAcceptanceOneAmountNot()),"该验收单的单价（不含税）不能为空");
                //计算出金额
                aa.setAcceptanceAmount(aa.getAcceptanceOneAmount().multiply(aa.getCurUnitAcceptanceQuantity()));
                aa.setAcceptanceAmountNot(aa.getAcceptanceOneAmountNot().multiply(aa.getCurUnitAcceptanceQuantity()));
                //总金额计算
                acceptanceVo.setFinalAcceptanceAmount(acceptanceVo.getFinalAcceptanceAmount().add(aa.getAcceptanceAmount()));
                acceptanceVo.setFinalAcceptanceAmountNot(acceptanceVo.getFinalAcceptanceAmountNot().add(aa.getAcceptanceAmountNot()));
            });
            if ("".equals(acceptanceSplitOrders)){
                acceptanceSplitOrders= acceptanceSplitOrders+acceptanceVo.getOrderNumber();

            }else {
                acceptanceSplitOrders= acceptanceSplitOrders+","+acceptanceVo.getOrderNumber();
            }
            integer++;
            InvoiceAcceptanceEntity invoiceAcceptanceEntity = this.nebulaToolkitService.copyObjectByWhiteList(acceptanceVo,InvoiceAcceptanceEntity.class,HashSet.class,ArrayList.class);
            List<InvoiceAcceptanceGoodsEntity> invoiceAcceptanceGoodsEntities = (List<InvoiceAcceptanceGoodsEntity>) this.nebulaToolkitService.copyCollectionByWhiteList(acceptanceVo.getGoodsList(),InvoiceAcceptanceGoodsVo.class,InvoiceAcceptanceGoodsEntity.class,HashSet.class,ArrayList.class);
            invoiceAcceptanceGoodsEntityList.addAll(invoiceAcceptanceGoodsEntities);
            entities.add(invoiceAcceptanceEntity);
        }
        Validate.isTrue(productNums.size()==goods.size(),"原验收单中的产品数量和拆分验收单中的产品数量对不上");
        for (Map.Entry<String, BigDecimal> productMap : productsMapTotal.entrySet()) {
            Validate.isTrue(Objects.nonNull(productsMap.get(productMap.getKey())),"原验收单中的产品在拆分后的验收单中未出现");
            Validate.isTrue(productMap.getValue().compareTo(productsMap.get(productMap.getKey())) == 0,"原验收单中的产品在拆分后的验收单中数量对不上");
        }
        //改变原验收单的数据
        entity.setOrderStatus(AcceptanceStatus.S300.getDictCode());
        entity.setOrderStatusMsg(AcceptanceOrderConstant.ACCEPTANCE_SPILT_MSG);
        entity.setAcceptanceSplitOrders(acceptanceSplitOrders);
        entity.setIsSplit(BooleanEnum.TRUE.getCapital());
        this.invoiceAcceptanceRepository.saveOrUpdate(entity);
        this.invoiceAcceptanceRepository.saveBatch(entities);
        this.invoiceAcceptanceGoodsRepository.saveBatch(invoiceAcceptanceGoodsEntityList);
    }

    /**
     * 撤销拆分验收单
     * @param id
     */
    @Override
    public void withdrawSplit(String id){
        if (StringUtils.isEmpty(id)){
            return;
        }
        InvoiceAcceptanceEntity invoiceAcceptanceEntity =  this.invoiceAcceptanceService.findDetailById(id);
        if (Objects.isNull(invoiceAcceptanceEntity)){
            return;
        }
        InvoiceAcceptanceEntity entity = this.invoiceAcceptanceService.findDetailById(invoiceAcceptanceEntity.getSplitBeforeId());
        Validate.isTrue(!Objects.isNull(entity),"该验收单未找到原验收单");
        Validate.isTrue(StringUtils.isNotBlank(entity.getAcceptanceSplitOrders()),"该子验收单与原验收单未关联上");
        String[] orderNumbers = entity.getAcceptanceSplitOrders().split(",");
        //删除该结算单的子验收单
        this.invoiceAcceptanceRepository.deleteByOrderNumbers(TenantUtils.getTenantCode(),Lists.newArrayList(orderNumbers));
        this.invoiceAcceptanceGoodsRepository.deleteByOrderNumbers(TenantUtils.getTenantCode(),Lists.newArrayList(orderNumbers));
        //更新原验收单的信息
        entity.setOrderStatus(AcceptanceStatus.S200.getDictCode());
        entity.setOrderStatusMsg(AcceptanceOrderConstant.ACCEPTANCE_ORDER_CONVERT_SUCCESS_MSG);
        entity.setAcceptanceSplitOrders("");
        entity.setIsSplit("");
        this.invoiceAcceptanceRepository.saveOrUpdate(entity);
    }
}
