package com.biz.crm.kms.business.invoice.expense.sheet.local.repository;


import com.biz.crm.kms.business.invoice.expense.sheet.local.entity.InvoiceExpenseSheet;
import com.biz.crm.kms.business.invoice.expense.sheet.local.mapper.InvoiceExpenseSheetMapper;
import com.biz.crm.kms.business.invoice.expense.sheet.local.model.ExpenseSheetConditionModel;
import com.biz.crm.kms.business.invoice.expense.sheet.sdk.dto.InvoiceExpenseSheetDto;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import org.springframework.stereotype.Component;
import org.springframework.data.domain.Pageable;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * (InvoiceExpenseSheet)表数据库访问层
 *
 * @author makejava
 * @since 2022-11-11 18:06:12
 */
@Component
public class InvoiceExpenseSheetRepository extends ServiceImpl<InvoiceExpenseSheetMapper, InvoiceExpenseSheet> {

    /**
     * 根据id集合获取详情集合
     *
     * @param ids
     * @return
     */
    public List<InvoiceExpenseSheet> findByIds(List<String> ids) {
        return this.lambdaQuery()
                .in(InvoiceExpenseSheet::getId, ids)
                .list();
    }

    /**
     * 根据id集合更新启禁用装填
     *
     * @param ids          分页对象
     * @param enableStatus 启禁用状态对象
     * @return
     */
    public void updateEnableStatusByIds(List<String> ids, EnableStatusEnum enableStatus) {
        this.lambdaUpdate()
                .in(InvoiceExpenseSheet::getId, ids)
                .set(InvoiceExpenseSheet::getEnableStatus, enableStatus.getCode())
                .update();
    }

    /**
     * @Description: 根据条件查询费用单
     * @Param: [dto]
     * @return: java.util.List<com.biz.crm.kms.business.invoice.expense.sheet.local.entity.InvoiceExpenseSheet>
     * @Author: ZS
     * @Date: 2022/11/16
     */
    public List<InvoiceExpenseSheet> findExpenseSheet(InvoiceExpenseSheetDto dto) {
        return this.baseMapper.findExpenseSheet(dto);
    }


    /**
     * 根据费用单编码查询费用单
     *
     * @param tenantCode
     * @param orderNumberList
     * @return
     */
    public List<InvoiceExpenseSheet> findByOrderNumberList(String tenantCode, List<String> orderNumberList) {
        return this.lambdaQuery()
                .eq(InvoiceExpenseSheet::getTenantCode, tenantCode)
                .in(InvoiceExpenseSheet::getInvoiceExpenseSheetCode, orderNumberList)
                .list();
    }


    /**
     * 多条件查询抓单规则列表
     *
     * @param model 条件model
     * @return 抓单规则列表
     */
    public List<InvoiceExpenseSheet> findByReturnOrderConditionModel(ExpenseSheetConditionModel model) {
        return this.baseMapper.findByReturnOrderConditionModel(model);
    }

    /**
     * @Description: 分页查询费用单
     * @Param: [pageable]
     * @Param: [dto]
     * @return: com.biz.crm.business.common.sdk.model.Result<com.baomidou.mybatisplus.extension.plugins.pagination.Page < com.biz.crm.kms.business.invoice.expense.sheet.sdk.dto.InvoiceExpenseSheetDto>>
     * @Author: heding
     * @Date: 2022/11/30
     */
    public Page<InvoiceExpenseSheetDto> findExpenseSheetPaging(Pageable pageable, InvoiceExpenseSheetDto dto) {
        Page<InvoiceExpenseSheetDto> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        Page<InvoiceExpenseSheetDto> pageList = this.baseMapper.findExpenseSheetPaging(page, dto);
        return pageList;
    }

    /**
     * 通过费用单号查找
     * @param tenantCode
     * @param code
     * @return
     */
    public InvoiceExpenseSheet findByInvoiceExpenseSheetCode(String tenantCode,String code){
        return this.lambdaQuery()
                .eq(InvoiceExpenseSheet::getTenantCode,tenantCode)
                .eq(InvoiceExpenseSheet::getInvoiceExpenseSheetCode,code)
                .one();
    }

    /**
     * 取消捆绑结算单
     * @param statementCode
     */
    public void cancel(String statementCode){
        this.lambdaUpdate()
                .eq(InvoiceExpenseSheet::getStatementCode,statementCode)
                .set(InvoiceExpenseSheet::getStatementCode,null)
                .update();
    }
    /**
     * 捆绑结算单号
     * @param orderNumbers
     * @param statementCode
     */
    public void match(List<String> orderNumbers,String statementCode){
        this.lambdaUpdate()
                .in(InvoiceExpenseSheet::getInvoiceExpenseSheetCode,orderNumbers)
                .set(InvoiceExpenseSheet::getStatementCode,statementCode)
                .update();
    }

    /**
     * 同步费用单的金额

     * @return
     */
    public List<InvoiceExpenseSheet> syncExpense(List<String> directCodes){
        if (CollectionUtils.isEmpty(directCodes)){
            return null;
        }
        return this.lambdaQuery()
                .in(!CollectionUtils.isEmpty(directCodes),InvoiceExpenseSheet::getDirectCode,directCodes)
                .isNull(InvoiceExpenseSheet::getAmount)
                .list();
    }

    /**
     * 通过费用单号集合批量查找
     * @param codes
     * @return
     */
    public List<InvoiceExpenseSheet> findByCodes(List<String> codes){
        if (CollectionUtils.isEmpty(codes)){
            return new ArrayList<>();
        }
        return this.lambdaQuery()
                .in(!CollectionUtils.isEmpty(codes),InvoiceExpenseSheet::getInvoiceExpenseSheetCode,codes)
                .list();
    }
}

