package com.biz.crm.kms.business.invoice.expense.sheet.local.service.internal;

import cn.hutool.core.collection.CollectionUtil;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.kms.business.direct.product.sdk.vo.DirectProductUnitVo;
import com.biz.crm.kms.business.direct.product.sdk.vo.DirectProductVo;
import com.biz.crm.kms.business.direct.sdk.vo.DirectVo;
import com.biz.crm.kms.business.direct.store.sdk.vo.DirectStoreVo;
import com.biz.crm.kms.business.invoice.expense.sheet.local.entity.InvoiceExpenseSheet;
import com.biz.crm.kms.business.invoice.expense.sheet.local.repository.InvoiceExpenseSheetGrabRepository;
import com.biz.crm.kms.business.invoice.expense.sheet.local.repository.InvoiceExpenseSheetRepository;
import com.biz.crm.kms.business.invoice.expense.sheet.local.service.InvoiceExpenseSheetGrabsThreadService;
import com.biz.crm.kms.business.invoice.expense.sheet.sdk.constant.ExpenseSheetConstant;
import com.biz.crm.kms.business.invoice.expense.sheet.sdk.enums.ExpenseSheetStatus;
import com.biz.crm.kms.business.invoice.sdk.enums.GrabTransStatus;
import com.biz.crm.kms.business.invoice.sdk.enums.InvoicesStatus;
import com.biz.crm.mn.common.base.service.RedisCrmService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @Class InvoiceExpenseSheetGrabsThreadServiceImpl
 * @Description
 * @Author : ZS
 * @Date: 2022/11/25  11:29
 */
@Slf4j
@Service
public class InvoiceExpenseSheetGrabsThreadServiceImpl implements InvoiceExpenseSheetGrabsThreadService {


    @Autowired
    private InvoiceExpenseSheetRepository invoiceExpenseSheetRepository;

    @Autowired
    private InvoiceExpenseSheetGrabRepository invoiceExpenseSheetGrabRepository;

    @Autowired
    private RedisCrmService redisCrmService;

    @Autowired(required = false)
    private RedisTemplate redisTemplate;

    /**
     * @Description: 存储数据
     * @Param: [entities, directStoreMap, directProductMap]
     * @return: void
     * @Author: ZS
     * @Date: 2022年12月15日 16:42:47
     */
    @Override
    @Async("grabThread")
    @Transactional(propagation = Propagation.NOT_SUPPORTED, rollbackFor = Exception.class)
    public void saveExpenseSheetData(List<InvoiceExpenseSheet> entityList, Map<String, List<DirectVo>> directMap , Map<String, List<DirectStoreVo>> directStoreMap, Map<String, List<DirectProductVo>> directProductMap) {
        log.info("--  进入线程池存储数据  --");
        List<String> orderNumbers = entityList.stream().map(InvoiceExpenseSheet::getInvoiceExpenseSheetCode).collect(Collectors.toList());
        Set<String> orderNumberSet = new HashSet<>();
        Set<String> orderNumberSuccessSet = new HashSet<>();
        Map<String, BigDecimal> map = this.redisTemplate.hasKey(ExpenseSheetConstant.CACHE_PRODUCT_TAX_RATE) ? (Map<String, BigDecimal>) this.redisTemplate.opsForValue().get(ExpenseSheetConstant.CACHE_PRODUCT_TAX_RATE) : null;
        entityList.forEach(entity -> {
            try {
                //处理数据前先进行解锁
                redisCrmService.hdel(ExpenseSheetConstant.EXPENSE_SHEET_CONVERT_LOCK, entity.getInvoiceExpenseSheetCode());
                //清除数据中的售达方，企业门店，区域等数据 第一步
                this.clearTransformInfoOne(entity,directMap);
                if (StringUtils.isNotBlank(entity.getSupermarketStoreCode())){
                    //拼接门店数据  第二步
                    String deliveryCode = this.buildOrderInfoTwo(entity, directStoreMap);
                    if(StringUtils.isNotBlank(entity.getKaProductCode())) {
                        //拼接商品数据  第三步
                        this.convertAcceptanceOrderItemThree(entity, directProductMap, map, deliveryCode);
                    }
                }else {
                    entity.setOrderStatusMsg(null);
                    entity.setOrderStatus(InvoicesStatus.S200.getDictCode());
                }
                //设置状态  第四步
                this.setOrderStatusAndMsgFour(entity);
                this.invoiceExpenseSheetRepository.saveOrUpdate(entity);
                if (Objects.equals(InvoicesStatus.S200.getDictCode(), entity.getOrderStatus())){
                    orderNumberSuccessSet.add(entity.getInvoiceExpenseSheetCode());
                }else {
                    orderNumberSet.add(entity.getInvoiceExpenseSheetCode());
                }
            } catch (Exception e) {
                log.error("保存费用单数据失败", e);
                entity.setOrderStatus(InvoicesStatus.S100.getDictCode());
                entity.setOrderStatusMsg("转换时异常!");
                try {
                    this.invoiceExpenseSheetRepository.saveOrUpdate(entity);
                    orderNumberSet.add(entity.getInvoiceExpenseSheetCode());
                } catch (Exception ex) {
                    log.error("", ex);
                }
            }
        });

        if (CollectionUtil.isEmpty(orderNumberSet) && CollectionUtils.isEmpty(orderNumberSuccessSet)) {
            return;
        }
        //处理数据前先进行解锁
        redisCrmService.hdel(ExpenseSheetConstant.EXPENSE_SHEET_CONVERT_LOCK, orderNumbers);
        try {
            if (!CollectionUtils.isEmpty(orderNumberSet)){
                invoiceExpenseSheetGrabRepository.updateTransStatusByOrderNumber(TenantUtils.getTenantCode(),
                        GrabTransStatus.TRANS_FAIL, new ArrayList<>(orderNumberSet));
            }

            if (!CollectionUtils.isEmpty(orderNumberSuccessSet)){
                invoiceExpenseSheetGrabRepository.updateTransStatusByOrderNumber(TenantUtils.getTenantCode(),
                        GrabTransStatus.HAVE_TRANS, new ArrayList<>(orderNumberSuccessSet));
            }

        } catch (Exception e) {
            log.error("", e);
        }
    }

    /**
     * @Description:第一步：清除售达方、送达方、门店、区域的信息
     * @Param: [entity]
     * @return: void
     * @Author: ZS
     * @Date: 2022年12月15日 16:42:43
     */
    public void clearTransformInfoOne(InvoiceExpenseSheet entity,Map<String, List<DirectVo>> directMap) {
        entity.setDeliveryPartyCode(null);
        entity.setDeliveryPartyName(null);
        entity.setAreaName(null);
        entity.setAreaCode(null);
        //单据时间转换
        if (StringUtils.isNotBlank(entity.getOrderDate())){
            try {
                entity.setOrderTime(new SimpleDateFormat("yyyy-MM-dd").parse(entity.getOrderDate()));
            } catch (ParseException e) {
                e.printStackTrace();
            }
        }
        //业态业务单元赋值
        List<DirectVo> directVos = directMap.get(entity.getDirectCode());
        if (CollectionUtils.isEmpty(directVos)){
            entity.setOrderStatusMsg("不存在该系统");
        }else {
            DirectVo directVo = directVos.get(0);
            entity.setBusinessFormatCode(directVo.getBusinessFormatCode());
            entity.setBusinessUnitCode(directVo.getBusinessUnitCode());
        }
    }


    /**
     * @Description: 第二步：添加门店信息
     * @Param: [entity, directStoreMap]
     * @return: void
     * @Author: ZS
     * @Date: 2022年12月15日 16:42:40
     */
    public String buildOrderInfoTwo(InvoiceExpenseSheet entity, Map<String, List<DirectStoreVo>> directStoreMap) {
        entity.setOrderStatus(ExpenseSheetStatus.S100.getDictCode());
        entity.setSoldToPartyName(null);
        entity.setSoldToPartyCode(null);
        if (StringUtils.isBlank(entity.getDirectCode())) {
            entity.setOrderStatusMsg("系统编码为空");
            return null;
        }
        if (StringUtils.isBlank(entity.getSupermarketStoreCode())) {
            entity.setOrderStatusMsg("单据商超门店编码为空");
            return null;
        }
        List<DirectStoreVo> directStoreList = directStoreMap.get(entity.getDirectCode() + "_" + entity.getSupermarketStoreCode());
        DirectStoreVo directStoreVo = null;
        if (CollectionUtil.isEmpty(directStoreList)) {
            entity.setOrderStatusMsg("未查询到与企业门店对应关系");
            return null;
        } else if (directStoreList.size() > 1) {
            entity.setOrderStatusMsg("商超门店与企业门店对应关系存在[" + directStoreList.size() + "]条");
            return null;
        } else {
            directStoreVo = directStoreList.get(0);
        }
        entity.setSupermarketStoreCode(directStoreVo.getSupermarketStoreCode());
        if(StringUtils.isEmpty(entity.getSupermarketStoreName())){
            entity.setSupermarketStoreName(directStoreVo.getSupermarketStoreName());
        }
        //售达方信息
        entity.setSoldToPartyCode(directStoreVo.getSoldToPartyCode());
        entity.setSoldToPartyName(directStoreVo.getSoldToPartyName());
        entity.setDeliveryPartyCode(directStoreVo.getTerminalCode());
        entity.setDeliveryPartyName(directStoreVo.getTerminalName());
        entity.setAreaName(directStoreVo.getRetailerRegion());
        entity.setBusinessArea(directStoreVo.getBusinessArea());
        entity.setProvinceCode(directStoreVo.getProvinceCode());
        entity.setProvinceName(directStoreVo.getProvinceName());
        entity.setOrderStatus(ExpenseSheetStatus.S200.getDictCode());
        return directStoreVo.getRetailerTerminalCode();
    }

    /**
     * @Description: 第三步：添加商品信息
     * @Param: [entity, directProductMap]
     * @return: void
     * @Author: ZS
     * @Date: 2022年12月15日 16:42:36
     */
    private void convertAcceptanceOrderItemThree(InvoiceExpenseSheet entity, Map<String, List<DirectProductVo>> directProductMap, Map<String, BigDecimal> map,String deliveryCode) {
        entity.setOrderStatus(ExpenseSheetStatus.S100.getDictCode());
        if (StringUtils.isBlank(entity.getKaProductCode())) {
            entity.setOrderStatusMsg("商品编码为空");
            return;
        }
        DirectProductVo productVo = null;
        List<DirectProductVo> productVoList = new ArrayList<>();
        List<DirectProductVo> productVoLists = directProductMap.get(entity.getDirectCode() + "_" +
                entity.getDeliveryPartyCode() + "_" + entity.getKaProductCode());
        if (CollectionUtil.isNotEmpty(productVoLists)){
            productVoList.addAll(productVoLists);
        }
        //如果送达方编码没有找到，那么会通过关联的送达方编码去查
        if (CollectionUtil.isEmpty(productVoList) && StringUtils.isNotBlank(deliveryCode)){
            List<DirectProductVo> productVos = directProductMap.get(entity.getDirectCode() + "_" +
                    deliveryCode + "_" + entity.getKaProductCode());
            if (CollectionUtil.isNotEmpty(productVos)){
                productVoList.addAll(productVos);
            }
        }
        if (CollectionUtil.isEmpty(productVoList)) {
            entity.setOrderStatusMsg("未查询到商品对应关系");
        } else if (productVoList.size() == 1) {
            productVo = productVoList.get(0);
        } else {
            int allSize = productVoList.size();
            //匹配到多条数据  用是否默认标记为是的数据
            productVoList = productVoList.stream()
                    .filter(k -> BooleanEnum.TRUE.getCapital().equals(k.getIsDefault()))
                    .collect(Collectors.toList());
            if (CollectionUtil.isEmpty(productVoList)) {
                entity.setOrderStatusMsg("共[" + allSize + "]条商品对应关系,无默认对应关系");
            } else if (productVoList.size() > 1) {
                entity.setOrderStatusMsg("[" + productVoList.size() + "]条默认商品对应关系");
            } else {
                productVo = productVoList.get(0);
            }
        }
        if (Objects.isNull(productVo)) {
            return;
        }
        //产品信息
        entity.setProductCode(productVo.getProductCode());
        entity.setProductName(productVo.getProductName());
        //含税金额转换
        entity = this.setAccount(entity,map);
        //转换状态
        entity.setOrderStatus(ExpenseSheetStatus.S200.getDictCode());
    }

    /**
     * 设置含税未税
     */
    private InvoiceExpenseSheet setAccount(InvoiceExpenseSheet entity, Map<String, BigDecimal> map) {
        String goodsCode = entity.getProductCode();
        if (!StringUtils.isEmpty(goodsCode)) {
            if (Objects.isNull(map)) {
                return entity;
            }
            BigDecimal bigDecimal = map.get(goodsCode);
            if (Objects.isNull(bigDecimal)){
                return entity;
            }
            if (Objects.nonNull(entity.getAmount()) && Objects.isNull(entity.getAmountNot())){
                entity.setAmountNot(entity.getAmount().divide(BigDecimal.ONE.add(bigDecimal),4,BigDecimal.ROUND_HALF_UP));
            }
            if (Objects.nonNull(entity.getAmountNot()) && Objects.isNull(entity.getAmount())){
                entity.setAmount(entity.getAmountNot().multiply(BigDecimal.ONE.add(bigDecimal)).setScale(4,BigDecimal.ROUND_HALF_UP));
            }
        }
        return entity;
    }

    /**
     * @Description: 第四布：转换状态
     * @Param: [entity]
     * @return: void
     * @Author: ZS
     * @Date: 2022年12月15日 16:42:33
     */
    public void setOrderStatusAndMsgFour(InvoiceExpenseSheet entity) {
        if (Objects.equals(entity.getOrderStatus(), ExpenseSheetStatus.S200.getDictCode())) {
            entity.setOrderStatusMsg(ExpenseSheetConstant.EXPENSE_SHEET_CONVERT_SUCCESS_MSG);
        }
    }
}
