package com.biz.crm.kms.business.invoice.expense.sheet.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.kms.business.invoice.expense.sheet.local.entity.InvoiceExpenseSheet;
import com.biz.crm.kms.business.invoice.expense.sheet.local.repository.InvoiceExpenseSheetGrabRepository;
import com.biz.crm.kms.business.invoice.expense.sheet.local.repository.InvoiceExpenseSheetRepository;
import com.biz.crm.kms.business.invoice.expense.sheet.local.service.InvoiceExpenseSheetGrabService;
import com.biz.crm.kms.business.invoice.expense.sheet.sdk.dto.InvoiceExpenseSheetDto;
import com.biz.crm.kms.business.invoice.expense.sheet.sdk.service.InvoiceExpenseSheetVoService;
import com.biz.crm.kms.business.invoice.sdk.enums.InvoicesStatus;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @Class InvoiceExpenseSheetVoServiceImpl
 * @Description
 * @Author : ZS
 * @Date: 2022/11/16  14:29
 */
@Service
@Slf4j
public class InvoiceExpenseSheetVoServiceImpl implements InvoiceExpenseSheetVoService {

    @Autowired
    private InvoiceExpenseSheetRepository invoiceExpenseSheetRepository;
    @Autowired
    private NebulaToolkitService nebulaToolkitService;

    @Autowired
    private InvoiceExpenseSheetGrabService invoiceExpenseSheetGrabService;

    /**
    * @Description: 根据条件查询费用单数据
    * @Param: [invoiceExpenseSheetDto]
    * @return: java.util.List<com.biz.crm.kms.business.invoice.expense.sheet.sdk.dto.InvoiceExpenseSheetDto>
    * @Author: ZS
    * @Date: 2022/11/16
    */
    @Override
    public List<InvoiceExpenseSheetDto> findExpenseSheet(InvoiceExpenseSheetDto invoiceExpenseSheetDto) {
        List<InvoiceExpenseSheet> expenseSheets = this.invoiceExpenseSheetRepository.findExpenseSheet(invoiceExpenseSheetDto);
        List<InvoiceExpenseSheetDto> expenseSheetDtos =(List<InvoiceExpenseSheetDto>) this.nebulaToolkitService.copyCollectionByWhiteList(expenseSheets,InvoiceExpenseSheet.class,InvoiceExpenseSheetDto.class, HashSet.class, ArrayList.class);
        return expenseSheetDtos;
    }
    /**
     * @Description: 分页查询费用单
     * @Param: [pageable]
     * @Param: [dto]
     * @return: com.baomidou.mybatisplus.extension.plugins.pagination.Page<com.biz.crm.kms.business.invoice.expense.sheet.sdk.dto.InvoiceExpenseSheetDto>
     * @Author: heding
     * @Date: 2022/11/30
     */
    @Override
    public Page<InvoiceExpenseSheetDto> findExpenseSheetPaging(Pageable pageable, InvoiceExpenseSheetDto dto) {
        if (Objects.isNull(dto)) {
            dto = new InvoiceExpenseSheetDto();
        }
        return invoiceExpenseSheetRepository.findExpenseSheetPaging(pageable, dto);
    }
    /**
    * @Description: 手动转换接口
    * @Param: [ids]
    * @return: void
    * @Author: ZS
    * @Date: 2022/12/15
    */
    @Override
    public void manualSwitch(List<String> ids) {
        Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空！");
        List<InvoiceExpenseSheet> entities = this.invoiceExpenseSheetRepository.findByIds(ids);
        Validate.isTrue(!CollectionUtils.isEmpty(entities), "未查询到数据，请刷新重试");
        Validate.isTrue(!ObjectUtils.notEqual(entities.size(),ids.size()), "数据转换个数不匹配");
        entities.forEach(entity -> Validate.isTrue(Lists.newArrayList(InvoicesStatus.S200.getDictCode(), InvoicesStatus.S101.getDictCode(), InvoicesStatus.S100.getDictCode()).contains(entity.getOrderStatus())
                , String.format("验收单[%s]已确认,无法继续匹配", entity.getInvoiceExpenseSheetCode())));
        List<String> orderNumbers = entities.stream()
                .filter(k -> StringUtils.isNotBlank(k.getInvoiceExpenseSheetCode()))
                .map(InvoiceExpenseSheet::getInvoiceExpenseSheetCode).distinct().collect(Collectors.toList());
        //过滤正在转换的费用单号
        this.invoiceExpenseSheetGrabService.filterSwitchIngOrderNumberList(orderNumbers);
        log.info("===== 费用单手动转换开始 ======");
        this.invoiceExpenseSheetGrabService.manualSwitch(orderNumbers);
        log.info("===== 费用单手动转换完成 ======");
    }
    /**
     * 手动关联结算单
     * @param orderNumbers
     * @param statementCode
     */
    @Override
    public void match(List<String> orderNumbers, String statementCode) {
        Validate.isTrue(!CollectionUtils.isEmpty(orderNumbers)&&StringUtils.isNotBlank(statementCode),"验收单号和结算单号不能为空");
        //取消结算单关联
        this.invoiceExpenseSheetRepository.cancel(statementCode);
        //增加结算单关联
        this.invoiceExpenseSheetRepository.match(orderNumbers,statementCode);
    }

    /**
     * 查询所有未同步含税金额的高鑫费用单
     * @param directCodes
     * @return
     */
    @Override
    public List<String> syncExpense(List<String> directCodes) {
        List<InvoiceExpenseSheet> invoiceExpenseSheets = this.invoiceExpenseSheetRepository.syncExpense(directCodes);
        if (CollectionUtils.isEmpty(invoiceExpenseSheets)){
            return new ArrayList<>();
        }
        List<String> expenseCodes = invoiceExpenseSheets.stream().map(InvoiceExpenseSheet::getInvoiceExpenseSheetCode).collect(Collectors.toList());
        return expenseCodes;
    }

    /**
     * 更新金额
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateExpense(Map<String , BigDecimal> amounts, List<String> decomentCodes){
        if (CollectionUtils.isEmpty(decomentCodes) && Objects.isNull(amounts)){
            return;
        }
        List<InvoiceExpenseSheet> expenseSheets = this.invoiceExpenseSheetRepository.findByCodes(decomentCodes);
        if (CollectionUtils.isEmpty(expenseSheets)){
            return;
        }
        //将含税金额同步
        expenseSheets.forEach(aa ->{
            BigDecimal amount = amounts.get(aa.getInvoiceExpenseSheetCode());
            aa.setAmount(amount);
        });
        this.invoiceExpenseSheetRepository.saveOrUpdateBatch(expenseSheets);
    }

    /**
     * 费用单同步兑付方式
     * @param documentCodes
     * @return
     */
    @Override
    public Map<String, String> findCashing(Set<String> documentCodes) {
        if (CollectionUtils.isEmpty(documentCodes)){
            return null;
        }
        List<InvoiceExpenseSheet> invoiceExpenseSheets = this.invoiceExpenseSheetRepository.findByCodes(Lists.newArrayList(documentCodes));
        if (CollectionUtils.isEmpty(invoiceExpenseSheets)) {
            return null;
        }
        Map<String ,String> chshingMap = invoiceExpenseSheets.stream().filter(f ->StringUtils.isNotBlank(f.getCashingType())).collect(Collectors.toMap(InvoiceExpenseSheet::getInvoiceExpenseSheetCode,InvoiceExpenseSheet::getCashingType));
        return chshingMap;
    }
}
