package com.biz.crm.kms.business.invoice.order.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.kms.business.invoice.order.local.entity.OrderEntity;
import com.biz.crm.kms.business.invoice.order.local.service.OrderService;
import com.biz.crm.kms.business.invoice.order.sdk.dto.OrderPageDto;
import com.google.common.collect.Sets;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

/**
 * 订货单: order: 订货单管理
 *
 * @author ning.zhang
 * @date 2022/10/25
 */
@Slf4j
@RestController
@RequestMapping("/v1/order/order")
@Api(tags = "订货单: order: 订货单管理")
public class OrderController {

  @Autowired
  private OrderService orderService;

  /**
   * 手动转换订货单
   *
   * @param ids 订货单id集合
   * @return 转换结果
   */
  @ApiOperation(value = "手动转换订货单")
  @PatchMapping("/manualTrans")
  public Result manualTrans(@RequestBody List<String> ids) {
    try {
      this.orderService.manualTrans(ids);
      return Result.ok("转换成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 确认订货单
   *
   * @param ids 订货单id集合
   * @return 转换结果
   */
  @ApiOperation(value = "确认订货单")
  @PatchMapping("/confirm")
  public Result confirm(@RequestBody List<String> ids) {
    try {
      this.orderService.confirm(ids);
      return Result.ok("确认成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 通过ID获取订货单详细信息
   *
   * @param id 订货单ID
   * @return 订货单信息
   */
  @ApiOperation(value = "通过ID获取订货单详细信息")
  @GetMapping("/findById")
  public Result<OrderEntity> findById(@RequestParam(value = "id", required = false) @ApiParam(name = "id", value = "主键ID") String id) {
    try {
      List<OrderEntity> list = this.orderService.findByIds(Sets.newHashSet(id));
      return Result.ok(CollectionUtils.isEmpty(list) ? null : list.get(0));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 订货单分页查询
   *
   * @param pageable 分页信息
   * @param dto      参数Dto
   * @return 订货单分页数据
   */
  @ApiOperation(value = "订货单分页查询")
  @GetMapping("/findByConditions")
  public Result<Page<OrderEntity>> findByConditions(@PageableDefault(50) Pageable pageable,
                                                   @ApiParam(name = "OrderPageDto", value = "分页Dto") OrderPageDto dto) {
    try {
      return Result.ok(this.orderService.findByConditions(pageable, dto));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
}
