package com.biz.crm.kms.business.invoice.order.local.repository;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.kms.business.invoice.order.local.entity.OrderEntity;
import com.biz.crm.kms.business.invoice.order.local.mapper.OrderMapper;
import com.biz.crm.kms.business.invoice.order.local.model.OrderConditionModel;
import com.biz.crm.kms.business.invoice.order.sdk.dto.OrderPageDto;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * 订货单表的数据库访问类 {@link OrderEntity}
 *
 * @author ning.zhang
 * @date 2022-10-12 16:37:35
 */
@Component
public class OrderRepository extends ServiceImpl<OrderMapper, OrderEntity> {

  /**
   * 根据订货单编码删除数据
   *
   * @param tenantCode 租户编码
   * @param orderNumber 订货单编码
   */
  public void deleteByOrderNumber(String tenantCode, String orderNumber) {
    this.lambdaUpdate()
        .eq(OrderEntity::getTenantCode, tenantCode)
        .eq(OrderEntity::getOrderNumber, orderNumber)
        .remove();
  }

  /**
   * 根据订货单编码获取订货单信息
   *
   * @param tenantCode 租户编码
   * @param orderNumber 订货单编码
   * @return 订货单信息
   */
  public OrderEntity findByOrderNumber(String tenantCode, String orderNumber) {
    return this.lambdaQuery()
        .eq(OrderEntity::getTenantCode, tenantCode)
        .eq(OrderEntity::getOrderNumber, orderNumber)
        .one();
  }

  /**
   * 多条件查询订货单规则列表
   *
   * @param model 条件model
   * @return 订货单规则列表
   */
  public List<OrderEntity> findByOrderConditionModel(OrderConditionModel model) {
    return this.baseMapper.findByOrderConditionModel(model);
  }

  /**
   * 订货单分页列表
   *
   * @param dto      请求参数dto
   * @param pageable 分页信息
   * @return Page<OrderEntity> 订货单分页信息
   */
  public Page<OrderEntity> findByConditions(Pageable pageable, OrderPageDto dto) {
    Page<OrderEntity> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.baseMapper.findByConditions(page, dto);
  }
}
