package com.biz.crm.kms.business.invoice.order.local.service.internal;

import com.biz.crm.kms.business.direct.product.sdk.vo.DirectProductUnitVo;
import com.biz.crm.kms.business.direct.product.sdk.vo.DirectProductVo;
import com.biz.crm.kms.business.invoice.order.local.entity.OrderEntity;
import com.biz.crm.kms.business.invoice.order.local.entity.OrderGoodsEntity;
import com.biz.crm.kms.business.invoice.order.local.model.OrderGrabSyncModel;
import com.biz.crm.kms.business.invoice.order.local.repository.OrderGoodsRepository;
import com.biz.crm.kms.business.invoice.order.local.service.OrderGoodsService;
import com.biz.crm.kms.business.invoice.order.sdk.enums.OrderStatus;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 订货单商品表服务实现类
 *
 * @author ning.zhang
 * @date 2022-10-12 16:37:45
 */
@Slf4j
@Service("orderGoodsService")
public class OrderGoodsServiceImpl implements OrderGoodsService {

  @Autowired
  private OrderGoodsRepository orderGoodsRepository;

  @Override
  @Transactional
  public void transform(OrderEntity entity, OrderGrabSyncModel syncModel) {
    if (CollectionUtils.isEmpty(entity.getGoodsList())) {
      entity.setOrderStatusMsg("商品行信息存在转换失败数据");
      return;
    }
    this.orderGoodsRepository.deleteByOrderNumber(entity.getTenantCode(), entity.getOrderNumber());
    for (OrderGoodsEntity orderGoodsEntity : entity.getGoodsList()) {
      this.clearTransformInfo(orderGoodsEntity);
      this.buildGoodsInfo(orderGoodsEntity, entity, syncModel);
    }
    this.orderGoodsRepository.saveBatch(entity.getGoodsList());
  }

  @Override
  public void buildGoodsInfo(OrderGoodsEntity orderGoodsEntity, OrderEntity entity, OrderGrabSyncModel syncModel) {
    orderGoodsEntity.setOrderStatus(OrderStatus.S100.getDictCode());
    List<DirectProductVo> productList = syncModel.getDirectProductMap().get(String.format("%s-%s", orderGoodsEntity.getDirectCode(), orderGoodsEntity.getKaGoodsCode()));
    orderGoodsEntity.setId(null);
    if (CollectionUtils.isEmpty(productList)) {
      orderGoodsEntity.setOrderStatusMsg("未匹配到与企业商品对应关系");
      return;
    }
    productList = productList.stream()
        .filter(dpv -> dpv.getSellPartyCode().equals(entity.getSoldToPartyCode()))
        .collect(Collectors.toList());
    if (CollectionUtils.isEmpty(productList)) {
      orderGoodsEntity.setOrderStatusMsg(String.format("售达方[%s]未查询到与企业商品对应关系", entity.getSoldToPartyCode()));
      return;
    } else if (productList.size() > 1) {
      orderGoodsEntity.setOrderStatusMsg("企业商品存在多个对应关系");
      return;
    }
    DirectProductVo directProductVo = productList.get(0);
    orderGoodsEntity.setGoodsCode(directProductVo.getProductCode());
    orderGoodsEntity.setGoodsName(directProductVo.getProductName());
    if (Objects.isNull(orderGoodsEntity.getCurUnitOrderQuantity())) {
      orderGoodsEntity.setOrderStatusMsg("商超商品数量错误");
      return;
    }
    if (StringUtils.isBlank(orderGoodsEntity.getCurUnit())) {
      orderGoodsEntity.setOrderStatus(OrderStatus.S200.getDictCode());
      orderGoodsEntity.setCurCompanyUnitOrderQuantity(orderGoodsEntity.getCurUnitOrderQuantity());
      return;
    }
    List<DirectProductUnitVo> unitList = directProductVo.getUnitList();
    if (CollectionUtils.isEmpty(unitList)) {
      orderGoodsEntity.setOrderStatusMsg("未匹配企业商品单位转换关系");
      return;
    }
    //过滤无效单位转换关系
    unitList = unitList.stream().filter(vo -> Objects.equals(vo.getKaUnitCode(), orderGoodsEntity.getCurUnit())
        && Objects.nonNull(vo.getKaUnitQuantity())
        && Objects.nonNull(vo.getUnitQuantity())).collect(Collectors.toList());
    if (CollectionUtils.isEmpty(unitList)) {
      orderGoodsEntity.setOrderStatusMsg("未匹配企业商品单位转换关系");
      return;
    } else if (unitList.size() > 1) {
      orderGoodsEntity.setOrderStatusMsg("企业商品存在多个对应关系");
      return;
    }
    DirectProductUnitVo unitVo = unitList.get(0);
    BigDecimal curCompanyUnitOrderQuantity = unitVo.getUnitQuantity()
        .multiply(orderGoodsEntity.getCurUnitOrderQuantity())
        .divide(unitVo.getKaUnitQuantity(), 6, BigDecimal.ROUND_HALF_UP);
    orderGoodsEntity.setCurCompanyUnitOrderQuantity(curCompanyUnitOrderQuantity);
    orderGoodsEntity.setCurCompanyUnit(unitVo.getUnitCode());
    orderGoodsEntity.setOrderStatus(OrderStatus.S200.getDictCode());
  }

  @Override
  public void clearTransformInfo(OrderGoodsEntity entity) {
    entity.setGoodsCode(null);
    entity.setGoodsName(null);
    entity.setCurCompanyUnitOrderQuantity(null);
    entity.setCurCompanyUnit(null);
    entity.setSoldToPartyName(null);
    entity.setSoldToPartyCode(null);
  }
}
