package com.biz.crm.kms.business.invoice.order.local.service.internal;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.kms.business.direct.product.sdk.dto.DirectProductDto;
import com.biz.crm.kms.business.direct.product.sdk.service.DirectProductVoService;
import com.biz.crm.kms.business.direct.product.sdk.vo.DirectProductVo;
import com.biz.crm.kms.business.direct.store.sdk.dto.DirectStoreConditionDto;
import com.biz.crm.kms.business.direct.store.sdk.service.DirectStoreVoService;
import com.biz.crm.kms.business.direct.store.sdk.vo.DirectStoreVo;
import com.biz.crm.kms.business.invoice.order.local.entity.OrderEntity;
import com.biz.crm.kms.business.invoice.order.local.entity.OrderGoodsGrabEntity;
import com.biz.crm.kms.business.invoice.order.local.entity.OrderGrabEntity;
import com.biz.crm.kms.business.invoice.order.local.model.OrderGrabConditionModel;
import com.biz.crm.kms.business.invoice.order.local.model.OrderGrabSyncModel;
import com.biz.crm.kms.business.invoice.order.local.repository.OrderGrabRepository;
import com.biz.crm.kms.business.invoice.order.local.service.OrderGrabService;
import com.biz.crm.kms.business.invoice.order.local.service.OrderService;
import com.biz.crm.kms.business.invoice.order.sdk.enums.OrderStatus;
import com.biz.crm.kms.business.invoice.sdk.enums.GrabTransStatus;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalSupplyVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalSupplyVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 订货单抓取数据(python)表服务实现类
 *
 * @author ning.zhang
 * @date 2022-10-12 16:37:55
 */
@Slf4j
@Service("orderGrabService")
public class OrderGrabServiceImpl implements OrderGrabService {

  @Autowired
  private OrderGrabRepository orderGrabRepository;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private OrderService orderService;
  @Autowired
  private DirectProductVoService directProductVoService;
  @Autowired
  private DirectStoreVoService directStoreVoService;
  @Autowired
  private TerminalSupplyVoService terminalSupplyVoService;

  @Override
  @DynamicTaskService(cornExpression = "0 0/10 * * * ?", taskDesc = "订货单自动解析定时任务")
  public void executeTask() {
    OrderGrabConditionModel model = new OrderGrabConditionModel();
    model.setTenantCode(TenantUtils.getTenantCode());
    model.setTransStatus(GrabTransStatus.NOT_TRANS.getDictCode());
    List<OrderGrabEntity> entities = this.orderGrabRepository.findByOrderGrabConditionModel(model);
    if (CollectionUtils.isEmpty(entities)) {
      return;
    }
    OrderGrabSyncModel syncModel = this.buildOrderGrabSyncModel(entities);
    Map<String, OrderEntity> orderMap = entities.stream().collect(Collectors.toMap(OrderGrabEntity::getOrderNumber
        , orderGrab -> {
          OrderEntity orderEntity = this.nebulaToolkitService.copyObjectByBlankList(orderGrab, OrderEntity.class, HashSet.class, ArrayList.class);
          orderEntity.setId(null);
          orderEntity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
          orderEntity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
          return orderEntity;
        }, (a, b) -> a.getVersionNumber().compareTo(b.getVersionNumber()) > 0 ? a : b));
    orderMap.values().forEach(entity -> {
      try {
        OrderEntity orderEntity = this.orderService.transform(entity, syncModel);
        this.orderGrabRepository.updateTransStatusByOrderNumber(entity.getTenantCode()
            , OrderStatus.S100.getDictCode().equals(orderEntity.getOrderStatus()) ? GrabTransStatus.TRANS_FAIL : GrabTransStatus.HAVE_TRANS, entity.getOrderNumber());
      } catch (Exception e) {
        this.orderGrabRepository.updateTransStatusByOrderNumber(entity.getTenantCode(), GrabTransStatus.TRANS_FAIL, entity.getOrderNumber());
      }
    });
  }

  /**
   * 构建订货单抓单同步数据
   *
   * @param entities 订货单抓单数据
   * @return 订货单抓单同步数据
   */
  @Override
  public OrderGrabSyncModel buildOrderGrabSyncModel(List<OrderGrabEntity> entities) {
    OrderGrabSyncModel syncModel = new OrderGrabSyncModel();
    syncModel.setDirectProductMap(Maps.newHashMap());
    syncModel.setDirectStoreMap(Maps.newHashMap());
    syncModel.setTerminalSupplyMap(Maps.newHashMap());
    Set<String> kaStoreCodes = entities.stream()
        .map(OrderGrabEntity::getKaStoreCode)
        .filter(StringUtils::isNotBlank)
        .collect(Collectors.toSet());
    Set<String> kaGoodsCodes = entities.stream()
        .filter(orderGrabEntity -> !CollectionUtils.isEmpty(orderGrabEntity.getGoodsList()))
        .flatMap(orderGrabEntity -> orderGrabEntity.getGoodsList().stream().map(OrderGoodsGrabEntity::getKaGoodsCode))
        .filter(StringUtils::isNotBlank)
        .collect(Collectors.toSet());
    Set<String> terminalCodes = Sets.newHashSet();
    if (!CollectionUtils.isEmpty(kaStoreCodes)) {
      DirectStoreConditionDto conditionDto = new DirectStoreConditionDto();
      conditionDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
      conditionDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
      conditionDto.setSupermarketStoreCodes(kaStoreCodes);
      List<DirectStoreVo> directStoreList = this.directStoreVoService.findByDirectStoreConditionDto(conditionDto);
      if (!CollectionUtils.isEmpty(directStoreList)) {
        syncModel.setDirectStoreMap(directStoreList.stream()
            .collect(Collectors.groupingBy(o -> String.format("%s-%s", o.getDirectCode(), o.getSupermarketStoreCode()))));
      }
    }
    if (!CollectionUtils.isEmpty(syncModel.getDirectStoreMap().values())) {
      terminalCodes = syncModel.getDirectStoreMap().values().stream()
          .flatMap(directStoreVos -> directStoreVos.stream().map(DirectStoreVo::getTerminalCode))
          .collect(Collectors.toSet());
    }
    if (!CollectionUtils.isEmpty(terminalCodes)) {
      List<TerminalSupplyVo> terminalSupplyList = this.terminalSupplyVoService.findByTerminalCodes(Lists.newArrayList(terminalCodes));
      if (!CollectionUtils.isEmpty(terminalSupplyList)) {
        syncModel.setTerminalSupplyMap(terminalSupplyList.stream().collect(Collectors.groupingBy(TerminalSupplyVo::getTerminalCode)));
      }
    }
    if (!CollectionUtils.isEmpty(kaGoodsCodes)) {
      DirectProductDto directProductDto = new DirectProductDto();
      directProductDto.setTenantCode(TenantUtils.getTenantCode());
      directProductDto.setKaProductCodes(Lists.newArrayList(kaGoodsCodes));
      List<DirectProductVo> productList = this.directProductVoService.findByDirectProductDto(directProductDto);
      if (!CollectionUtils.isEmpty(productList)) {
        syncModel.setDirectProductMap(productList.stream()
            .collect(Collectors.groupingBy(o -> String.format("%s-%s", o.getDirectCode(), o.getKaProductCode()))));
      }
    }
    return syncModel;
  }
}
