package com.biz.crm.kms.business.invoice.order.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.kms.business.direct.store.sdk.vo.DirectStoreVo;
import com.biz.crm.kms.business.invoice.order.local.entity.OrderEntity;
import com.biz.crm.kms.business.invoice.order.local.entity.OrderGoodsEntity;
import com.biz.crm.kms.business.invoice.order.local.entity.OrderGrabEntity;
import com.biz.crm.kms.business.invoice.order.local.model.OrderConditionModel;
import com.biz.crm.kms.business.invoice.order.local.model.OrderGrabSyncModel;
import com.biz.crm.kms.business.invoice.order.local.repository.OrderRepository;
import com.biz.crm.kms.business.invoice.order.local.service.OrderGoodsService;
import com.biz.crm.kms.business.invoice.order.local.service.OrderGrabService;
import com.biz.crm.kms.business.invoice.order.local.service.OrderService;
import com.biz.crm.kms.business.invoice.order.sdk.dto.OrderPageDto;
import com.biz.crm.kms.business.invoice.order.sdk.enums.OrderStatus;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalSupplyVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 订货单表服务实现类
 *
 * @author ning.zhang
 * @date 2022-10-12 16:37:38
 */
@Slf4j
@Service("orderService")
public class OrderServiceImpl implements OrderService {

  @Autowired
  private OrderRepository orderRepository;
  @Autowired
  private OrderGoodsService orderGoodsService;
  @Autowired
  private OrderGrabService orderGrabService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Override
  @Transactional
  public OrderEntity transform(OrderEntity entity, OrderGrabSyncModel syncModel) {
    OrderEntity orderEntity = this.orderRepository.findByOrderNumber(entity.getTenantCode(), entity.getOrderNumber());
    //非可转换状态不用转换
    if (Objects.nonNull(orderEntity) && BooleanEnum.FALSE.getCapital().equals(orderEntity.getTransFlag())) {
      return orderEntity;
    }
    this.clearTransformInfo(entity);
    this.buildOrderInfo(entity, syncModel);
    this.orderGoodsService.transform(entity, syncModel);
    this.buildOrderStatus(entity);
    //订单赋值
    if (Objects.nonNull(orderEntity)) {
      BeanUtils.copyProperties(entity,orderEntity);
    } else {
      orderEntity = entity;
    }
    this.orderRepository.saveOrUpdate(orderEntity);
    return entity;
  }

  @Override
  public void clearTransformInfo(OrderEntity entity) {
    entity.setSoldToPartyName(null);
    entity.setSoldToPartyCode(null);
    entity.setStoreCode(null);
    entity.setStoreName(null);
    entity.setAreaCode(null);
    entity.setAreaName(null);
  }

  @Override
  public void buildOrderStatus(OrderEntity entity) {
    List<String> goodsErrorList = entity.getGoodsList().stream()
        .filter(orderGoodsEntity -> Objects.equals(orderGoodsEntity.getOrderStatus(), OrderStatus.S100.getDictCode()))
        .map(OrderGoodsEntity::getOrderStatusMsg)
        .collect(Collectors.toList());
    if (Objects.equals(entity.getOrderStatus(), OrderStatus.S200.getDictCode())) {
      if (CollectionUtils.isEmpty(goodsErrorList)) {
        entity.setOrderStatusMsg("转换成功");
      } else {
        entity.setOrderStatus(OrderStatus.S100.getDictCode());
        entity.setOrderStatusMsg("商品行信息存在转换失败数据");
      }
    } else if (!CollectionUtils.isEmpty(goodsErrorList)) {
      entity.setOrderStatusMsg(String.format("%s|%s", entity.getOrderStatusMsg(), "商品行信息存在转换失败数据"));
    }
  }

  @Override
  @Transactional
  public void manualTrans(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    OrderConditionModel model = new OrderConditionModel();
    model.setIds(Sets.newHashSet(ids));
    model.setTenantCode(TenantUtils.getTenantCode());
    List<OrderEntity> entities = this.orderRepository.findByOrderConditionModel(model);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据转换个数不匹配");
    entities.forEach(entity -> Validate.isTrue(Lists.newArrayList(OrderStatus.S200.getDictCode(), OrderStatus.S101.getDictCode(), OrderStatus.S100.getDictCode()).contains(entity.getOrderStatus())
        , String.format("订货单[%s]已确认,无法继续匹配", entity.getOrderNumber())));
    List<OrderGrabEntity> orderGrabEntities = (List<OrderGrabEntity>) this.nebulaToolkitService.copyCollectionByBlankList(entities, OrderEntity.class
        , OrderGrabEntity.class, HashSet.class, ArrayList.class);
    OrderGrabSyncModel syncModel = this.orderGrabService.buildOrderGrabSyncModel(orderGrabEntities);
    entities.forEach(entity -> {
      entity.setOrderStatusMsg(null);
      this.transform(entity, syncModel);
    });
  }

  @Override
  @Transactional
  public void create(List<OrderEntity> orderEntities) {
    if (CollectionUtils.isEmpty(orderEntities)) {
      return;
    }
    //分批次保存
    List<List<OrderEntity>> splitEntities = Lists.partition(orderEntities, 2000);
    splitEntities.forEach(entities -> {
      this.orderRepository.saveBatch(entities);
    });
  }

  @Override
  public List<OrderEntity> findByIds(Set<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return Lists.newLinkedList();
    }
    OrderConditionModel model = new OrderConditionModel();
    model.setIds(ids);
    model.setTenantCode(TenantUtils.getTenantCode());
    return this.orderRepository.findByOrderConditionModel(model);
  }

  @Override
  @Transactional
  public void confirm(List<String> ids) {
    Validate.isTrue(!CollectionUtils.isEmpty(ids), "缺失id");
    OrderConditionModel model = new OrderConditionModel();
    model.setIds(Sets.newHashSet(ids));
    model.setTenantCode(TenantUtils.getTenantCode());
    List<OrderEntity> entities = this.orderRepository.findByOrderConditionModel(model);
    Validate.isTrue(!CollectionUtils.isEmpty(entities) && entities.size() == ids.size(), "数据确认个数不匹配");
    entities.forEach(entity -> {
      Validate.isTrue(OrderStatus.S200.getDictCode().equals(entity.getOrderStatus())
          , String.format("订货单[%s]非待确认状态,无法确认", entity.getOrderNumber()));
      entity.setOrderStatus(OrderStatus.S300.getDictCode());
      entity.setOrderStatusMsg(OrderStatus.S300.getValue());
      entity.setTransFlag(BooleanEnum.FALSE.getCapital());
    });
    this.orderRepository.updateBatchById(entities);
  }

  @Override
  public Page<OrderEntity> findByConditions(Pageable pageable, OrderPageDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
    dto = ObjectUtils.defaultIfNull(dto, new OrderPageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    return this.orderRepository.findByConditions(pageable, dto);
  }

  @Override
  public void buildOrderInfo(OrderEntity entity, OrderGrabSyncModel syncModel) {
    entity.setOrderStatus(OrderStatus.S100.getDictCode());
    entity.setTransFlag(BooleanEnum.TRUE.getCapital());
    if (StringUtils.isBlank(entity.getKaStoreCode())) {
      entity.setOrderStatusMsg("未查询到与企业门店对应关系");
      return;
    }
    List<DirectStoreVo> directStoreList = syncModel.getDirectStoreMap().get(String.format("%s-%s", entity.getDirectCode(), entity.getKaStoreCode()));
    if (CollectionUtils.isEmpty(directStoreList)) {
      entity.setOrderStatusMsg("未查询到与企业门店对应关系");
      return;
    } else if (directStoreList.size() > 1) {
      entity.setOrderStatusMsg("商超门店存在多个对应关系");
      return;
    }
    DirectStoreVo directStoreVo = directStoreList.get(0);
    entity.setKaCode(directStoreVo.getSupermarketCode());
    entity.setKaName(directStoreVo.getSupermarketName());
    entity.setStoreCode(directStoreVo.getTerminalCode());
    entity.setStoreName(directStoreVo.getTerminalName());
    entity.setAreaCode(directStoreVo.getSupermarketAreaCode());
    entity.setAreaName(directStoreVo.getSupermarketAreaName());
    List<TerminalSupplyVo> terminalSupplyList = syncModel.getTerminalSupplyMap().get(entity.getStoreCode());
    if (!CollectionUtils.isEmpty(terminalSupplyList)) {
      terminalSupplyList = terminalSupplyList.stream().filter(terminalSupplyVo
          -> StringUtils.isNotBlank(terminalSupplyVo.getCustomerCode())).collect(Collectors.toList());
    }
    if (CollectionUtils.isEmpty(terminalSupplyList)) {
      entity.setOrderStatusMsg("未匹配售达方信息");
      return;
    } else if (terminalSupplyList.size() > 1) {
      entity.setOrderStatusMsg(String.format("终端门店[%s]找到多个售达方", entity.getStoreCode()));
      return;
    }
    TerminalSupplyVo terminalSupplyVo = terminalSupplyList.get(0);
    entity.setSoldToPartyCode(terminalSupplyVo.getCustomerCode());
    entity.setSoldToPartyName(terminalSupplyVo.getCustomerName());
    entity.setOrderStatus(OrderStatus.S200.getDictCode());
  }
}
