package com.biz.crm.kms.business.invoice.sales.data.local.imports.service;

import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.kms.business.invoice.sales.data.local.entity.SalesDataEntity;
import com.biz.crm.kms.business.invoice.sales.data.local.imports.model.InvoiceSalesImportsVo;
import com.biz.crm.kms.business.invoice.sales.data.local.service.InvoiceSalesDataImportsService;
import com.biz.crm.kms.business.invoice.sales.data.local.util.DictValueUtil;
import com.biz.crm.kms.business.invoice.sdk.constant.DirectConstant;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @Class InvoiceSalesDataImportsProcess
 * @Description 单据-销售数据导入服务层
 * @Author : CYJ
 * @Date: 2022/11/28  15:19
 */
@Component
@Slf4j
public class InvoiceSalesDataImportsProcess implements ImportProcess<InvoiceSalesImportsVo> {

    @Autowired
    private InvoiceSalesDataImportsService importsService;

    @Resource
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private DictDataVoService dictDataVoService;

    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, InvoiceSalesImportsVo> data,
                                        TaskGlobalParamsVo paramsVo,
                                        Map<String, Object> params) {
        log.info("导入数据数据{}",data);
        Validate.notEmpty(data, "导入数据不能为空！");

        List<SalesDataEntity> importList = this.validate(data);

        log.info("保存数据LOG：{}",importList);
        importsService.importsCreate(importList);
        return null;
    }

    /**
     * 校验并转换数据
     **/
    private List<SalesDataEntity> validate(LinkedHashMap<Integer, InvoiceSalesImportsVo> data) {
        List<InvoiceSalesImportsVo> invoiceStatementImportsVos = new ArrayList<>(data.values());
        Set<String> orderNumberList = invoiceStatementImportsVos.stream().map(InvoiceSalesImportsVo::getOrderNumber).collect(Collectors.toSet());
        Validate.isTrue(Objects.equals(invoiceStatementImportsVos.size(),orderNumberList.size()),"销售单号有重复");
        List<SalesDataEntity> salesDataEntities = new ArrayList<>();
        invoiceStatementImportsVos.forEach(importsVo -> {
            log.info("数据展示",importsVo);
            SalesDataEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(importsVo,SalesDataEntity.class,HashSet.class,ArrayList.class);
            //转换金额，单位数量等信息
            this.amountTrans(importsVo,entity);
            salesDataEntities.add(entity);
        });
        if (CollectionUtils.isEmpty(salesDataEntities)){
            return new ArrayList<>();
        }
        this.adjustData(salesDataEntities);
        return salesDataEntities;
    }

    /**
     * 校验销售数据 ，转换金额，单位数量等信息
     * @param vo
     * @param entity
     */
    public void amountTrans(InvoiceSalesImportsVo vo,SalesDataEntity entity){
        Validate.isTrue(StringUtils.isNotBlank(vo.getOrderNumber()),"KMS销售数据单号不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getDirectCode()),"系统编码不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getKaName()),"系统名称不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getBusinessFormatCode()),"业态不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getBusinessUnitCode()),"业务单元不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getSoldToPartyCode()),"售达方编码不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getSoldToPartyName()),"售达方名称不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getDeliveryPartyCode()),"送达方编码不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getDeliveryPartyName()),"送达方名称不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getKaStoreCode()),"零售商门店编码不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getKaStoreName()),"零售商门店名称不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getKaGoodsCode()),"零售商产品编码不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getKaGoodsName()),"零售商产品名称不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getGoodsCode()),"产品编码不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getGoodsName()),"产品编码不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getCurCompanyUnit()),"当前单位不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getCurCompanyUnitOrderQuantityStr()),"当前数量不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getCurUnitSalesQuantityStr()),"零售商产品数量不能为空");
//        Validate.isTrue(StringUtils.isNotBlank(vo.getSalesCostStr()),"销售成本不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getSalesAmountNoTaxStr()),"销售金额(不含税)不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getSalesAmountStr()),"销售金额(含税)不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getSalesDate()),"销售日期不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getReceiptAmountStr()),"收货金额不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getPromotionDeductionStr()),"促销扣款不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getPromotionAmountStr()),"促销金额不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getCouponAmountStr()),"优惠券金额不能为空");
        if (StringUtils.isNotBlank(vo.getCurCompanyUnitOrderQuantityStr())) {
            BigDecimal curCompanyUnitOrderQuantity = new BigDecimal(vo.getCurCompanyUnitOrderQuantityStr());
            entity.setCurCompanyUnitOrderQuantity(curCompanyUnitOrderQuantity);
        }
        if (StringUtils.isNotBlank(vo.getCurUnitSalesQuantityStr())) {
            BigDecimal curUnitSalesQuantity = new BigDecimal(vo.getCurUnitSalesQuantityStr());
            entity.setCurUnitSalesQuantity(curUnitSalesQuantity);
        }
        if (StringUtils.isNotBlank(vo.getSalesCostStr())){
            BigDecimal salesCost = new BigDecimal(vo.getSalesCostStr());
            entity.setSalesCost(salesCost);
        }
        if (StringUtils.isNotBlank(vo.getSalesCostNotStr())){
            BigDecimal salesCostNot = new BigDecimal(vo.getSalesCostNotStr());
            entity.setSalesCostNot(salesCostNot);

        }
        if (StringUtils.isNotBlank(vo.getSalesAmountNoTaxStr())){
            BigDecimal salesAmountNoTax = new BigDecimal(vo.getSalesAmountNoTaxStr());
            entity.setSalesAmountNoTax(salesAmountNoTax);
        }
        if (StringUtils.isNotBlank(vo.getSalesAmountStr())){
            BigDecimal salesAmount = new BigDecimal(vo.getSalesAmountStr());
            entity.setSalesAmount(salesAmount);
        }
        if (StringUtils.isNotBlank(vo.getReceiptAmountStr())){
            BigDecimal receiptAmount = new BigDecimal(vo.getReceiptAmountStr());
            entity.setReceiptAmount(receiptAmount);
        }
        if (StringUtils.isNotBlank(vo.getPromotionDeductionStr())){
            BigDecimal promotionDeduction = new BigDecimal(vo.getPromotionDeductionStr());
            entity.setPromotionDeduction(promotionDeduction);
        }
        if (StringUtils.isNotBlank(vo.getPromotionAmountStr())){
            BigDecimal PromotionAmount = new BigDecimal(vo.getPromotionAmountStr());
            entity.setPromotionAmount(PromotionAmount);
        }
        if (StringUtils.isNotBlank(vo.getCouponAmountStr())){
            BigDecimal couponAmount = new BigDecimal(vo.getCouponAmountStr());
            entity.setCouponAmount(couponAmount);
        }


    }

    /**
     * 调整数据
     *
     * @param data 待调整数据
     * @return 调整后数据
     */
    private List<SalesDataEntity> adjustData(List<SalesDataEntity> data) {
        if (CollectionUtils.isEmpty(data)) {
            return new ArrayList<>();
        }
        Map<String, List<DictDataVo>> fomatMap = this.dictDataVoService.findByDictTypeCodeList(Lists.newArrayList(DirectConstant.MDM_BUSINESS_FORMAT));
        Map<String, List<DictDataVo>> unitMap = this.dictDataVoService.findByDictTypeCodeList(Lists.newArrayList(DirectConstant.MDM_BUSINESS_UNIT));
        for (SalesDataEntity entity :data)
        {
            entity.setBusinessFormatCode(DictValueUtil.findDictValue(fomatMap, DirectConstant.MDM_BUSINESS_FORMAT, entity.getBusinessFormatCode()));
            entity.setBusinessUnitCode(DictValueUtil.findDictValue(unitMap, DirectConstant.MDM_BUSINESS_UNIT, entity.getBusinessUnitCode()));

        }

        return data;
    }

    @Override
    public Class<InvoiceSalesImportsVo> findCrmExcelVoClass() {
        return InvoiceSalesImportsVo.class;
    }

    @Override
    public String getTemplateCode() {
        return "KMS_INVOICE_SALES_DATA_IMPORT";
    }

    @Override
    public String getTemplateName() {
        return "KMS单据--POS数据";
    }
}
