package com.biz.crm.kms.business.invoice.sales.data.local.service.internal;

import cn.hutool.core.collection.CollectionUtil;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.kms.business.direct.product.sdk.vo.DirectProductUnitVo;
import com.biz.crm.kms.business.direct.product.sdk.vo.DirectProductVo;
import com.biz.crm.kms.business.direct.sdk.vo.DirectVo;
import com.biz.crm.kms.business.direct.store.sdk.vo.DirectStoreVo;
import com.biz.crm.kms.business.invoice.returns.order.sdk.constant.ReturnOrderConstant;
import com.biz.crm.kms.business.invoice.sales.data.local.entity.SalesDataEntity;
import com.biz.crm.kms.business.invoice.sales.data.local.repository.SalesDataGrabRepository;
import com.biz.crm.kms.business.invoice.sales.data.local.repository.SalesDataRepository;
import com.biz.crm.kms.business.invoice.sales.data.local.service.SalesDataGrabsThreadService;
import com.biz.crm.kms.business.invoice.sales.data.sdk.constant.SalesDataConstant;
import com.biz.crm.kms.business.invoice.sales.data.sdk.enums.SalesDataStatus;
import com.biz.crm.kms.business.invoice.sdk.enums.GrabTransStatus;
import com.biz.crm.kms.business.invoice.sdk.enums.InvoicesStatus;
import com.biz.crm.mn.common.base.service.RedisCrmService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @Title: SalesDataGrabsThreadServiceImpl
 * @Description:
 * @Author: cyj
 * @Date: 2022/11/25 - 16:46 PM
 */
@Slf4j
@Service("salesDataGrabsThreadService")
public class SalesDataGrabsThreadServiceImpl implements SalesDataGrabsThreadService {

    @Autowired
    private SalesDataRepository salesDataRepository;
    
    @Autowired
    private SalesDataGrabRepository salesDataGrabRepository;

    @Autowired
    private RedisCrmService redisCrmService;

    @Autowired(required = false)
    private RedisTemplate redisTemplate;

    @Override
    @Async("grab2Thread")
    @Transactional(propagation = Propagation.NOT_SUPPORTED,rollbackFor = Exception.class)
    public void saveSalesData(List<SalesDataEntity> entities, Map<String, List<DirectVo>> directMap, Map<String, List<DirectStoreVo>> directStoreMap, Map<String, List<DirectProductVo>> directProductMap) {
        log.info("--  进入线程池存储数据  --");
        List<String> orderNumbers = entities.stream().map(SalesDataEntity::getOrderNumber).collect(Collectors.toList());
        Set<String> orderNumberSet = new HashSet<>();
        Set<String> orderNumberSuccessSet = new HashSet<>();
        entities.forEach(entity -> {
            try {
                //处理数据前先进行解锁
                redisCrmService.hdel(SalesDataConstant.SALES_DATA_CONVERT_LOCK, entity.getOrderNumber());
                //清除数据中的售达方，企业门店，区域等数据 第一步
                this.clearTransformInfoOne(entity,directMap);
                if (StringUtils.isNotBlank(entity.getKaStoreCode())){
                    //拼接门店数据  第二步
                    String deliveryCode = this.buildOrderInfoTwo(entity, directStoreMap);
                    if (StringUtils.isEmpty(entity.getKaGoodsCode())){
                        //拼接商品数据  第三步
                        this.convertAcceptanceOrderItemThree(entity, directProductMap, deliveryCode);
                    }
                }else {
                    entity.setOrderStatusMsg(null);
                    entity.setOrderStatus(InvoicesStatus.S200.getDictCode());
                }
                // 设置含税未税
                entity = setAccount(entity);
                //设置状态  第四步
                this.setOrderStatusAndMsgFour(entity);
                this.salesDataRepository.saveOrUpdate(entity);
                if (Objects.equals(InvoicesStatus.S200.getDictCode(),entity.getOrderStatus())){
                    orderNumberSuccessSet.add(entity.getOrderNumber());
                }else {
                    orderNumberSet.add(entity.getOrderNumber());
                }

            } catch (Exception e) {
                log.error("保存销售数据数据失败", e);
                entity.setOrderStatus(InvoicesStatus.S100.getDictCode());
                entity.setTransFlag(BooleanEnum.TRUE.getCapital());
                entity.setOrderStatusMsg("转换时异常!");
                try {
                    this.salesDataRepository.saveOrUpdate(entity);
                    orderNumberSet.add(entity.getOrderNumber());
                } catch (Exception ex) {
                    log.error("", ex);
                }
            }
        });


        if (CollectionUtil.isEmpty(orderNumberSet) && CollectionUtils.isEmpty(orderNumberSuccessSet)) {
            return;
        }
        //处理数据前先进行解锁
        redisCrmService.hdel(SalesDataConstant.SALES_DATA_CONVERT_LOCK, orderNumbers);
        try {
            if (!CollectionUtils.isEmpty(orderNumberSet)){
                salesDataGrabRepository.updateTransStatusByOrderNumberList(TenantUtils.getTenantCode(),
                        GrabTransStatus.TRANS_FAIL, new ArrayList<>(orderNumberSet));
            }
            if (!CollectionUtils.isEmpty(orderNumberSuccessSet)){
                salesDataGrabRepository.updateTransStatusByOrderNumberList(TenantUtils.getTenantCode(),
                        GrabTransStatus.HAVE_TRANS, new ArrayList<>(orderNumberSuccessSet));
            }


        } catch (Exception e) {
            log.error("", e);
        }
    }

    /**
     * 含税未税
     * @param entity
     * @return
     */
    private SalesDataEntity setAccount(SalesDataEntity entity) {
        BigDecimal decimal = new BigDecimal(1);
        Map<String, BigDecimal> map = this.redisTemplate.hasKey(ReturnOrderConstant.CACHE_PRODUCT_TAX_RATE) ? (Map<String, BigDecimal>) this.redisTemplate.opsForValue().get(ReturnOrderConstant.CACHE_PRODUCT_TAX_RATE) : null;
        String goodsCode = entity.getGoodsCode();
        if (!StringUtils.isEmpty(goodsCode)) {
            if (Objects.isNull(map)) {
                return entity;
            }
            BigDecimal bigDecimal = map.get(goodsCode);
            if (Objects.isNull(bigDecimal)){
                return entity;
            }
            BigDecimal amount = entity.getSalesAmount();
            BigDecimal amountNot = entity.getSalesAmountNoTax();
            if (Objects.nonNull(amount)){
                if (BigDecimal.ZERO.compareTo(amount) != 0) {
                    BigDecimal multiply = amount.divide(decimal.add(bigDecimal).setScale(4, BigDecimal.ROUND_HALF_UP),4,BigDecimal.ROUND_HALF_UP);
                    entity.setSalesAmountNoTax(multiply);
                }
            }
            if (Objects.nonNull(amountNot)){
                if (BigDecimal.ZERO.compareTo(amountNot) != 0) {
                    BigDecimal multiply = amountNot.multiply(decimal.add(bigDecimal).setScale(4, BigDecimal.ROUND_HALF_UP));
                    entity.setSalesAmount(multiply.setScale(4, BigDecimal.ROUND_HALF_UP));
                }
            }

        }
        return entity;
    }

    /**
    * @Description: 第一步：清除销售组织、区域、商超
    * @Param: [entity]
    * @return: void
    * @Author: ZS
    * @Date: 2022/12/16
    */
    public void clearTransformInfoOne(SalesDataEntity entity,Map<String, List<DirectVo>> directMap) {
        entity.setStoreCode(null);
        entity.setStoreName(null);
        entity.setAreaName(null);
        entity.setAreaCode(null);
        //单据时间转换
        if (StringUtils.isNotBlank(entity.getSalesDate())){
            try {
                entity.setOrderTime(new SimpleDateFormat("yyyy-MM-dd").parse(entity.getSalesDate()));
            } catch (ParseException e) {
                e.printStackTrace();
            }
        }
        //业态业务单元赋值
        List<DirectVo> directVos = directMap.get(entity.getDirectCode());
        if (CollectionUtils.isEmpty(directVos)){
            entity.setOrderStatusMsg("不存在该系统");
        }else {
            DirectVo directVo = directVos.get(0);
            entity.setBusinessFormatCode(directVo.getBusinessFormatCode());
            entity.setBusinessUnitCode(directVo.getBusinessUnitCode());
            entity.setDirectCode(directVo.getDirectCode());
            entity.setDirectName(directVo.getSupermarketName());
            entity.setKaName(directVo.getSupermarketName());
        }
    }

    /**
    * @Description: 第二步：拼接门店数据
    * @Param: [entity, directStoreMap]
    * @return: void
    * @Author: ZS
    * @Date: 2022/12/16
    */
    public String buildOrderInfoTwo(SalesDataEntity entity, Map<String, List<DirectStoreVo>> directStoreMap) {
        entity.setOrderStatus(SalesDataStatus.S100.getDictCode());
        entity.setSoldToPartyName(null);
        entity.setSoldToPartyCode(null);
        if (StringUtils.isBlank(entity.getDirectCode())) {
            entity.setOrderStatusMsg("系统编码为空");
            return null;
        }
        if (StringUtils.isBlank(entity.getKaStoreCode())) {
            entity.setOrderStatusMsg("单据商超门店编码为空");
            return null;
        }
        List<DirectStoreVo> directStoreList = directStoreMap.get(entity.getDirectCode() + "_" + entity.getKaStoreCode());
        DirectStoreVo directStoreVo = null;
        if (CollectionUtil.isEmpty(directStoreList)) {
            entity.setOrderStatusMsg("未查询到与企业门店对应关系");
            return null;
        } else if (directStoreList.size() > 1) {
            entity.setOrderStatusMsg("商超门店与企业门店对应关系存在[" + directStoreList.size() + "]条");
            return null;
        } else {
            directStoreVo = directStoreList.get(0);
        }

        //售达方信息
        entity.setSoldToPartyCode(directStoreVo.getSoldToPartyCode());
        entity.setSoldToPartyName(directStoreVo.getSoldToPartyName());
        entity.setDeliveryPartyCode(directStoreVo.getTerminalCode());
        entity.setDeliveryPartyName(directStoreVo.getTerminalName());
        entity.setKaStoreCode(directStoreVo.getSupermarketStoreCode());
        if(StringUtils.isEmpty(entity.getKaStoreName())){
            entity.setKaStoreName(directStoreVo.getSupermarketStoreName());
        }
        entity.setAreaName(directStoreVo.getRetailerRegion());
        entity.setBusinessArea(directStoreVo.getBusinessArea());
        entity.setProvinceCode(directStoreVo.getProvinceCode());
        entity.setProvinceName(directStoreVo.getProvinceName());
        return directStoreVo.getRetailerTerminalCode();
    }

    /**
    * @Description: 第三步：拼接商品数据
    * @Param: [entity, directProductMap]
    * @return: void
    * @Author: ZS
    * @Date: 2022/12/16
    */
    public void convertAcceptanceOrderItemThree(SalesDataEntity entity, Map<String, List<DirectProductVo>> directProductMap, String deliveryCode) {
        if (StringUtils.isBlank(entity.getKaGoodsCode())) {
            entity.setOrderStatusMsg("商品编码为空");
            return;
        }
        DirectProductVo productVo = null;
        List<DirectProductVo> productVoList = new ArrayList<>();
        List<DirectProductVo> productVoLists = directProductMap.get(entity.getDirectCode() + "_" +
                entity.getDeliveryPartyCode() + "_" + entity.getKaGoodsCode());
        if (CollectionUtil.isNotEmpty(productVoLists)){
            productVoList.addAll(productVoLists);
        }
        //如果送达方编码没有找到，那么会通过关联的送达方编码去查
        if (CollectionUtil.isEmpty(productVoList) && StringUtils.isNotBlank(deliveryCode)){
            List<DirectProductVo> productVos = directProductMap.get(entity.getDirectCode() + "_" +
                    deliveryCode + "_" + entity.getKaGoodsCode());
            if (CollectionUtil.isNotEmpty(productVos)){
                productVoList.addAll(productVos);
            }
        }
        if (CollectionUtil.isEmpty(productVoList)) {
            entity.setOrderStatusMsg("未查询到商品对应关系");
        } else if (productVoList.size() == 1) {
            productVo = productVoList.get(0);
        } else {
            int allSize = productVoList.size();
            //匹配到多条数据  用是否默认标记为是的数据
            productVoList = productVoList.stream()
                    .filter(k -> BooleanEnum.TRUE.getCapital().equals(k.getIsDefault()))
                    .collect(Collectors.toList());
            if (CollectionUtil.isEmpty(productVoList)) {
                entity.setOrderStatusMsg("共[" + allSize + "]条商品对应关系,无默认对应关系");
            } else if (productVoList.size() > 1) {
                entity.setOrderStatusMsg("[" + productVoList.size() + "]条默认商品对应关系");
            } else {
                productVo = productVoList.get(0);
            }
        }
        if (Objects.isNull(productVo)) {
            return;
        }
        //产品信息
        entity.setGoodsCode(productVo.getProductCode());
        entity.setGoodsName(productVo.getProductName());
        //产品单位信息
        List<DirectProductUnitVo> unitList = productVo.getUnitList();
        if (StringUtils.isNotBlank(entity.getCurUnit())){
            //商超单位不为空
            unitList = unitList.stream()
                    .filter(vo -> Objects.nonNull(vo.getKaUnitQuantity())
                            && Objects.nonNull(vo.getUnitQuantity()))
                    .filter(vo -> Objects.equals(entity.getCurUnit(), vo.getKaUnitCode()))
                    .collect(Collectors.toList());
        }
        //当商超单位找不到对应的单位关系时，这种情况就找默认上架产品+相应单据+单位为空的第一个转换关系
        if (CollectionUtils.isEmpty(unitList)){
            //商超单位为空
            unitList = productVo.getUnitList().stream()
                    .filter(vo -> Objects.nonNull(vo.getKaUnitQuantity())
                            && Objects.nonNull(vo.getUnitQuantity()))
                    .filter(vo -> StringUtils.isBlank(vo.getKaUnitCode()))
                    .collect(Collectors.toList());
        }

        if (CollectionUtils.isEmpty(unitList)) {
            entity.setOrderStatusMsg("未匹配企业商品单位转换关系");
            return;
        }

        DirectProductUnitVo unitVo = unitList.get(0);
        if (!StringUtils.isEmpty(unitVo.getUnitName())){
            entity.setCurCompanyUnit(unitVo.getUnitName());
        }else {
            entity.setOrderStatusMsg("企业单位名称为空");
        }
        if (Objects.nonNull(entity.getCurUnitSalesQuantity())) {
            BigDecimal curUnitAcceptanceQuantity = unitVo.getUnitQuantity()
                    .multiply(entity.getCurUnitSalesQuantity())
                    .divide(unitVo.getKaUnitQuantity(), 6, BigDecimal.ROUND_HALF_UP);
            entity.setCurCompanyUnitOrderQuantity(curUnitAcceptanceQuantity);
        }
        //转换状态
        entity.setOrderStatus(InvoicesStatus.S200.getDictCode());
    }

    /**
    * @Description: 第四步，转换状态
    * @Param: [entity]
    * @return: void
    * @Author: ZS
    * @Date: 2022/12/16
    */
    public void setOrderStatusAndMsgFour(SalesDataEntity entity) {
        if (Objects.equals(entity.getOrderStatus(), SalesDataStatus.S200.getDictCode())) {
            entity.setOrderStatusMsg(SalesDataConstant.SALES_DATA_CONVERT_SUCCESS_MSG);
        }
    }
}
