package com.biz.crm.kms.business.invoice.sales.data.local.service.internal;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.IdUtil;
import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.kms.business.direct.account.sdk.service.DirectAccountVoService;
import com.biz.crm.kms.business.direct.account.sdk.vo.DirectAccountVo;
import com.biz.crm.kms.business.direct.product.sdk.dto.DirectProductDto;
import com.biz.crm.kms.business.direct.product.sdk.service.DirectProductVoService;
import com.biz.crm.kms.business.direct.product.sdk.vo.DirectProductUnitVo;
import com.biz.crm.kms.business.direct.product.sdk.vo.DirectProductVo;
import com.biz.crm.kms.business.direct.sdk.service.DirectVoService;
import com.biz.crm.kms.business.direct.sdk.vo.DirectVo;
import com.biz.crm.kms.business.direct.store.sdk.dto.DirectStoreConditionDto;
import com.biz.crm.kms.business.direct.store.sdk.service.DirectStoreVoService;
import com.biz.crm.kms.business.direct.store.sdk.vo.DirectStoreVo;
import com.biz.crm.kms.business.invoice.returns.order.sdk.constant.ReturnOrderConstant;
import com.biz.crm.kms.business.invoice.sales.data.local.entity.SalesDataEntity;
import com.biz.crm.kms.business.invoice.sales.data.local.model.SalesDataConditionModel;
import com.biz.crm.kms.business.invoice.sales.data.local.register.SaleInvoiceTypeRegister;
import com.biz.crm.kms.business.invoice.sales.data.local.repository.SalesDataRepository;
import com.biz.crm.kms.business.invoice.sales.data.local.service.SalesDataGrabsService;
import com.biz.crm.kms.business.invoice.sales.data.local.service.SalesDataService;
import com.biz.crm.kms.business.invoice.sales.data.sdk.constant.SalesDataConstant;
import com.biz.crm.kms.business.invoice.sdk.constant.DirectConstant;
import com.biz.crm.kms.business.invoice.sdk.enums.ConstantEnums;
import com.biz.crm.kms.business.invoice.sdk.enums.InvoicesStatus;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.third.system.dataphin.sdk.DataphinService;
import com.biz.crm.mn.third.system.dataphin.sdk.vo.PosDataVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 销售数据表服务实现类
 *
 * @author ning.zhang
 * @date 2022-11-08 10:03:14
 */
@Slf4j
@Service("salesDataService")
public class SalesDataServiceImpl implements SalesDataService {

    @Autowired
    private SalesDataRepository salesDataRepository;
    @Autowired
    private DataphinService dataphinService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;
    @Autowired
    private LoginUserService loginUserService;

    @Autowired
    private SalesDataGrabsService salesDataGrabsService;

    @Autowired(required = false)
    private SaleInvoiceTypeRegister saleInvoiceTypeRegister;

    @Autowired(required = false)
    private DirectAccountVoService directAccountVoService;

    @Autowired(required = false)
    private DirectVoService directVoService;

    @Autowired(required = false)
    private DirectStoreVoService directStoreVoService;

    @Autowired(required = false)
    private DirectProductVoService directProductVoService;

    @Autowired(required = false)
    private RedisTemplate redisTemplate;

    /**
     * 新增销售数据
     *
     * @param salesDataEntities
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void create(List<SalesDataEntity> salesDataEntities) {
        if (CollectionUtils.isEmpty(salesDataEntities)) {
            return;
        }
        salesDataEntities.forEach(aa -> {
            this.salesDataRepository.save(aa);
        });

    }

    /**
     * 查找id集合
     *
     * @param ids 销售数据ID集合
     * @return
     */
    @Override
    public List<SalesDataEntity> findByIds(Set<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return Lists.newLinkedList();
        }
        SalesDataConditionModel model = new SalesDataConditionModel();
        model.setIds(ids);
        model.setTenantCode(TenantUtils.getTenantCode());
        return this.salesDataRepository.findBySalesDataConditionModel(model);
    }

    /**
     * @Description: 手动转换销售数据
     * @Param: [ids]
     * @return: void
     * @Author: ZS
     * @Date: 2022/12/16
     */
    @Override
    public void manualTransSaleData(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids), "id集合不能为空！");
        List<SalesDataEntity> entities = this.salesDataRepository.findByIds(ids);
        Validate.isTrue(!CollectionUtils.isEmpty(entities), "未查询到数据，请刷新重试");
        Validate.isTrue(!ObjectUtils.notEqual(entities.size(), ids.size()), "数据转换个数不匹配");
        entities.forEach(entity -> Validate.isTrue(Lists.newArrayList(InvoicesStatus.S200.getDictCode(), InvoicesStatus.S101.getDictCode(), InvoicesStatus.S100.getDictCode()).contains(entity.getOrderStatus())
                , String.format("销售数据[%s]已确认,无法继续匹配", entity.getOrderNumber())));
        List<String> orderNumbers = entities.stream()
                .filter(k -> StringUtils.isNotBlank(k.getOrderNumber()))
                .map(SalesDataEntity::getOrderNumber).distinct().collect(Collectors.toList());
        //过滤正在转换的销售数据号
        this.salesDataGrabsService.filterSwitchIngOrderNumberList(orderNumbers);
        log.info("===== 销售数据手动转换开始 ======");
        this.salesDataGrabsService.manualSwitch(orderNumbers);
        log.info("===== 销售数据手动转换完成 ======");
    }

    /**
     * 每天晚上10点同步数据中台的POS销售单
     */
    @DynamicTaskService(cornExpression = "0 0 22 * * ?", taskDesc = "同步数据中台POS销售数据")
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void syncPosData() {
        Date nowTime = new Date();
        String date = DateUtil.format(DateUtil.dateAddDay(nowTime, -1), "yyyyMMdd");
        this.posData(date);
    }

    /**
     * @param date 预留日期参数期初视情况调用
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void posData(String date) {
        Date nowTime = new Date();
        if (StringUtils.isEmpty(date)) {
            date = DateUtil.format(DateUtil.dateAddDay(nowTime, -1), "yyyyMMdd");
        }
        int pageSize = 1000;
        int pageNumber = 0;
        Boolean exitFlag = Boolean.TRUE;
        while (exitFlag) {
            log.info("开始查询pos数据");
            List<PosDataVo> data = dataphinService.getPosData(date, pageNumber, pageSize);
            log.info("获取到pos数据{}", JSON.toJSON(data));
            if (!CollectionUtils.isEmpty(data)) {
                //第一个数据库 ：高鑫和麦德龙
                //用户编码集
                Set<String> user = data.stream().map(PosDataVo::getUserName).collect(Collectors.toSet());
                List<DirectAccountVo> directAccountVos = this.directAccountVoService.findByAccountList(user.stream().collect(Collectors.toList()));
                if (CollectionUtils.isEmpty(directAccountVos)) {
                    return;
                }
                Set<String> directCodeSet = directAccountVos.stream().map(DirectAccountVo::getDirectCode).collect(Collectors.toSet());
                List<DirectVo> directVos = this.directVoService.findByDirectCodes(directCodeSet.stream().collect(Collectors.toList()));
                //获取系统数据
                Map<String, List<DirectVo>> directMap = Maps.newHashMap();
                if (!CollectionUtils.isEmpty(directVos)) {
                    directMap.putAll(directVos.stream()
                            .filter(k -> StringUtils.isNotBlank(k.getBusinessUnitCode()) && StringUtils.isNotBlank(k.getBusinessFormatCode()))
                            .collect(Collectors.groupingBy(DirectVo::getDirectCode)));
                }
                //高鑫和麦德龙的数据
                this.coversAndSave(data, directCodeSet, directMap);
            }
            else if (CollectionUtils.isEmpty(data)){
                exitFlag = false;
                return;
            }
            pageNumber++;
        }
    }

    /**
     * @param date 预留日期参数期初视情况调用
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void getPostDataOne(String date) {
        Date nowTime = new Date();
        if (StringUtils.isEmpty(date)) {
            date = DateUtil.format(DateUtil.dateAddDay(nowTime, -1), "yyyyMMdd");
        }
        int pageSize = 1000;
        int pageNumber = 0;
        Boolean exitFlag = Boolean.TRUE;
        while (exitFlag) {
            log.info("开始查询pos2数据");
            List<PosDataVo> dataTwo = dataphinService.getPosDataTwo(date, pageNumber, pageSize);
            log.info("获取到pos2数据{}", JSON.toJSON(dataTwo));
            if (!CollectionUtils.isEmpty(dataTwo)) {
                //第二个数据库 ：沃尔玛
                List<DirectVo> directVos = this.directVoService.findByDirectCodes(Lists.newArrayList(DirectConstant.KMS_WEM_DIRECTCODE));
                if (CollectionUtils.isEmpty(directVos)) {
                    return;
                }
                Map<String, List<DirectVo>> directMapTwo = directVos.stream()
                        .filter(k -> StringUtils.isNotBlank(k.getBusinessUnitCode()) && StringUtils.isNotBlank(k.getBusinessFormatCode()))
                        .collect(Collectors.groupingBy(DirectVo::getDirectCode));
                //沃尔玛的数据
                this.coversAndSave(dataTwo, Sets.newHashSet(DirectConstant.KMS_WEM_DIRECTCODE), directMapTwo);
            }
            else if (CollectionUtils.isEmpty(dataTwo)){
                exitFlag = false;
                return;
            }
            pageNumber++;
        }
    }

    /**
     * @param data
     * @param directCodeSet
     * @param directMap
     */
    private void coversAndSave(List<PosDataVo> data, Set<String> directCodeSet, Map<String, List<DirectVo>> directMap) {
        AbstractCrmUserIdentity loginUser = loginUserService.getAbstractLoginUser();
        //零售商门店集合
        Set<String> kaStoreCodes = data.stream().map(PosDataVo::getKaStoreCode).collect(Collectors.toSet());
        //零售商产品集合
        Set<String> kaGoodsCodes = data.stream().map(PosDataVo::getKaGoodsCode).collect(Collectors.toSet());

        Map<String, List<DirectStoreVo>> directStoreMap = this.buildStoreInfo(directCodeSet, kaStoreCodes);
        //门店中的送达方集合
        Set<String> deliveryPartyCodeSet = Sets.newHashSet();
        if (CollectionUtil.isNotEmpty(directStoreMap)) {
            directStoreMap.values().stream()
                    .filter(CollectionUtil::isNotEmpty)
                    .forEach(directStoreVoList -> {
                        directStoreVoList.forEach(entity -> {
                            deliveryPartyCodeSet.add(entity.getTerminalCode());
                        });
                    });
        }
        //商品map 系统编码+送达方编码+商超产品编码 构成唯一
        Map<String, List<DirectProductVo>> directProductMap = this.buildProductInfo(directCodeSet, deliveryPartyCodeSet, kaGoodsCodes);
        List<String> username = new ArrayList<>();
        Collection<SalesDataEntity> entities = nebulaToolkitService.copyCollectionByWhiteList(data, PosDataVo.class, SalesDataEntity.class, HashSet.class, ArrayList.class);
        entities.forEach(o -> {
            if (StringUtils.isBlank(o.getUserName())) {
                o.setOrderNumber("WEM-" + o.getKaStoreCode() + "-" + o.getKaGoodsCode() + "-" + o.getSalesDate().replaceAll("-", "").substring(0, 8));
                o.setDirectCode(DirectConstant.KMS_WEM_DIRECTCODE);
            } else {
                o.setOrderNumber(o.getUserName() + "-" + o.getKaStoreCode() + "-" + o.getKaGoodsCode() + "-" + o.getSalesDate().replaceAll("-", "").substring(0, 8));
                username.clear();
                username.add(o.getUserName());
                List<DirectAccountVo> directAccountVoList = this.directAccountVoService.findByAccountList(username);
                if (CollectionUtils.isEmpty(directAccountVoList)) {
                    return;
                }
                o.setDirectCode(directAccountVoList.get(0).getDirectCode());
            }
            //单据时间转换
            if (StringUtils.isNotBlank(o.getSalesDate())){
                try {
                    o.setOrderTime(new SimpleDateFormat("yyyy-MM-dd").parse(o.getSalesDate()));
                } catch (ParseException e) {
                    e.printStackTrace();
                }
            }
            o.setId(IdUtil.fastUUID().replace("-", ""));
            if (!CollectionUtils.isEmpty(directMap)) {
                List<DirectVo> directVoList = directMap.get(o.getDirectCode());
                DirectVo directVo = directVoList.get(0);
                o.setBusinessFormatCode(directVo.getBusinessFormatCode());
                o.setBusinessUnitCode(directVo.getBusinessUnitCode());
                o.setDirectCode(directVo.getDirectCode());
                o.setDirectName(directVo.getSupermarketName());
                o.setKaCode(directVo.getSupermarketCode());
                o.setKaName(directVo.getSupermarketName());
            }
            if (!CollectionUtils.isEmpty(directStoreMap)) {
                List<DirectStoreVo> directStoreVoList = directStoreMap.get(o.getDirectCode() + "_" + o.getKaStoreCode());
                if (!CollectionUtils.isEmpty(directStoreVoList)) {
                    DirectStoreVo directStoreVo = directStoreVoList.get(0);
                    o.setSoldToPartyCode(directStoreVo.getSoldToPartyCode());
                    o.setSoldToPartyName(directStoreVo.getSoldToPartyName());
                    o.setDeliveryPartyCode(directStoreVo.getTerminalCode());
                    o.setDeliveryPartyName(directStoreVo.getTerminalName());
                    o.setAreaName(directStoreVo.getRetailerRegion());
                    o.setBusinessArea(directStoreVo.getBusinessArea());
                }
            }
            if (!CollectionUtils.isEmpty(directProductMap)) {
                if (StringUtils.isBlank(o.getKaGoodsCode())) {
                    o.setOrderStatusMsg("商品编码为空");
                    return;
                }
                DirectProductVo productVo = null;
                List<DirectProductVo> productVoList = directProductMap.get(o.getDirectCode() + "_" +
                        o.getDeliveryPartyCode() + "_" + o.getKaGoodsCode());
                if (CollectionUtil.isEmpty(productVoList)) {
                    o.setOrderStatusMsg("未查询到商品对应关系");
                } else if (productVoList.size() == 1) {
                    productVo = productVoList.get(0);
                } else {
                    int allSize = productVoList.size();
                    //匹配到多条数据  用是否默认标记为是的数据
                    productVoList = productVoList.stream()
                            .filter(k -> BooleanEnum.TRUE.getCapital().equals(k.getIsDefault()))
                            .collect(Collectors.toList());
                    if (CollectionUtil.isEmpty(productVoList)) {
                        o.setOrderStatusMsg("共[" + allSize + "]条商品对应关系,无默认对应关系");
                    } else if (productVoList.size() > 1) {
                        o.setOrderStatusMsg("[" + productVoList.size() + "]条默认商品对应关系");
                    } else {
                        productVo = productVoList.get(0);
                    }
                }
                if (Objects.isNull(productVo)) {
                    return;
                }
                //产品信息
                o.setGoodsCode(productVo.getProductCode());
                o.setGoodsName(productVo.getProductName());
                //产品单位信息
                List<DirectProductUnitVo> unitList = productVo.getUnitList();
                if (StringUtils.isNotBlank(o.getCurUnit())) {
                    String curUnit = o.getCurUnit();
                    //商超单位不为空
                    unitList = unitList.stream()
                            .filter(vo -> Objects.nonNull(vo.getKaUnitQuantity())
                                    && Objects.nonNull(vo.getUnitQuantity()))
                            .filter(vo -> Objects.equals(curUnit, vo.getKaUnitCode()))
                            .collect(Collectors.toList());
                }
                //当商超单位找不到对应的单位关系时，这种情况就找默认上架产品+相应单据+单位为空的第一个转换关系
                if (CollectionUtils.isEmpty(unitList)) {
                    //商超单位为空
                    unitList = productVo.getUnitList().stream()
                            .filter(vo -> Objects.nonNull(vo.getKaUnitQuantity())
                                    && Objects.nonNull(vo.getUnitQuantity()))
                            .filter(vo -> StringUtils.isBlank(vo.getKaUnitCode()))
                            .collect(Collectors.toList());
                }

                if (CollectionUtils.isEmpty(unitList)) {
                    o.setOrderStatusMsg("未匹配企业商品单位转换关系");
                    return;
                }

                DirectProductUnitVo unitVo = unitList.get(0);
                o.setCurCompanyUnit(unitVo.getUnitName());
                if (!StringUtils.isEmpty(unitVo.getUnitName())) {
                    o.setCurCompanyUnit(unitVo.getUnitName());
                } else {
                    o.setOrderStatusMsg("企业单位名称为空");
                }
                if (Objects.nonNull(o.getCurUnitSalesQuantity())) {
                    BigDecimal curUnitAcceptanceQuantity = unitVo.getUnitQuantity()
                            .multiply(o.getCurUnitSalesQuantity())
                            .divide(unitVo.getKaUnitQuantity(), 6, BigDecimal.ROUND_HALF_UP);
                    o.setCurCompanyUnitOrderQuantity(curUnitAcceptanceQuantity);
                }
            }
            o.setTenantCode(TenantUtils.getTenantCode());
            o.setCreateName(loginUser.getRealName());
            o.setCreateAccount(loginUser.getAccount());
            o.setCreateTime(new Date());
            o.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            o.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            if (StringUtils.isEmpty(o.getOrderStatusMsg())){
                o.setOrderStatus(InvoicesStatus.S200.getDictCode());
                o.setOrderStatusMsg(SalesDataConstant.SALES_DATA_CONVERT_SUCCESS_MSG);
            }else {
                o.setOrderStatus(InvoicesStatus.S100.getDictCode());
            }
            o.setInvoicesSource(SalesDataConstant.SALES_DATA_IMPORT_INTERFACE);
            // 设置含税未税
            o = setAccount(o);
        });
        Set<String> orderNumbers = entities.stream().map(SalesDataEntity::getOrderNumber).collect(Collectors.toSet());
        List<SalesDataEntity> salesDataEntities = this.salesDataRepository.findByOrderNumbers(orderNumbers);
        if (!CollectionUtils.isEmpty(salesDataEntities)) {
            List<String> ids = salesDataEntities.stream().map(SalesDataEntity::getId).collect(Collectors.toList());
            this.salesDataRepository.removeByIds(ids);
        }
        entities = entities.stream()
                .filter(f -> StringUtils.isNotBlank(f.getDirectCode()))
                .filter(f -> StringUtils.isNotBlank(f.getDeliveryPartyCode()))
                .filter(f -> StringUtils.isNotBlank(f.getGoodsCode()))
                .filter(f -> Objects.nonNull(f.getCurCompanyUnitOrderQuantity()))
                .collect(Collectors.toMap(SalesDataEntity::getOrderNumber, v -> v, (oldValue, newValue) -> newValue)).values();
        List<SalesDataEntity> salesDataEntityList = entities.stream().collect(Collectors.toList());
        salesDataRepository.saveBatch(salesDataEntityList);
    }

    /**
     * 构建门店信息
     * <p>
     * 门店map 系统编码+商超门店编码 构成唯一
     *
     * @param directCodeSet 系统编码集合
     * @param kaStoreCodes  商超门店编码集合
     */
    private Map<String, List<DirectStoreVo>> buildStoreInfo(Set<String> directCodeSet, Set<String> kaStoreCodes) {
        Map<String, List<DirectStoreVo>> directStoreMap = Maps.newHashMap();
        if (CollectionUtil.isEmpty(kaStoreCodes)
                || CollectionUtil.isEmpty(directCodeSet)) {
            return directStoreMap;
        }
        DirectStoreConditionDto conditionDto = new DirectStoreConditionDto();
        conditionDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        conditionDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        conditionDto.setSupermarketStoreCodes(kaStoreCodes);
        conditionDto.setDirectCodes(directCodeSet);
        List<DirectStoreVo> directStoreList = this.directStoreVoService.findByDirectStoreConditionDto(conditionDto);
        if (CollectionUtil.isNotEmpty(directStoreList)) {
            directStoreMap.putAll(directStoreList.stream()
                    .filter(k -> StringUtils.isNotBlank(k.getDirectCode()))
                    .filter(k -> StringUtils.isNotBlank(k.getSupermarketStoreCode()))
                    .collect(Collectors.groupingBy(k -> k.getDirectCode()
                            + "_" + k.getSupermarketStoreCode())));
        }
        return directStoreMap;
    }

    /**
     * 构建商品信息
     * <p>
     * 门店map 系统编码+商超门店编码 构成唯一
     *
     * @param directCodeSet        系统编码集合
     * @param deliveryPartyCodeSet 商超门店编码集合
     * @param kaGoodsCodes         商超商品编码集合
     */
    private Map<String, List<DirectProductVo>> buildProductInfo(Set<String> directCodeSet,
                                                                Set<String> deliveryPartyCodeSet,
                                                                Set<String> kaGoodsCodes) {
        Map<String, List<DirectProductVo>> directProductMap = Maps.newHashMap();
        if (CollectionUtil.isEmpty(directCodeSet)
                || CollectionUtil.isEmpty(deliveryPartyCodeSet)
                || CollectionUtil.isEmpty(kaGoodsCodes)) {
            return directProductMap;
        }
        /**
         * deliveryPartyCode 送达方编码
         * terminalCode 企业门店编码
         * storeCode 企业门店编码
         *
         * 三个字段含义一样 值一样
         */

        List<List<String>> kaProductCodeGroupList = Lists.partition(Lists.newArrayList(kaGoodsCodes), ConstantEnums.PRODUCT_SIZE.getValue());
        DirectProductDto directProductDto = new DirectProductDto();
        directProductDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        directProductDto.setTenantCode(TenantUtils.getTenantCode());
        directProductDto.setOnShelfStatus(BooleanEnum.TRUE.getCapital());
        directProductDto.setDirectCodes(new ArrayList<>(directCodeSet));
        directProductDto.setDeliveryPartyCodes(new ArrayList<>(deliveryPartyCodeSet));
        directProductDto.setOrderType(saleInvoiceTypeRegister.type());
        kaProductCodeGroupList.forEach(kaProductCodeList -> {
            directProductDto.setKaProductCodes(kaProductCodeList);
            List<DirectProductVo> productList = this.directProductVoService.findByDirectProductDto(directProductDto);
            if (CollectionUtil.isNotEmpty(productList)) {
                directProductMap.putAll(productList.stream()
                        .filter(k -> BooleanEnum.TRUE.getCapital().equals(k.getOnShelfStatus()))
                        .filter(k -> StringUtils.isNotBlank(k.getDirectCode()))
                        .filter(k -> StringUtils.isNotBlank(k.getDeliveryPartyCode()))
                        .filter(k -> StringUtils.isNotBlank(k.getKaProductCode()))
                        .collect(Collectors.groupingBy(k -> k.getDirectCode()
                                + "_" + k.getDeliveryPartyCode()
                                + "_" + k.getKaProductCode())));
            }
        });
        return directProductMap;
    }

    /**
     * 含税未税
     * @param entity
     * @return
     */
    private SalesDataEntity setAccount(SalesDataEntity entity) {
        BigDecimal decimal = new BigDecimal(1);
        Map<String, BigDecimal> map = this.redisTemplate.hasKey(ReturnOrderConstant.CACHE_PRODUCT_TAX_RATE) ? (Map<String, BigDecimal>) this.redisTemplate.opsForValue().get(ReturnOrderConstant.CACHE_PRODUCT_TAX_RATE) : null;
        String goodsCode = entity.getGoodsCode();
        if (!StringUtils.isEmpty(goodsCode)) {
            if (Objects.isNull(map)) {
                return entity;
            }
            BigDecimal bigDecimal = map.get(goodsCode);
            if (Objects.isNull(bigDecimal)){
                return entity;
            }
            BigDecimal amount = entity.getSalesAmount();
            BigDecimal amountNot = entity.getSalesAmountNoTax();
            if (Objects.nonNull(amount)){
                if (BigDecimal.ZERO.compareTo(amount) != 0) {
                    BigDecimal multiply = amount.divide(decimal.add(bigDecimal).setScale(4, BigDecimal.ROUND_HALF_UP),4,BigDecimal.ROUND_HALF_UP);
                    entity.setSalesAmountNoTax(multiply);
                }
            }
            if (Objects.nonNull(amountNot)){
                if (BigDecimal.ZERO.compareTo(amountNot) != 0) {
                    BigDecimal multiply = amountNot.multiply(decimal.add(bigDecimal).setScale(4, BigDecimal.ROUND_HALF_UP));
                    entity.setSalesAmount(multiply.setScale(4, BigDecimal.ROUND_HALF_UP));
                }
            }
            if ( Objects.nonNull(entity.getCurUnitSalesQuantity()) && BigDecimal.ZERO.compareTo(entity.getCurUnitSalesQuantity()) !=0){
                if(Objects.nonNull(entity.getSalesAmountNoTax()))
                {
                    entity.setSalesCostNot(entity.getSalesAmountNoTax().divide(entity.getCurUnitSalesQuantity(),4,BigDecimal.ROUND_HALF_UP));
                }
                if(Objects.nonNull(entity.getSalesAmount()))
                {
                    entity.setSalesCost(entity.getSalesAmount().divide(entity.getCurUnitSalesQuantity(),4,BigDecimal.ROUND_HALF_UP));
                }
            }

        }
        return entity;
    }
}
