package com.biz.crm.kms.business.invoice.stock.local.imports.service;

import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.kms.business.invoice.sdk.constant.DirectConstant;
import com.biz.crm.kms.business.invoice.stock.local.entity.Stock;
import com.biz.crm.kms.business.invoice.stock.local.imports.model.StockImportsVo;
import com.biz.crm.kms.business.invoice.stock.local.repository.StockRepository;
import com.biz.crm.kms.business.invoice.stock.local.service.StockService;
import com.biz.crm.kms.business.invoice.stock.local.util.DictValueUtil;
import com.biz.crm.kms.business.supermarket.sdk.service.SupermarketVoService;
import com.biz.crm.kms.business.supermarket.sdk.vo.SupermarketVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
@Component
public class StockImportsProcess implements ImportProcess<StockImportsVo> {

    @Resource
    private StockService stockService;

    @Resource
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private StockRepository stockRepository;
    
    @Autowired(required = false)
    private DictDataVoService dictDataVoService;

    @Autowired(required = false)
    private SupermarketVoService supermarketVoService;

    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, StockImportsVo> data,
                                        TaskGlobalParamsVo paramsVo,
                                        Map<String, Object> params) {
        Validate.notEmpty(data, "导入数据不能为空！");

        List<Stock> importList = this.validate(data);

        stockService.importSave(importList);
        return null;
    }

    /**
     * 校验并转换数据
     **/
    private List<Stock> validate(LinkedHashMap<Integer, StockImportsVo> data) {
        List<StockImportsVo> stockImportsVos = new ArrayList<>(data.values());
        Set<String> groupGoods = new HashSet<>();
        Set<String> supermarketnames = stockImportsVos.stream().map(StockImportsVo::getKaName).collect(Collectors.toSet());
        List<SupermarketVo> supermarketVos = this.supermarketVoService.findBySupermaketNames(supermarketnames);
        Map<String ,SupermarketVo> supermarketMaps = supermarketVos.stream().collect(Collectors.toMap(SupermarketVo::getSupermarketName, Function.identity()));
        List<Stock> entities = new ArrayList<>();
        stockImportsVos.forEach(importsVo -> {
            Stock entity = this.nebulaToolkitService.copyObjectByWhiteList(importsVo,Stock.class,HashSet.class,ArrayList.class);
            entity.setKaCode(supermarketMaps.get(entity.getKaName()).getSupermarketCode());
            //库存数据
            this.amountTrans(importsVo,entity);
            entity.setGroupGoods(entity.getKaCode()+"-"+entity.getDirectCode()+"-"+entity.getKaStoreCode()+"-"+entity.getKaGoodsCode());
            entities.add(entity);
            groupGoods.add(entity.getGroupGoods());
        });
        this.adjustData(entities);
        //查询已有库存
        List<Stock> stockList = this.stockRepository.findStockListByGroupGoods(TenantUtils.getTenantCode(), groupGoods);
        if (!CollectionUtils.isEmpty(stockList)){
            Map<String, String> stockMap = stockList.stream().collect(Collectors.toMap(Stock::getGroupGoods, Stock::getId, (a, b) -> a));
            entities.forEach(x->{
                if (stockMap.containsKey(x.getGroupGoods())){
                    x.setId(stockMap.get(x.getGroupGoods()));
                }
            });
        }
        return entities;
    }

    /**
     * 转换库存的金额，数量
     * @param vo
     * @param entity
     */
    public void amountTrans(StockImportsVo vo,Stock entity){
        Validate.isTrue(StringUtils.isNotBlank(vo.getInventoryDate()),"库存时间不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getDirectCode()),"系统编码不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getKaName()),"系统名称不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getBusinessFormatCode()),"业态不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getBusinessUnitCode()),"业务单元不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getSoldToPartyCode()),"售达方编码不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getSoldToPartyName()),"售达方名称不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getDeliveryPartyCode()),"送达方编码不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getDeliveryPartyName()),"送达方名称不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getKaStoreCode()),"门店编码不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getKaStoreName()),"门店名称不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getKaGoodsCode()),"零售商产品编码不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getKaGoodsName()),"零售商产品名称不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getProductCode()),"产品编码不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getProductName()),"产品名称不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getCustomerInventoryStr()),"客户库存数不能为空");
        Validate.isTrue(StringUtils.isNotBlank(vo.getInventoryAmountStr()),"库存金额不能为空");
        if (StringUtils.isNotBlank(vo.getCustomerInventoryStr())){
            BigDecimal customerInventory = new BigDecimal(vo.getCustomerInventoryStr());
            entity.setCustomerInventory(customerInventory);
        }
        if (StringUtils.isNotBlank(vo.getInTransitAmountStr())) {
            BigDecimal inTransitAmount = new BigDecimal(vo.getInTransitAmountStr());
            entity.setInTransitAmount(inTransitAmount);
        }
        if (StringUtils.isNotBlank(vo.getInventoryAmountStr())){
            BigDecimal inventoryAmount = new BigDecimal(vo.getInventoryAmountStr());
            entity.setInventoryAmount(inventoryAmount);
        }
        if (StringUtils.isNotBlank(vo.getInTransitInventoryStr())){
            BigDecimal inTransitInventory = new BigDecimal(vo.getInTransitInventoryStr());
            entity.setInTransitInventory(inTransitInventory);
        }
    }

    /**
     * 调整数据
     *
     * @param data 待调整数据
     * @return 调整后数据
     */
    private List<Stock> adjustData(List<Stock> data) {
        if (CollectionUtils.isEmpty(data)) {
            return new ArrayList<>();
        }
        Map<String, List<DictDataVo>> fomatMap = this.dictDataVoService.findByDictTypeCodeList(Lists.newArrayList(DirectConstant.MDM_BUSINESS_FORMAT));
        Map<String, List<DictDataVo>> unitMap = this.dictDataVoService.findByDictTypeCodeList(Lists.newArrayList(DirectConstant.MDM_BUSINESS_UNIT));
        for (Stock entity :data)
        {
            entity.setBusinessFormatCode(DictValueUtil.findDictValue(fomatMap, DirectConstant.MDM_BUSINESS_FORMAT, entity.getBusinessFormatCode()));
            entity.setBusinessUnitCode(DictValueUtil.findDictValue(unitMap, DirectConstant.MDM_BUSINESS_UNIT, entity.getBusinessUnitCode()));

        }

        return data;
    }

    @Override
    public Class<StockImportsVo> findCrmExcelVoClass() {
        return StockImportsVo.class;
    }

    @Override
    public String getTemplateCode() {
        return "KMS_STOCK_PLAN_IMPORT";
    }

    @Override
    public String getTemplateName() {
        return "KMS库存数据导入";
    }
}
