package com.biz.crm.kms.business.invoice.stock.local.service.internal;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.kms.business.direct.product.sdk.dto.DirectProductDto;
import com.biz.crm.kms.business.direct.product.sdk.service.DirectProductVoService;
import com.biz.crm.kms.business.direct.product.sdk.vo.DirectProductVo;
import com.biz.crm.kms.business.direct.sdk.service.DirectVoService;
import com.biz.crm.kms.business.direct.sdk.vo.DirectVo;
import com.biz.crm.kms.business.direct.store.sdk.dto.DirectStoreConditionDto;
import com.biz.crm.kms.business.direct.store.sdk.service.DirectStoreVoService;
import com.biz.crm.kms.business.direct.store.sdk.vo.DirectStoreVo;
import com.biz.crm.kms.business.invoice.sdk.constant.DirectConstant;
import com.biz.crm.kms.business.invoice.sdk.enums.AutoConvertsEnum;
import com.biz.crm.kms.business.invoice.sdk.enums.ConstantEnums;
import com.biz.crm.kms.business.invoice.sdk.enums.InvoicesStatus;
import com.biz.crm.kms.business.invoice.stock.local.register.InvoiceStockTypeRegister;
import com.biz.crm.kms.business.invoice.stock.local.repository.StockRepository;
import com.biz.crm.kms.business.invoice.stock.sdk.constant.StockConstant;
import com.biz.crm.kms.business.invoice.sdk.enums.GrabTransStatus;
import com.biz.crm.kms.business.invoice.stock.local.entity.KmsInvoiceStockGrab;
import com.biz.crm.kms.business.invoice.stock.local.entity.Stock;
import com.biz.crm.kms.business.invoice.stock.local.repository.KmsInvoiceStockGrabRepository;
import com.biz.crm.kms.business.invoice.stock.local.service.StockGrabsService;
import com.biz.crm.kms.business.invoice.stock.local.service.StockGrabsThreadService;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mn.common.base.service.RedisCrmService;
import com.biz.crm.mn.common.rocketmq.service.RocketMqProducer;
import com.biz.crm.mn.common.rocketmq.util.RocketMqUtil;
import com.biz.crm.mn.common.rocketmq.vo.MqMessageVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.security.core.parameters.P;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;


/**
 * 库存数据抓取数据(python)表服务实现类
 */
@Slf4j
@Service("stockGrabsService")
public class StockGrabsServiceImpl implements StockGrabsService {

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private RocketMqProducer rocketMqProducer;

    @Autowired(required = false)
    private DirectStoreVoService directStoreVoService;

    @Autowired(required = false)
    private RedisCrmService redisCrmService;

    @Autowired(required = false)
    private RedisMutexService redisMutexService;

    @Autowired(required = false)
    private DirectProductVoService directProductVoService;

    @Autowired(required = false)
    private StockRepository stockRepository;

    @Autowired(required = false)
    private InvoiceStockTypeRegister invoiceStockTypeRegister;
    
    @Autowired(required = false)
    private KmsInvoiceStockGrabRepository stockGrabRepository;

    @Autowired(required = false)
    private StockGrabsThreadService stockGrabsThreadService;

    @Autowired(required = false)
    private DirectVoService directVoService;

    @Autowired(required = false)
    private DictDataVoService dictDataVoService;

    /**
     * 库存数据未转换单据自动转换定时任务
     */
    @Override
    public void autoNotConvertAcceptanceOrder() {
        //库存数据自动转换锁定时间
        log.info("=====>    库存数据未转换单据自动转换 start   <=====");
        boolean lock = redisMutexService.tryLock(StockConstant.STOCK_AUTO_NOT_CONVERT_LOCK, TimeUnit.SECONDS, StockConstant.STOCK_REDIS_TIME);
        if (!lock) {
            log.warn("库存数据未转换单据自动转换中,本次自动转换被忽略!");
            log.info("=====>    库存数据未转换单据自动转换 end   <=====");
            return;
        }
        Pageable pageable = PageRequest.of(1,StockConstant.STOCK_PAGE_SIZE);
        List<String> transStatusList = new ArrayList<>();
        transStatusList.add(GrabTransStatus.NOT_TRANS.getDictCode());
        String day = this.getDays(AutoConvertsEnum.NOT_CONVERTS.getDictCode());
        Set<String> currentOrderNumberSet = new HashSet<>(StockConstant.STOCK_PAGE_SIZE);
        try {
            //查询待转换库存数据信息
            Page<String> orderNumberPage = this.stockGrabRepository.findStockGrabPage(pageable, TenantUtils.getTenantCode(), transStatusList,day);
            currentOrderNumberSet.addAll(orderNumberPage.getRecords());
            this.sendMqMessage(orderNumberPage.getRecords());
            while (orderNumberPage.hasNext()
                    && StockConstant.STOCK_LOOP_MAX >= pageable.getPageNumber()) {
                pageable = pageable.next();
                orderNumberPage = this.stockGrabRepository.findStockGrabPage(pageable, TenantUtils.getTenantCode(), transStatusList,day);
                currentOrderNumberSet.clear();
                currentOrderNumberSet.addAll(orderNumberPage.getRecords());
                this.sendMqMessage(orderNumberPage.getRecords());
            }
        } catch (Exception e) {
            log.error("", e);
            if (CollectionUtil.isNotEmpty(currentOrderNumberSet)) {
                //异常时,删除当前加锁redis
                redisCrmService.hdel(StockConstant.STOCK_CONVERT_LOCK, currentOrderNumberSet.toArray());
            }
        } finally {
            redisMutexService.unlock(StockConstant.STOCK_AUTO_NOT_CONVERT_LOCK);
        }
        log.info("=====>    库存数据未转换单据自动转换 end   <=====");
    }

    /**
     * 库存数据转换失败单据自动转换定时任务
     */
    @Override
    public void autoFailConvertAcceptanceOrder() {
        //库存数据自动转换锁定时间
        log.info("=====>    库存数据转换失败单据自动转换 start   <=====");
        boolean lock = redisMutexService.tryLock(StockConstant.STOCK_AUTO_FAIL_CONVERT_LOCK, TimeUnit.SECONDS, StockConstant.STOCK_REDIS_TIME);
        if (!lock) {
            log.warn("库存数据转换失败单据自动转换中,本次自动转换被忽略!");
            log.info("=====>    库存数据转换失败单据自动转换 end   <=====");
            return;
        }
        Pageable pageable = Pageable.ofSize(StockConstant.STOCK_PAGE_SIZE);
        List<String> transStatusList = new ArrayList<>();
        transStatusList.add(GrabTransStatus.NOT_TRANS.getDictCode());
        String day = this.getDays(AutoConvertsEnum.FAIL_CONVERTS.getDictCode());
        Set<String> currentOrderNumberSet = new HashSet<>(StockConstant.STOCK_PAGE_SIZE);
        try {
            //查询待转换库存数据信息
            Page<String> orderNumberPage = this.stockGrabRepository.findStockGrabPage(pageable, TenantUtils.getTenantCode(), transStatusList,day);
            currentOrderNumberSet.addAll(orderNumberPage.getRecords());
            this.sendMqMessage(orderNumberPage.getRecords());
            while (orderNumberPage.hasNext()
                    && StockConstant.STOCK_LOOP_MAX >= pageable.getPageNumber()) {
                pageable = pageable.next();
                orderNumberPage = this.stockGrabRepository.findStockGrabPage(pageable, TenantUtils.getTenantCode(), transStatusList,day);
                currentOrderNumberSet.clear();
                currentOrderNumberSet.addAll(orderNumberPage.getRecords());
                this.sendMqMessage(orderNumberPage.getRecords());
            }
        } catch (Exception e) {
            log.error("", e);
            if (CollectionUtil.isNotEmpty(currentOrderNumberSet)) {
                //异常时,删除当前加锁redis
                redisCrmService.hdel(StockConstant.STOCK_CONVERT_LOCK, currentOrderNumberSet.toArray());
            }
        } finally {
            redisMutexService.unlock(StockConstant.STOCK_AUTO_FAIL_CONVERT_LOCK);
        }
        log.info("=====>    库存数据转换失败单据自动转换 end   <=====");
    }

    /**
     * 自动转单库存数据发送 MQ消息
     *
     * @param idList 库存数据集合
     */
    private void sendMqMessage(List<String> idList) {
        if (CollectionUtils.isEmpty(idList)) {
            return;
        }
        //过滤单号为空的数据
        idList = idList.stream()
                .filter(StringUtils::isNotBlank)
                .distinct()
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(idList)) {
            log.error("库存数据原始单据有空数据[id]存在!");
            return;
        }
        //过滤正在转换的库存数据号
        this.filterSwitchIngidList(idList);
        if (CollectionUtil.isEmpty(idList)) {
            return;
        }

        //发送MQ消息开始转换单据
        MqMessageVo message = new MqMessageVo();
        message.setMsgBody(JSON.toJSONString(idList));
        message.setTopic(StockConstant.KMS_STOCK_TOPIC + RocketMqUtil.mqEnvironment());
        message.setTag(StockConstant.STOCK_MESSAGE_TAG);
        rocketMqProducer.sendMqMsg(message);
        try {
            //单位：毫秒 防止MQ消息发送过于频繁
            Thread.sleep(200);
        } catch (Exception e) {
            log.error("", e);
        }
    }


    /**
     * 根据数据id 转换单据信息
     *
     * @param idList 数据id
     * @return void
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.15 13:42
     */
    @Override
    public void manualSwitch(List<String> idList) {
        if (CollectionUtil.isEmpty(idList)) {
            log.warn("未接收到需转换库存数据号！");
            return;
        }
        //组装单据 根据数据id  拼接数据
        List<Stock> orderList = this.buildOrder(idList);
        //转换单据新
        this.convertOrder(orderList);
    }

    /**
     * 组装单据
     *
     * @param idList
     * @return
     */
    private List<Stock> buildOrder(List<String> idList) {
        log.info("--  组装单据  --");
        if (CollectionUtil.isEmpty(idList)) {
            return Collections.emptyList();
        }
        List<KmsInvoiceStockGrab> entityGrabList = this.stockGrabRepository.findByidList(TenantUtils.getTenantCode(), idList);
        if (CollectionUtil.isEmpty(entityGrabList)) {
            log.warn("不存在或已删除！");
            return Collections.emptyList();
        }
        //默认版本号  很重要  很重要  很重要
        int defaultVersionNumber = 1;

        // 设置基础字段和版本号
        entityGrabList.forEach(entity -> {
            entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            if (Objects.isNull(entity.getVersionNumber())) {
                entity.setVersionNumber(defaultVersionNumber);
            }
        });
        //单据转换
        List<Stock> entityList = (List<Stock>) this.nebulaToolkitService.copyCollectionByBlankList(entityGrabList, KmsInvoiceStockGrab.class,
                Stock.class, HashSet.class, ArrayList.class);

        //获取最新大版本号数据
        Map<String, Stock> entityMap = entityList.stream().collect(
                Collectors.toMap(Stock::getId,
                        Function.identity(),
                        (a, b) -> a.getVersionNumber() > b.getVersionNumber() ? a : b));
        //拼接商品数据
        List<Stock> convertData = new ArrayList<>(entityMap.values());
        return convertData;
    }

    /**
     * 库存数据商超门店和商超产品转换
     *
     * @param entityList
     * @return void
     * @author: zs
     * @version: v1.0.0
     * @date: 2022.12.14 14:58
     */
    @Override
    public void convertOrder(List<Stock> entityList) {
        log.info("--  单据所需数据查询  --");
        if (CollectionUtils.isEmpty(entityList)) {
            return;
        }
        entityList = entityList.stream()
                .filter(k -> StringUtils.isNotBlank(k.getId())).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(entityList)) {
            return;
        }
        //系统编码集合
        Set<String> directCodeSet = Sets.newHashSet();
        //需转换的ID编码集合
        List<String> idList = Lists.newArrayList();
        //需转换的门店编码集合
        Set<String> kaStoreCodes = Sets.newHashSet();
        //需转换的商品编码集合
        Set<String> kaGoodsCodes = Sets.newHashSet();
        //把商城编码+系统编码+门店编码+产品编码作为唯一键
        Set<String> groupGoods = Sets.newHashSet();

        //需转换的终端编码集合（终端业务员）和产品集合
        entityList.forEach(entity -> {
            //初始化单据状态
            entity.setOrderStatus(InvoicesStatus.S101.getDictCode());
            if (StringUtils.isNotEmpty(entity.getKaStoreCode())) {
                kaStoreCodes.add(entity.getKaStoreCode());
            }
            if (StringUtils.isNotEmpty(entity.getDirectCode())) {
                directCodeSet.add(entity.getDirectCode());
            }
            idList.add(entity.getId());
            if (StringUtils.isNotBlank(entity.getKaGoodsCode())){
                kaGoodsCodes.add(entity.getKaGoodsCode());
            }
            groupGoods.add(entity.getKaCode()+"-"+
                    entity.getDirectCode()+"-"+
                    entity.getKaStoreCode()+"-"+
                    entity.getKaGoodsCode());
        });
        //已转换过的单据信息更新到最新的单据上
        entityList = this.updateOldDataToNewData(groupGoods, entityList);
        if (CollectionUtils.isEmpty(entityList)) {
            return;
        }
        List<DirectVo> directVos = this.directVoService.findByDirectCodes(directCodeSet.stream().collect(Collectors.toList()));
        //获取系统数据
        Map<String, List<DirectVo>> directMap = Maps.newHashMap();
        if (!CollectionUtils.isEmpty(directVos)){
            directMap.putAll(directVos.stream()
                    .filter(k ->StringUtils.isNotBlank(k.getBusinessUnitCode()) && StringUtils.isNotBlank(k.getBusinessFormatCode()))
                    .collect(Collectors.groupingBy(DirectVo::getDirectCode)));
        }
        //门店map 系统编码+商超门店编码 构成唯一
        Map<String, List<DirectStoreVo>> directStoreMap = this.buildStoreInfo(directCodeSet, kaStoreCodes);
        Set<String> deliveryPartyCodeSet = Sets.newHashSet();
        if (CollectionUtil.isNotEmpty(directStoreMap)) {
            directStoreMap.values().stream()
                    .filter(CollectionUtil::isNotEmpty)
                    .forEach(directStoreVoList -> {
                        directStoreVoList.forEach(entity -> {
                            deliveryPartyCodeSet.add(entity.getTerminalCode());
                            if (StringUtils.isNotBlank(entity.getRetailerTerminalCode())){
                                deliveryPartyCodeSet.add(entity.getRetailerTerminalCode());
                            }
                        });
                    });
        }
        //商品map 系统编码+送达方编码+商超产品编码 构成唯一
        Map<String, List<DirectProductVo>> directProductMap = this.buildProductInfo(directCodeSet, deliveryPartyCodeSet, kaGoodsCodes);

        //异步线程进行组装数据，并且保存
        //注：异步线程的方法不能和调用方在同一个类中，否则不会走线程池
        List<List<Stock>> entityGroupList = Lists.partition(entityList, ConstantEnums.THREAD_SIZE.getValue());

        entityGroupList.stream()
                .filter(CollectionUtil::isNotEmpty)
                .forEach(asyncList -> {
                    stockGrabsThreadService.saveStockData(asyncList, directMap, directStoreMap, directProductMap);
                });

    }

    /**
     * 构建门店信息
     * <p>
     * 门店map 系统编码+商超门店编码 构成唯一
     *
     * @param directCodeSet 系统编码集合
     * @param kaStoreCodes  商超门店编码集合
     */
    private Map<String, List<DirectStoreVo>> buildStoreInfo(Set<String> directCodeSet, Set<String> kaStoreCodes) {
        Map<String, List<DirectStoreVo>> directStoreMap = Maps.newHashMap();
        if (CollectionUtil.isEmpty(kaStoreCodes)
                || CollectionUtil.isEmpty(directCodeSet)) {
            return directStoreMap;
        }
        DirectStoreConditionDto conditionDto = new DirectStoreConditionDto();
        conditionDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        conditionDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        conditionDto.setSupermarketStoreCodes(kaStoreCodes);
        conditionDto.setDirectCodes(directCodeSet);
        List<DirectStoreVo> directStoreList = this.directStoreVoService.findByDirectStoreConditionDto(conditionDto);
        if (CollectionUtil.isNotEmpty(directStoreList)) {
            directStoreMap.putAll(directStoreList.stream()
                    .filter(k -> StringUtils.isNotBlank(k.getDirectCode()))
                    .filter(k -> StringUtils.isNotBlank(k.getSupermarketStoreCode()))
                    .collect(Collectors.groupingBy(k -> k.getDirectCode()
                                    + "_" + k.getSupermarketStoreCode())));
        }
        return directStoreMap;
    }

    /**
     * 构建商品信息
     * <p>
     * 门店map 系统编码+商超门店编码 构成唯一
     *
     * @param directCodeSet        系统编码集合
     * @param deliveryPartyCodeSet 送达方编码集合
     * @param kaGoodsCodes         商超商品编码集合
     */
    private Map<String, List<DirectProductVo>> buildProductInfo(Set<String> directCodeSet,
                                                                Set<String> deliveryPartyCodeSet,
                                                                Set<String> kaGoodsCodes) {
        Map<String, List<DirectProductVo>> directProductMap = Maps.newHashMap();
        if (CollectionUtil.isEmpty(directCodeSet)
                || CollectionUtil.isEmpty(deliveryPartyCodeSet)
                || CollectionUtil.isEmpty(kaGoodsCodes)) {
            return directProductMap;
        }

        List<List<String>> kaProductCodeGroupList = Lists.partition(Lists.newArrayList(kaGoodsCodes), ConstantEnums.PRODUCT_SIZE.getValue());
        DirectProductDto directProductDto = new DirectProductDto();
        directProductDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        directProductDto.setTenantCode(TenantUtils.getTenantCode());
        directProductDto.setOnShelfStatus(BooleanEnum.TRUE.getCapital());
        directProductDto.setDirectCodes(new ArrayList<>(directCodeSet));
        directProductDto.setDeliveryPartyCodes(new ArrayList<>(deliveryPartyCodeSet));
        directProductDto.setOrderType(invoiceStockTypeRegister.type());
        kaProductCodeGroupList.forEach(kaProductCodeList -> {
            directProductDto.setKaProductCodes(kaProductCodeList);
            List<DirectProductVo> productList = this.directProductVoService.findByDirectProductDto(directProductDto);
            if (CollectionUtil.isNotEmpty(productList)) {
                directProductMap.putAll(productList.stream()
                        .filter(k -> BooleanEnum.TRUE.getCapital().equals(k.getOnShelfStatus()))
                        .filter(k -> StringUtils.isNotBlank(k.getDirectCode()))
                        .filter(k -> StringUtils.isNotBlank(k.getDeliveryPartyCode()))
                        .filter(k -> StringUtils.isNotBlank(k.getKaProductCode()))
                        .collect(Collectors.groupingBy(k -> k.getDirectCode()
                                + "_" + k.getDeliveryPartyCode()
                                + "_" + k.getKaProductCode())));
            }
        });
        return directProductMap;
    }


    /**
     * 筛选出
     * 并过滤当前状态不可转换的单据
     *
     * @param groupGoods
     * @param entityList
     */
    private List<Stock> updateOldDataToNewData(Set<String> groupGoods, List<Stock> entityList) {
        if (CollectionUtil.isEmpty(groupGoods)
                || CollectionUtil.isEmpty(entityList)) {
            return entityList;
        }
        //拼接唯一键
        entityList.forEach(aa ->{
            aa.setGroupGoods(aa.getKaCode()+"-"+aa.getDirectCode()+"-"+aa.getKaStoreCode()+"-"+aa.getKaGoodsCode());
        });
        List<Stock> newEntityList = new ArrayList<>();
        //已转换的库存数据数据
        List<Stock> oldEntityList = this.stockRepository.findStockListByGroupGoods(TenantUtils.getTenantCode(), groupGoods);
        if (!CollectionUtils.isEmpty(oldEntityList)){
            List<String> oldGroupGoods = oldEntityList.stream().map(Stock::getGroupGoods).collect(Collectors.toList());
            List<String> newGroupGoodss = groupGoods.stream().filter(f ->!oldGroupGoods.contains(f)).collect(Collectors.toList());
            //筛选存在的库存信息。
            Map<String, List<Stock>> newStockMap = entityList.stream().filter(f ->oldGroupGoods.contains(f.getGroupGoods())).collect(Collectors.groupingBy(Stock::getGroupGoods));
            for (Stock entity : oldEntityList){
                //以时间进行排序,拿到时间最近的一条
                newStockMap.get(entity.getGroupGoods()).sort((t1,t2)-> t2.getCreateTime().compareTo(t1.getCreateTime()));
                Stock stock = newStockMap.get(entity.getGroupGoods()).get(0);
                //拿到所有的id集合
                List<String> idList = newStockMap.get(entity.getGroupGoods()).stream().map(Stock::getId).collect(Collectors.toList());
                //若存在时间比当前库存数据更新的数据，则删除原数据
                if (entity.getCreateTime().compareTo(stock.getCreateTime()) == -1){
                    idList = idList.stream().filter(id ->stock.getId().equals(id)).collect(Collectors.toList());
                    this.stockRepository.removeById(entity.getId());
                    newEntityList.add(stock);
                }
                //如果为手动抓单，创建时间则相同
                if (entity.getCreateTime().compareTo(stock.getCreateTime()) == 0){
                    this.stockRepository.removeById(entity.getId());
                    newEntityList.add(stock);
                }
                //转换唯一键相同的时间不是最新的库存原始数据状态
                this.stockGrabRepository.updateTransStatusByIds(TenantUtils.getTenantCode(),GrabTransStatus.HAVE_TRANS,idList);
            }
            //保存最新的库存数据
            if (CollectionUtils.isEmpty(newGroupGoodss)){
                Map<String, List<Stock>> newStockMapTwo = entityList.stream().filter(f ->!oldGroupGoods.contains(f.getGroupGoods())).collect(Collectors.groupingBy(Stock::getGroupGoods));
                newGroupGoodss.forEach(aa ->{
                    newStockMapTwo.get(aa).sort((t1,t2)-> t2.getCreateTime().compareTo(t1.getCreateTime()));
                    Stock stock = newStockMapTwo.get(aa).get(0);
                    newEntityList.add(stock);
                });
            }
        } else{
            //如果没有旧的库存唯一键，则直接新增
            Map<String,List<Stock>> newStockMap = entityList.stream().collect(Collectors.groupingBy(Stock::getGroupGoods));
            groupGoods.forEach(bb ->{
                newStockMap.get(bb).sort((t1,t2)-> t2.getCreateTime().compareTo(t1.getCreateTime()));
                Stock stock = newStockMap.get(bb).get(0);
                newEntityList.add(stock);
            });
        }

        return newEntityList;
    }

    /**
     * 过滤正在转换的库存数据集合
     *
     * @param groupGoods 库存数据集合
     * @return void
     * @author: zs
     * @version: v1.0.0
     * @date: 2022.12.14 1:45
     */
    @Override
    public void filterSwitchIngidList(List<String> groupGoods) {
        if (CollectionUtil.isEmpty(groupGoods)) {
            return;
        }
        //获取redis已存在key
        List<Object> updateList = redisCrmService.hmget(StockConstant.STOCK_CONVERT_LOCK, new HashSet<>(groupGoods));

        if (CollectionUtil.isNotEmpty(updateList)) {
            groupGoods.removeAll(updateList.stream()
                    .filter(Objects::nonNull)
                    .filter(k -> StringUtils.isNotEmpty(k.toString()))
                    .map(Object::toString)
                    .collect(Collectors.toList()));
        }

        Map<String, String> redisMap = groupGoods.stream()
                .collect(Collectors.toMap(key -> key, key -> key, (oldValue, newValue) -> newValue));

        //当前执行效率很高,锁定30分钟即可
        redisCrmService.hmset(StockConstant.STOCK_CONVERT_LOCK, redisMap, TimeUnit.SECONDS, StockConstant.STOCK_REDIS_TIME);
    }

    /**
     * 获取数据字典中设置的转换时间
     * @param type
     * @return
     */
    private String getDays(String type){
        Map<String, List<DictDataVo>> fomatMap = this.dictDataVoService.findByDictTypeCodeList(Lists.newArrayList(DirectConstant.KMS_AUTO_CONVERTS));
        final List<DictDataVo> vos = fomatMap.get(DirectConstant.KMS_AUTO_CONVERTS);
        final Optional<String> first =
                vos.stream()
                        .filter(a -> a.getDictCode().equals(type))
                        .map(DictDataVo::getDictValue)
                        .findFirst();
        int past = Integer.parseInt(first.orElse(StringUtils.EMPTY));
        //获取数据字典中设置的时间
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.DATE, calendar.get(Calendar.DATE) - past);
        String day = sdf.format(calendar.getTime());
        return day;
    }
}
