package com.biz.crm.kms.business.invoice.stock.local.service.internal;

import cn.hutool.core.collection.CollectionUtil;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.kms.business.direct.product.sdk.vo.DirectProductVo;
import com.biz.crm.kms.business.direct.sdk.vo.DirectVo;
import com.biz.crm.kms.business.direct.store.sdk.vo.DirectStoreVo;
import com.biz.crm.kms.business.invoice.sdk.enums.GrabTransStatus;
import com.biz.crm.kms.business.invoice.sdk.enums.InvoicesStatus;
import com.biz.crm.kms.business.invoice.stock.local.entity.Stock;
import com.biz.crm.kms.business.invoice.stock.local.repository.KmsInvoiceStockGrabRepository;
import com.biz.crm.kms.business.invoice.stock.local.repository.StockRepository;
import com.biz.crm.kms.business.invoice.stock.local.service.StockGrabsThreadService;
import com.biz.crm.kms.business.invoice.stock.sdk.constant.StockConstant;
import com.biz.crm.kms.business.invoice.stock.sdk.enums.StockStatus;
import com.biz.crm.mn.common.base.service.RedisCrmService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @Title: SalesDataGrabsThreadServiceImpl
 * @Description:
 * @Author: cyj
 * @Date: 2022/11/25 - 16:46 PM
 */
@Slf4j
@Service("stockGrabsThreadService")
public class StockGrabsThreadServiceImpl implements StockGrabsThreadService {

    @Autowired
    private RedisService redisService;

    @Autowired
    private StockRepository stockRepository;

    @Autowired
    private KmsInvoiceStockGrabRepository stockGrabRepository;

    @Autowired
    private RedisCrmService redisCrmService;

    @Override
    @Async("grab3Thread")
    @Transactional(propagation = Propagation.NOT_SUPPORTED, rollbackFor = Exception.class)
    public void saveStockData(List<Stock> entities, Map<String, List<DirectVo>> directMap, Map<String, List<DirectStoreVo>> directStoreMap, Map<String, List<DirectProductVo>> directProductMap) {
        log.info("--  线程池存储单据 --");
        if (CollectionUtil.isEmpty(entities)) {
            return;
        }
        List<String> ids = entities.stream().map(Stock::getId).collect(Collectors.toList());
        Set<String> idsSet = new HashSet<>();
        Set<String> idsSuccessSet = new HashSet<>();
        entities.stream()
                .filter(Objects::nonNull)
                .filter(k -> StringUtils.isNotBlank(k.getId()))
                .forEach(entity -> {
                    try {
                        //清除数据中的售达方，企业门店，区域等数据 第一步
                        this.clearTransformInfoOne(entity,directMap);
                        //凭借门店中库存数据  第二步
                        String deliveryCode = this.buildOrderInfoTwo(entity, directStoreMap);
                        //拼接产品中库存数据  第三步
                        this.convertAcceptanceOrderItemThree(entity, directProductMap, deliveryCode);
                        //设置状态  第四步
                        this.setOrderStatusAndMsgFour(entity);
                        this.stockRepository.saveOrUpdate(entity);
                        if (Objects.equals(InvoicesStatus.S200.getDictCode(),entity.getOrderStatus())){
                            idsSuccessSet.add(entity.getId());
                        }else {
                            idsSet.add(entity.getId());
                        }
                    } catch (Exception e) {
                        log.error("保存库存数据数据失败", e);
                        entity.setOrderStatus(InvoicesStatus.S100.getDictCode());
                        entity.setTransFlag(BooleanEnum.TRUE.getCapital());
                        entity.setOrderStatusMsg("转换时异常!");
                        try {
                            this.stockRepository.saveOrUpdate(entity);
                        } catch (Exception ex) {
                            log.error("", ex);
                        }
                    }

                });


        if (CollectionUtil.isEmpty(idsSet) && CollectionUtils.isEmpty(idsSuccessSet)) {
            return;
        }
        //处理数据前先进行解锁
        redisCrmService.hdel(StockConstant.STOCK_CONVERT_LOCK, ids);
        try {
            if (!CollectionUtil.isEmpty(idsSet)){
                stockGrabRepository.updateTransStatusByIds(TenantUtils.getTenantCode(),
                        GrabTransStatus.TRANS_FAIL, new ArrayList<>(idsSet));
            }
            if (!CollectionUtil.isEmpty(idsSuccessSet)){
                stockGrabRepository.updateTransStatusByIds(TenantUtils.getTenantCode(),
                        GrabTransStatus.TRANS_FAIL, new ArrayList<>(idsSuccessSet));
            }

        } catch (Exception e) {
            log.error("", e);
        }
    }

    /**
     * @Description: 第一步：清楚数据中的售达方、门店、销售区域
     * @Param: [entity]
     * @return: void
     * @Author: ZS
     * @Date: 2022/12/16
     */
    public void clearTransformInfoOne(Stock entity,Map<String, List<DirectVo>> directMap) {
        entity.setSoldToPartyName(null);
        entity.setSoldToPartyCode(null);
        entity.setDeliveryPartyCode(null);
        entity.setDeliveryPartyName(null);
        entity.setAreaName(null);
        entity.setAreaCode(null);
        //单据时间转换
        if (StringUtils.isNotBlank(entity.getInventoryDate())){
            try {
                entity.setOrderTime(new SimpleDateFormat("yyyy-MM-dd").parse(entity.getInventoryDate()));
            } catch (ParseException e) {
                e.printStackTrace();
            }
        }
        //业态业务单元赋值
        List<DirectVo> directVos = directMap.get(entity.getDirectCode());
        if (CollectionUtils.isEmpty(directVos)){
            entity.setOrderStatusMsg("不存在该系统");
        }else {
            DirectVo directVo = directVos.get(0);
            entity.setBusinessFormatCode(directVo.getBusinessFormatCode());
            entity.setBusinessUnitCode(directVo.getBusinessUnitCode());
            entity.setKaName(directVo.getSupermarketName());
        }
    }

    /**
     * @Description:第二步： 拼接门店数据
     * @Param: [entity, directStoreMap]
     * @return: void
     * @Author: ZS
     * @Date: 2022/12/16
     */
    public String buildOrderInfoTwo(Stock entity, Map<String, List<DirectStoreVo>> directStoreMap) {
        entity.setOrderStatus(StockStatus.S100.getDictCode());
        entity.setTransFlag(BooleanEnum.TRUE.getCapital());
        if (StringUtils.isBlank(entity.getDirectCode())) {
            entity.setOrderStatusMsg("系统编码为空");
            return null;
        }
        if (StringUtils.isBlank(entity.getKaStoreCode())) {
            entity.setOrderStatusMsg("单据商超门店编码为空");
            return null;
        }
        List<DirectStoreVo> directStoreList = directStoreMap.get(entity.getDirectCode() + "_" + entity.getKaStoreCode());
        DirectStoreVo directStoreVo = null;
        if (CollectionUtil.isEmpty(directStoreList)) {
            entity.setOrderStatusMsg("未查询到与企业门店对应关系");
            return null;
        } else if (directStoreList.size() > 1) {
            entity.setOrderStatusMsg("商超门店与企业门店对应关系存在[" + directStoreList.size() + "]条");
            return null;
        } else {
            directStoreVo = directStoreList.get(0);
        }
        entity.setKaStoreCode(directStoreVo.getSupermarketStoreCode());
        if(StringUtils.isEmpty(entity.getKaStoreName())){
            entity.setKaStoreName(directStoreVo.getSupermarketStoreName());
        }
        //售达方信息
        entity.setSoldToPartyCode(directStoreVo.getSoldToPartyCode());
        entity.setSoldToPartyName(directStoreVo.getSoldToPartyName());
        entity.setDeliveryPartyCode(directStoreVo.getTerminalCode());
        entity.setDeliveryPartyName(directStoreVo.getTerminalName());
        entity.setAreaName(directStoreVo.getRetailerRegion());
        entity.setBusinessArea(directStoreVo.getBusinessArea());
        entity.setProvinceCode(directStoreVo.getProvinceCode());
        entity.setProvinceName(directStoreVo.getProvinceName());
        return directStoreVo.getRetailerTerminalCode();
    }

    /**
     * @Description: 第三步：拼接产品数据
     * @Param: [entity, directProductMap]
     * @return: void
     * @Author: ZS
     * @Date: 2022/12/16
     */
    public void convertAcceptanceOrderItemThree(Stock entity, Map<String, List<DirectProductVo>> directProductMap, String deliveryCode) {
        entity.setOrderStatus(StockStatus.S100.getDictCode());
        if (StringUtils.isBlank(entity.getKaGoodsCode())) {
            entity.setOrderStatusMsg("商品编码为空");
            return;
        }
        DirectProductVo productVo = null;
        List<DirectProductVo> productVoList = new ArrayList<>();
        List<DirectProductVo> productVoLists = directProductMap.get(entity.getDirectCode() + "_" +
                entity.getDeliveryPartyCode() + "_" + entity.getKaGoodsCode());
        if (CollectionUtil.isNotEmpty(productVoLists)){
            productVoList.addAll(productVoLists);
        }
        //如果送达方编码没有找到，那么会通过关联的送达方编码去查
        if (CollectionUtil.isEmpty(productVoList) && StringUtils.isNotBlank(deliveryCode)){
            List<DirectProductVo> productVos = directProductMap.get(entity.getDirectCode() + "_" +
                    deliveryCode + "_" + entity.getKaGoodsCode());
            if (CollectionUtil.isNotEmpty(productVos)){
                productVoList.addAll(productVos);
            }
        }
        if (CollectionUtil.isEmpty(productVoList)) {
            entity.setOrderStatusMsg("未查询到商品对应关系");
        } else if (productVoList.size() == 1) {
            productVo = productVoList.get(0);
        } else {
            int allSize = productVoList.size();
            //匹配到多条数据  用是否默认标记为是的数据
            productVoList = productVoList.stream()
                    .filter(k -> BooleanEnum.TRUE.getCapital().equals(k.getIsDefault()))
                    .collect(Collectors.toList());
            if (CollectionUtil.isEmpty(productVoList)) {
                entity.setOrderStatusMsg("共[" + allSize + "]条商品对应关系,无默认对应关系");
            } else if (productVoList.size() > 1) {
                entity.setOrderStatusMsg("[" + productVoList.size() + "]条默认商品对应关系");
            } else {
                productVo = productVoList.get(0);
            }
        }
        if (Objects.isNull(productVo)) {
            return;
        }
        //系统编码
        entity.setDirectCode(productVo.getDirectCode());
        //产品信息
        entity.setProductCode(productVo.getProductCode());
        entity.setProductName(productVo.getProductName());
        entity.setOrderStatus(StockStatus.S200.getDictCode());
    }

    /**
     * @Description: 第四步：转换状态设置
     * @Param: [entity]
     * @return: void
     * @Author: ZS
     * @Date: 2022/12/16
     */
    public void setOrderStatusAndMsgFour(Stock entity) {
        if (Objects.equals(entity.getOrderStatus(), StockStatus.S200.getDictCode())) {
            entity.setOrderStatusMsg("转换成功");
        }
    }
}
