package com.biz.crm.kms.business.statement.local.service.internal;


import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.kms.business.direct.sdk.service.DirectVoService;
import com.biz.crm.kms.business.direct.sdk.vo.DirectVo;
import com.biz.crm.kms.business.direct.store.sdk.dto.DirectStoreConditionDto;
import com.biz.crm.kms.business.direct.store.sdk.service.DirectStoreVoService;
import com.biz.crm.kms.business.direct.store.sdk.vo.DirectStoreVo;
import com.biz.crm.kms.business.reconciliation.manage.sdk.dto.ExposureConditionsDto;
import com.biz.crm.kms.business.reconciliation.manage.sdk.dto.ReconciliationDto;
import com.biz.crm.kms.business.reconciliation.manage.sdk.service.ReconciliationService;
import com.biz.crm.kms.business.reconciliation.manage.sdk.vo.ReconciliationVo;
import com.biz.crm.kms.business.statement.local.entity.ExposureEntity;
import com.biz.crm.kms.business.statement.local.entity.SapExposureEntity;
import com.biz.crm.kms.business.statement.local.repository.ExposureVoRepository;
import com.biz.crm.kms.business.statement.local.repository.KmsSd120Repository;
import com.biz.crm.kms.business.statement.local.service.ExposureTransService;
import com.biz.crm.kms.business.statement.sdk.constant.ExposureConstant;
import com.biz.crm.kms.business.statement.sdk.dto.KmsSd120Dto;
import com.biz.crm.kms.business.statement.sdk.service.SapExposureService;
import com.biz.crm.kms.business.statement.sdk.vo.ExposureVo;
import com.biz.crm.kms.business.statement.sdk.service.ExposureVoService;
import com.biz.crm.kms.business.statement.sdk.vo.KmsSd120Vo;
import com.biz.crm.kms.business.statement.sdk.vo.SapExposureVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.common.base.util.UuidCrmUtil;
import com.biz.crm.mn.third.system.sd.sdk.dto.LineOfCreditDataDto;
import com.biz.crm.mn.third.system.sd.sdk.service.SapSdApiService;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.springframework.stereotype.Service;
import org.springframework.beans.factory.annotation.Autowired;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.data.domain.Pageable;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 敞口管理Vo(ExposureVo)表服务实现类
 *
 * @author zs
 * @since 2022-11-05 15:59:19
 */
@Service("exposureVoService")
@Slf4j
public class ExposureVoServiceImpl implements ExposureVoService {

    @Autowired(required = false)
    private ExposureVoRepository exposureVoRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private SapExposureService sapExposureService;

    @Autowired(required = false)
    private ReconciliationService reconciliationService;

    @Autowired(required = false)
    private ExposureTransService exposureTransService;

    @Autowired(required = false)
    private RedisMutexService redisMutexService;

    @Autowired(required = false)
    private SapSdApiService sapSdApiService;

    @Autowired(required = false)
    private DirectStoreVoService directStoreVoService;

    @Autowired(required = false)
    private DirectVoService directVoService;

    @Autowired(required = false)
    private KmsSd120Repository kmsSd120Repository;

    @Autowired(required = false)
    private DictDataVoService dictDataVoService;


    /**
     * 分页查询数据
     *
     * @param pageable   分页对象
     * @param exposureVo 实体对象
     * @return
     */
    @Override
    public Page<ExposureVo> findByConditions(Pageable pageable, ExposureVo exposureVo) {
        ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(exposureVo)) {
            exposureVo = new ExposureVo();
        }
        return this.exposureVoRepository.findByConditions(pageable, exposureVo);
    }


    /**
     * 根据商超系统查找敞口信息
     *
     * @param supermarketCode
     * @return
     */
    @Override
    public List<ExposureVo> findBySupermarket(String supermarketCode, String supermarketName) {
        List<ExposureEntity> entityList = this.exposureVoRepository.findBySupermarket(supermarketCode, supermarketName);
        List<ExposureVo> voList = new ArrayList<>();
        entityList.forEach(aa -> {
            ExposureVo vo = this.nebulaToolkitService.copyObjectByWhiteList(aa, ExposureVo.class, HashSet.class, ArrayList.class);
            voList.add(vo);
        });
        return voList;
    }

    /**
     * 通过主键查询单条数据
     *
     * @param id 主键
     * @return 单条数据
     */
    @Override
    public ExposureVo findById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        ExposureEntity entity = this.exposureVoRepository.getById(id);
        ExposureVo vo = this.nebulaToolkitService.copyObjectByWhiteList(entity, ExposureVo.class, HashSet.class, ArrayList.class);
        return vo;
    }

    /**
     * 新增数据
     *
     * @param exposureVo 实体对象
     * @return 新增结果
     */
    @Transactional
    @Override
    public ExposureVo create(ExposureVo exposureVo) {
        this.createValidate(exposureVo);
        ExposureEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(exposureVo, ExposureEntity.class, HashSet.class, ArrayList.class);
        entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        entity.setTenantCode(TenantUtils.getTenantCode());
        this.exposureVoRepository.saveOrUpdate(entity);
        return exposureVo;
    }

    /**
     * 修改新据
     *
     * @param exposureVo 实体对象
     * @return 修改结果
     */
    @Transactional
    @Override
    public ExposureVo update(ExposureVo exposureVo) {
        this.updateValidate(exposureVo);
        ExposureEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(exposureVo, ExposureEntity.class, HashSet.class, ArrayList.class);
        this.exposureVoRepository.saveOrUpdate(entity);
        return exposureVo;
    }

    /**
     * 删除数据
     *
     * @param idList 主键结合
     * @return 删除结果
     */
    @Transactional
    @Override
    public void delete(List<String> idList) {
        Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
        this.exposureVoRepository.removeByIds(idList);
    }

    /**
     * 启用（单个或者批量）
     *
     * @param idList 主键结合
     * @return 启用结果
     */
    @Transactional
    @Override
    public void enable(List<String> idList) {
        Validate.isTrue(!CollectionUtils.isEmpty(idList), "启用数据时，主键集合不能为空！");
        this.exposureVoRepository.updateEnableStatusByIds(idList, EnableStatusEnum.ENABLE);
    }

    /**
     * 禁用（单个或者批量）
     *
     * @param idList 主键结合
     * @return 禁用结果
     */
    @Transactional
    @Override
    public void disable(List<String> idList) {
        Validate.isTrue(!CollectionUtils.isEmpty(idList), "禁用数据时，主键集合不能为空！");
        this.exposureVoRepository.updateEnableStatusByIds(idList, EnableStatusEnum.DISABLE);
    }

    /**
     * 同步SAP数据
     */
    @Override
    public void synchronize() {
        boolean lock = true;
        String lockKey = DateUtil.format(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY);
        try {
            lock = this.lock(lockKey);
            if (!lock) {
                return;
            }
            Pageable reconciliationPageable = PageRequest.of(1,ExposureConstant.DEFAULT_SIZE);
            //获取对账单数据
            List<ReconciliationVo> reconciliationVoList = new ArrayList<>();
            ReconciliationDto reconciliationDto = new ReconciliationDto();
            Page<ReconciliationVo> reconciliationVoPage = this.reconciliationService.findByConditions(reconciliationPageable, reconciliationDto);
            reconciliationVoList.addAll(reconciliationVoPage.getRecords());
            while (reconciliationVoPage.hasNext()
                    && ExposureConstant.DEFAULT_SIZE >= reconciliationPageable.getPageNumber()) {
                reconciliationPageable = reconciliationPageable.next();
                reconciliationVoPage = this.reconciliationService.findByConditions(reconciliationPageable, reconciliationDto);
                reconciliationVoList.addAll(reconciliationVoPage.getRecords());
            }
            Map<String,List<ReconciliationVo>> reconciliationMap = new HashMap<>();
            if (!CollectionUtils.isEmpty(reconciliationVoList)){
                reconciliationMap = reconciliationVoList.stream().filter(vo -> StringUtils.isNotEmpty(vo.getSoldToPartyCode())).distinct().collect(Collectors.groupingBy(ReconciliationVo::getSoldToPartyCode));
            }
            //获取信贷数据
            KmsSd120Dto dto = new KmsSd120Dto();
            dto.setTenantCode(TenantUtils.getTenantCode());
            dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            dto.setEnableStatus(DelFlagStatusEnum.NORMAL.getCode());
            Pageable sd120Pageable = PageRequest.of(1,ExposureConstant.DEFAULT_SIZE);
            List<KmsSd120Vo> sd120VoList = new ArrayList<>();
            Page<KmsSd120Vo> kmsSd120VoPage = this.kmsSd120Repository.findByConditions(sd120Pageable,dto);
            if (CollectionUtils.isEmpty(kmsSd120VoPage.getRecords())){
                return;
            }
            sd120VoList.addAll(kmsSd120VoPage.getRecords());
            while (kmsSd120VoPage.hasNext()
                    && ExposureConstant.DEFAULT_SIZE >= sd120Pageable.getPageNumber()) {
                sd120Pageable = sd120Pageable.next();
                kmsSd120VoPage = this.kmsSd120Repository.findByConditions(sd120Pageable, dto);
                sd120VoList.addAll(kmsSd120VoPage.getRecords());
            }
            //获取系统
            Set<String> sellPartCode = sd120VoList.stream().map(KmsSd120Vo::getKunnr).collect(Collectors.toSet());
            DirectStoreConditionDto storeConditionDto = new DirectStoreConditionDto();
            storeConditionDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            storeConditionDto.setEnableStatus(DelFlagStatusEnum.NORMAL.getCode());
            storeConditionDto.setSoldToPartyCodes(sellPartCode);
            List<DirectStoreVo> directStoreVoList = directStoreVoService.findByDirectStoreConditionDto(storeConditionDto);
            Map<String, DirectVo> directVoMap = new HashMap<>();
            if (!CollectionUtils.isEmpty(directStoreVoList)){
                directVoMap = this.buildDirectVoInfo(directStoreVoList);
            }

            if (!CollectionUtils.isEmpty(sd120VoList)){
                List<ExposureEntity> exposureEntityList = new ArrayList<>();
                for (KmsSd120Vo kmsSd120Vo : sd120VoList) {
                    ExposureEntity exposureEntity = new ExposureEntity();
                    exposureEntity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                    exposureEntity.setEnableStatus(DelFlagStatusEnum.NORMAL.getCode());
                    exposureEntity.setTenantCode(TenantUtils.getTenantCode());
                    if (MapUtils.isNotEmpty(directVoMap)){
                        DirectVo directVo = directVoMap.get(kmsSd120Vo.getKunnr());
                        if (Objects.nonNull(directVo)){
                            exposureEntity.setDirectCode(directVo.getDirectCode());
                            exposureEntity.setDirectName(directVo.getSupermarketName());
                            exposureEntity.setSupermarketCode(directVo.getSupermarketCode());
                            exposureEntity.setSupermarketName(directVo.getSupermarketName());
                            exposureEntity.setBusinessFormatCode(directVo.getBusinessFormatCode());
                            exposureEntity.setBusinessUnitCode(directVo.getBusinessUnitCode());
                        }
                    }
                    // 已占用敞口
                    List<ReconciliationVo> reconciliationVos = reconciliationMap.get(kmsSd120Vo.getKunnr());
                    if (!CollectionUtils.isEmpty(reconciliationVos)){
                        BigDecimal amountExposure = BigDecimal.ZERO;
                        for (ReconciliationVo reconciliationVo : reconciliationVos) {
                            amountExposure = amountExposure.add(reconciliationVo.getOccupiedExposure() == null ? BigDecimal.ZERO:reconciliationVo.getOccupiedExposure());
                        }
                        exposureEntity.setExposureCurrent(amountExposure);
                    }else {
                        exposureEntity.setExposureCurrent(BigDecimal.ZERO);
                    }
                    // 信贷额度
                    exposureEntity.setExposureInitial(kmsSd120Vo.getKlimk() == null ? BigDecimal.ZERO:kmsSd120Vo.getKlimk());
                    //销售值
                    exposureEntity.setSalesAmount(kmsSd120Vo.getSauft());
                    // SAP剩余信贷额度(信贷限额-销售值-应收总额)
                    exposureEntity.setSapExposureCurrent(exposureEntity.getExposureInitial().subtract(kmsSd120Vo.getSauft()).subtract(kmsSd120Vo.getSkfor()));
                    //剩余信贷额度：信贷额度-已占用敞口
                    exposureEntity.setRemainingCreditLine(exposureEntity.getExposureInitial().subtract(exposureEntity.getExposureCurrent()).setScale(6, BigDecimal.ROUND_DOWN));

                    //售达方
                    exposureEntity.setSellPartyCode(kmsSd120Vo.getKunnr());
                    exposureEntity.setSellPartyName(kmsSd120Vo.getName1());

                    exposureEntityList.add(exposureEntity);
                }
                //更新敞口数据
                Pageable exposurePageable = PageRequest.of(1,ExposureConstant.DEFAULT_SIZE);
                List<ExposureVo> exposureVoList = new ArrayList<>();
                ExposureVo exposureVo = new ExposureVo();
                exposureVo.setTenantCode(TenantUtils.getTenantCode());
                Page<ExposureVo> exposureVoPage = this.exposureVoRepository.findByConditions(exposurePageable, exposureVo);
                exposureVoList.addAll(exposureVoPage.getRecords());
                while (exposureVoPage.hasNext()
                        && ExposureConstant.DEFAULT_SIZE >= exposurePageable.getPageNumber()) {
                    exposurePageable = exposurePageable.next();
                    exposureVoPage = this.exposureVoRepository.findByConditions(exposurePageable, exposureVo);
                    exposureVoList.addAll(exposureVoPage.getRecords());
                }
                Map<String,String> exposureVoMap = new HashMap<>();
                if (!CollectionUtils.isEmpty(exposureVoList)){
                    exposureVoMap = exposureVoList.stream().collect(Collectors.toMap(ExposureVo::getSellPartyCode,ExposureVo::getId,(a,b)->a));
                }
                for (ExposureEntity exposureEntity : exposureEntityList) {
                    String id = exposureVoMap.get(exposureEntity.getSellPartyCode());
                    if (StringUtils.isEmpty(id)) {
                        exposureEntity.setId(id);
                    }
                }
                this.exposureVoRepository.saveOrUpdateBatch(exposureEntityList);
            }
        } finally {
            if (lock) {
                this.unLock(lockKey);
            }
        }
    }


    /**
     * 释放锁
     *
     * @param lockKey
     */
    private void unLock(String lockKey) {
        if (StringUtils.isEmpty(lockKey)) {
            throw new RuntimeException("拉取库存数据解锁失败，日期不能为空！");
        }
        redisMutexService.unlock(ExposureConstant.EXPOSURE_REPORT_LOCK + lockKey);
    }

    /**
     * 获取锁
     *
     * @param lockKey
     * @return
     */
    private boolean lock(String lockKey) {
        if (StringUtils.isEmpty(lockKey)) {
            throw new RuntimeException("拉取库存数据加锁失败，日期不能为空！");
        }
        return this.redisMutexService.tryLock(ExposureConstant.EXPOSURE_REPORT_LOCK + lockKey, TimeUnit.HOURS, 12);
    }

    /**
     * 信贷接口发送测试
     */
    @Override
    public void toSD(){
        List<DictDataVo> salesCodes = dictDataVoService.findByDictTypeCode("kms_exposure_range");
        if (CollectionUtils.isEmpty(salesCodes)){
            return;
        }
        Map<String, String> salesMap = salesCodes.stream().collect(Collectors.toMap(DictDataVo::getDictCode, DictDataVo::getDictValue, (a, b) -> a));
        LineOfCreditDataDto dto = new LineOfCreditDataDto();
        LineOfCreditDataDto.MessageHeader header = new LineOfCreditDataDto.MessageHeader();
        header.setMessageId(UuidCrmUtil.general());
        header.setInterFace("SI_TPM120_XDEDCX_ASYN_OUT");
        header.setSender("TPM");
        header.setSendTime(DateUtil.date_yyyyMMddHHmmss.format(new Date()));
        header.setReceiver("ECC");
        dto.setMessageHeader(header);
        //获取系统
        List<DirectVo> directVos = this.directVoService.findDirectCodesByBusinessUnitCode("DY00000009");
        if (CollectionUtils.isEmpty(directVos)){
            return;
        }
        Map<String, DirectVo> directVoMap = directVos.stream().collect(Collectors.toMap(DirectVo::getDirectCode, Function.identity(), (a, b) -> a));
        List<String> directCodes = directVos.stream().map(DirectVo::getDirectCode).distinct().collect(Collectors.toList());
        List<DirectStoreVo> soldPartyCodeByDirectCodes = this.directStoreVoService.findSoldPartyCodeByDirectCodes(directCodes);
        if (!CollectionUtils.isEmpty(soldPartyCodeByDirectCodes)){
            List<LineOfCreditDataDto.Item1> item1s = new ArrayList<>();
            for (DirectStoreVo directStoreVo : soldPartyCodeByDirectCodes) {
                DirectVo directVo = directVoMap.get(directStoreVo.getDirectCode());
                if (Objects.nonNull(directVo)){
                    LineOfCreditDataDto.Item1 item1 = new LineOfCreditDataDto.Item1();
                    if (!StringUtils.isEmpty(directVo.getSalesOrgCode()) && !StringUtils.isEmpty(directStoreVo.getSoldToPartyCode())){
                        String salesCode = salesMap.get(directVo.getSalesOrgCode());
                        if (!StringUtils.isEmpty(salesCode)){
                            item1.setKkber(salesCode);
                            item1.setKunnr(directStoreVo.getSoldToPartyCode());
                            item1s.add(item1);
                        }
                    }
                }
            }
            item1s = item1s.stream().distinct().collect(Collectors.toList());
            dto.setItems(item1s);
            this.sapSdApiService.queryLineOfCredit(dto);
        }
    }

    @Override
    public void updateSD() {
        //先删除敞口数据在拉取
        this.kmsSd120Repository.deleteAll();
        this.toSD();
    }

    /**
     * 创建验证
     *
     * @param exposureVo
     */
    private void createValidate(ExposureVo exposureVo) {
        Validate.notNull(exposureVo, "新增时，对象信息不能为空！");
        exposureVo.setId(null);

    }

    /**
     * 修改验证
     *
     * @param exposureVo
     */
    private void updateValidate(ExposureVo exposureVo) {
        Validate.notNull(exposureVo, "修改时，对象信息不能为空！");

    }

    private Map<String, DirectVo> buildDirectVoInfo(List<DirectStoreVo> directStoreVoList) {
        if (CollectionUtils.isEmpty(directStoreVoList)) {
            return null;
        }
        //根据送达方去找系统
        Map<String, String> directStoreVoMap = directStoreVoList.stream().collect(Collectors.toMap(DirectStoreVo::getSoldToPartyCode, DirectStoreVo::getDirectCode, (a, b) -> a));
        List<String> directCodes = directStoreVoList.stream().map(DirectStoreVo::getDirectCode).distinct().collect(Collectors.toList());
        List<DirectVo> directVoList = directVoService.findByDirectCodes(directCodes);
        Map<String, DirectVo> directVoMap = directVoList.stream().collect(Collectors.toMap(DirectVo::getDirectCode, Function.identity()));
        Map<String, DirectVo> resultDirectVoMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(directVoMap)) {
            for (String soldPartyCode : directStoreVoMap.keySet()) {
                if (directVoMap.containsKey(directStoreVoMap.get(soldPartyCode))) {
                    resultDirectVoMap.put(soldPartyCode, directVoMap.get(directStoreVoMap.get(soldPartyCode)));
                }
            }
        }
        return resultDirectVoMap;
    }
}

