package com.biz.crm.kms.business.supermarket.parameter.local.service.internal;

import com.biz.crm.kms.business.supermarket.parameter.local.entity.SupermarketParameterOptionEntity;
import com.biz.crm.kms.business.supermarket.parameter.local.repository.SupermarketParameterOptionRepository;
import com.biz.crm.kms.business.supermarket.parameter.sdk.dto.SupermarketParameterOptionDto;
import com.biz.crm.kms.business.supermarket.parameter.sdk.event.SupermarketParameterOptionEventListener;
import com.biz.crm.kms.business.supermarket.parameter.sdk.service.SupermarketParameterOptionVoService;
import com.biz.crm.kms.business.supermarket.parameter.sdk.vo.SupermarketParameterOptionVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import lombok.extern.slf4j.Slf4j;

/**
 * 商超抓单参数表附属选项表(SupermarketParameterOption)表服务实现类
 *
 * @author xi.peng
 * @since 2022-09-23 10:49:15
 */
@Slf4j
@Service
public class SupermarketParameterOptionVoServiceImpl implements SupermarketParameterOptionVoService {

  @Autowired(required = false)
  private SupermarketParameterOptionRepository supermarketParameterOptionRepository;

  @Autowired(required = false)
  private List<SupermarketParameterOptionEventListener> supermarketParameterOptionEventListeners;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public SupermarketParameterOptionVo findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    SupermarketParameterOptionEntity entity = this.supermarketParameterOptionRepository.findById(id);
    if (entity == null) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, SupermarketParameterOptionVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public List<SupermarketParameterOptionVo> findParameterIds(List<String> parameterIds) {
    if (CollectionUtils.isEmpty(parameterIds)) {
      return null;
    }
    List<SupermarketParameterOptionEntity> optionEntities = this.supermarketParameterOptionRepository.findByParameterIds(parameterIds);
    if (CollectionUtils.isEmpty(optionEntities)) {
      return null;
    }
    return (List<SupermarketParameterOptionVo>) this.nebulaToolkitService.copyCollectionByWhiteList(optionEntities, SupermarketParameterOptionEntity.class, SupermarketParameterOptionVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public void createBatch(List<SupermarketParameterOptionDto> dtoList) {
    this.validation(dtoList);
    List<SupermarketParameterOptionEntity> optionEntities = (List<SupermarketParameterOptionEntity>) this.nebulaToolkitService.copyCollectionByWhiteList(dtoList, SupermarketParameterOptionDto.class, SupermarketParameterOptionEntity.class, HashSet.class, ArrayList.class);
    this.supermarketParameterOptionRepository.saveBatch(optionEntities);
    // 发送批量创建通知
    if (CollectionUtils.isNotEmpty(supermarketParameterOptionEventListeners)) {
      List<SupermarketParameterOptionVo> voList = (List<SupermarketParameterOptionVo>) this.nebulaToolkitService.copyCollectionByWhiteList(optionEntities, SupermarketParameterOptionEntity.class, SupermarketParameterOptionVo.class, HashSet.class, ArrayList.class);
      this.supermarketParameterOptionEventListeners.forEach(event -> event.onCreateBatch(voList));
    }
  }

  @Override
  @Transactional
  public void deleteByParameterId(String parameterId) {
    Validate.notBlank(parameterId, "接口参数ID不能为空！");
    List<SupermarketParameterOptionEntity> optionEntities = this.supermarketParameterOptionRepository.findByParameterId(parameterId);
    if (CollectionUtils.isEmpty(optionEntities)) {
      // 如果已删除或不存在就不继续向下执行
      return;
    }
    this.supermarketParameterOptionRepository.deleteByParameterId(parameterId);
    // 根据parameterId删除时发送通知
    if (CollectionUtils.isNotEmpty(supermarketParameterOptionEventListeners)) {
      this.supermarketParameterOptionEventListeners.forEach(event -> event.onDeleteByParameterId(parameterId));
    }
  }

  private void validation(List<SupermarketParameterOptionDto> dtoList) {
    Validate.notEmpty(dtoList, "选项集合对象不能为空！");
    for (int i = 0; i < dtoList.size(); i++) {
      SupermarketParameterOptionDto option = dtoList.get(i);
      Validate.notBlank(option.getParameterId(), "参数表ID不能为空！");
      Validate.notBlank(option.getNameLabel(), "选项名称不能为空！");
      Validate.notBlank(option.getValueLabel(), "选项值不能为空！");
      if (null == option.getSortIndex()) {
        // 前端未设置排序值默认按传入顺序排序
        option.setSortIndex(i);
      }
      //长度校验
      Validate.isTrue(option.getNameLabel().length()<=64, "选项名称最多64位字符串");
      Validate.isTrue(option.getValueLabel().length()<=64, "选项值最多64位字符串");
    }
  }
  
}
