package com.biz.crm.kms.business.supermarket.parameter.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.kms.business.supermarket.parameter.local.entity.SupermarketParameterEntity;
import com.biz.crm.kms.business.supermarket.parameter.local.repository.SupermarketParameterRepository;
import com.biz.crm.kms.business.supermarket.parameter.sdk.dto.SupermarketParameterDto;
import com.biz.crm.kms.business.supermarket.parameter.sdk.dto.SupermarketParameterPageDto;
import com.biz.crm.kms.business.supermarket.parameter.sdk.event.SupermarketParameterEventListener;
import com.biz.crm.kms.business.supermarket.parameter.sdk.service.SupermarketParameterLabelVoService;
import com.biz.crm.kms.business.supermarket.parameter.sdk.service.SupermarketParameterOptionVoService;
import com.biz.crm.kms.business.supermarket.parameter.sdk.service.SupermarketParameterVoService;
import com.biz.crm.kms.business.supermarket.parameter.sdk.vo.SupermarketParameterLabelVo;
import com.biz.crm.kms.business.supermarket.parameter.sdk.vo.SupermarketParameterOptionVo;
import com.biz.crm.kms.business.supermarket.parameter.sdk.vo.SupermarketParameterVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import lombok.extern.slf4j.Slf4j;

/**
 * 商超抓单参数表(SupermarketParameter)表服务实现类
 *
 * @author xi.peng
 * @since 2022-09-23 10:47:58
 */
@Slf4j
@Service
public class SupermarketParameterVoServiceImpl implements SupermarketParameterVoService {

  @Autowired(required = false)
  private SupermarketParameterRepository supermarketParameterRepository;

  @Autowired(required = false)
  private SupermarketParameterLabelVoService supermarketParameterLabelVoService;

  @Autowired(required = false)
  private SupermarketParameterOptionVoService supermarketParameterOptionVoService;

  @Autowired(required = false)
  private List<SupermarketParameterEventListener> supermarketParameterEventListeners;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public Page<SupermarketParameterVo> findByConditions(Pageable pageable, SupermarketParameterPageDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(1, 50));
    dto = Optional.ofNullable(dto).orElse(new SupermarketParameterPageDto());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    dto.setTenantCode(TenantUtils.getTenantCode());
    Page<SupermarketParameterPageDto> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.supermarketParameterRepository.findByConditions(page, dto);
  }

  @Override
  public List<SupermarketParameterVo> findListByConditions(SupermarketParameterDto dto) {
    dto = Optional.ofNullable(dto).orElse(new SupermarketParameterDto());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    dto.setTenantCode(TenantUtils.getTenantCode());
    List<SupermarketParameterVo> parameterVoList = this.supermarketParameterRepository.findListByConditions(dto);
    if (CollectionUtils.isEmpty(parameterVoList)) {
      return parameterVoList;
    }
    List<String> parameterIds = parameterVoList.stream().map(SupermarketParameterVo::getId).collect(Collectors.toList());
    List<SupermarketParameterLabelVo> labelVoList = this.supermarketParameterLabelVoService.findParameterIds(parameterIds);
    Map<String, List<SupermarketParameterLabelVo>> labelMap = Optional.ofNullable(labelVoList).orElse(Lists.newArrayList()).stream().collect(Collectors.groupingBy(SupermarketParameterLabelVo::getParameterId));
    List<SupermarketParameterOptionVo> optionVoList = this.supermarketParameterOptionVoService.findParameterIds(parameterIds);
    Map<String, List<SupermarketParameterOptionVo>> optionMap = Optional.ofNullable(optionVoList).orElse(Lists.newArrayList()).stream().collect(Collectors.groupingBy(SupermarketParameterOptionVo::getParameterId));
    parameterVoList.forEach(parameterVo->{
      parameterVo.setLabelList(labelMap.get(parameterVo.getId()));
      parameterVo.setOptionList(optionMap.get(parameterVo.getId()));
    });
    return parameterVoList;
  }

  @Override
  public List<SupermarketParameterVo> findBySupermarketCodes(List<String> supermarketCodes) {
    if (CollectionUtils.isEmpty(supermarketCodes)) {
      return Lists.newArrayList();
    }
    List<SupermarketParameterEntity> supermarketParameterEntities = this.supermarketParameterRepository.findBySupermarketCodes(supermarketCodes);
    if (CollectionUtils.isEmpty(supermarketParameterEntities)) {
      return Lists.newArrayList();
    }
    return (List<SupermarketParameterVo>) this.nebulaToolkitService.copyCollectionByWhiteList(supermarketParameterEntities, SupermarketParameterEntity.class, SupermarketParameterVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public SupermarketParameterVo findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    SupermarketParameterEntity entity = this.supermarketParameterRepository.findById(id);
    if (entity == null) {
      return null;
    }
    SupermarketParameterVo parameterVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, SupermarketParameterVo.class, HashSet.class, ArrayList.class);
    List<SupermarketParameterLabelVo> labelVoList = this.supermarketParameterLabelVoService.findParameterIds(Collections.singletonList(entity.getId()));
    List<SupermarketParameterOptionVo> optionVoList = this.supermarketParameterOptionVoService.findParameterIds(Collections.singletonList(entity.getId()));
    parameterVo.setLabelList(labelVoList);
    parameterVo.setOptionList(optionVoList);
    return parameterVo;
  }

  @Override
  @Transactional
  public void create(SupermarketParameterDto dto) {
    this.createValidation(dto);
    SupermarketParameterEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, SupermarketParameterEntity.class, HashSet.class, ArrayList.class);
    this.supermarketParameterRepository.save(entity);

    // 创建附属标签
    if (CollectionUtils.isNotEmpty(dto.getLabelList())) {
      dto.getLabelList().forEach(label-> label.setParameterId(entity.getId()));
      this.supermarketParameterLabelVoService.createBatch(dto.getLabelList());
    }

    // 创建附属选项
    if (CollectionUtils.isNotEmpty(dto.getOptionList())) {
      dto.getOptionList().forEach(option-> option.setParameterId(entity.getId()));
      this.supermarketParameterOptionVoService.createBatch(dto.getOptionList());
    }

    // 发送通知
    if (CollectionUtils.isNotEmpty(supermarketParameterEventListeners)) {
      SupermarketParameterVo vo = this.nebulaToolkitService.copyObjectByWhiteList(entity, SupermarketParameterVo.class, HashSet.class, ArrayList.class);
      this.supermarketParameterEventListeners.forEach(event -> event.onCreate(vo));
    }
  }

  @Override
  @Transactional
  public void update(SupermarketParameterDto dto) {
    this.updateValidation(dto);
    SupermarketParameterEntity oldEntity = this.supermarketParameterRepository.findById(dto.getId());
    Validate.notNull(oldEntity, "不存在或已删除！");
    SupermarketParameterEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, SupermarketParameterEntity.class, HashSet.class, ArrayList.class);
    this.supermarketParameterRepository.updateById(entity);

    // 删除原附属标签
    this.supermarketParameterLabelVoService.deleteByParameterId(dto.getId());

    // 删除原附属选项
    this.supermarketParameterOptionVoService.deleteByParameterId(dto.getId());

    // 创建新附属标签
    if (CollectionUtils.isNotEmpty(dto.getLabelList())) {
      dto.getLabelList().forEach(label-> label.setParameterId(entity.getId()));
      this.supermarketParameterLabelVoService.createBatch(dto.getLabelList());
    }

    // 创建新附属选项
    if (CollectionUtils.isNotEmpty(dto.getOptionList())) {
      dto.getOptionList().forEach(option-> option.setParameterId(entity.getId()));
      this.supermarketParameterOptionVoService.createBatch(dto.getOptionList());
    }

    // 发送修改通知
    if (CollectionUtils.isNotEmpty(supermarketParameterEventListeners)) {
      SupermarketParameterVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(oldEntity, SupermarketParameterVo.class, HashSet.class, ArrayList.class);
      SupermarketParameterVo newVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, SupermarketParameterVo.class, HashSet.class, ArrayList.class);
      this.supermarketParameterEventListeners.forEach(event -> event.onUpdate(oldVo, newVo));
    }
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空！");
    List<SupermarketParameterEntity> entities = this.supermarketParameterRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    this.supermarketParameterRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
    // 发送启用通知
    if (CollectionUtils.isNotEmpty(supermarketParameterEventListeners)) {
      List<SupermarketParameterVo> voList = (List<SupermarketParameterVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, SupermarketParameterEntity.class, SupermarketParameterVo.class, HashSet.class, ArrayList.class);
      this.supermarketParameterEventListeners.forEach(event -> event.onEnable(voList));
    }
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空！");
    List<SupermarketParameterEntity> entities = this.supermarketParameterRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    this.supermarketParameterRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
    // 发送禁用通知
    if (CollectionUtils.isNotEmpty(supermarketParameterEventListeners)) {
      List<SupermarketParameterVo> voList = (List<SupermarketParameterVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, SupermarketParameterEntity.class, SupermarketParameterVo.class, HashSet.class, ArrayList.class);
      this.supermarketParameterEventListeners.forEach(event -> event.onDisable(voList));
    }
  }

  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空！");
    List<SupermarketParameterEntity> entities = this.supermarketParameterRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    this.supermarketParameterRepository.updateDelFlagByIds(ids);
    // 发送删除通知
    if (CollectionUtils.isNotEmpty(supermarketParameterEventListeners)) {
      List<SupermarketParameterVo> voList = (List<SupermarketParameterVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, SupermarketParameterEntity.class, SupermarketParameterVo.class, HashSet.class, ArrayList.class);
      this.supermarketParameterEventListeners.forEach(event -> event.onDelete(voList));
    }
  }

  private void createValidation(SupermarketParameterDto dto) {
    this.validation(dto);
    dto.setId(null);
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
  }

  private void updateValidation(SupermarketParameterDto dto) {
    this.validation(dto);
    Validate.notBlank(dto.getId(), "ID不能为空！");
  }

  private void validation(SupermarketParameterDto dto) {
    Validate.notNull(dto, "对象不能为空！");
    Validate.notBlank(dto.getSupermarketCode(), "商超编码不能为空！");
    Validate.notBlank(dto.getInterfaceParameter(), "接口参数不能为空！");
    Validate.notBlank(dto.getControlType(), "控件类型不能为空！");
    Validate.notBlank(dto.getParameterAttr(), "参数属性不能为空！");
    Validate.notBlank(dto.getParameterText(), "参数文本不能为空！");
    Validate.notBlank(dto.getIsRequired(), "是否必填不能为空！");
    Validate.notBlank(dto.getIsShow(), "是否显示不能为空！");
    //长度校验
    Validate.isTrue(dto.getParameterText().length()<=64, "参数名称最多64位字符串");
    Validate.isTrue(dto.getParameterAttr().length()<=32, "参数属性最多32位字符串");
    if (StringUtils.isNotBlank(dto.getPromptText())) {
      Validate.isTrue(dto.getPromptText().length()<=256, "提示文本最多256位字符串");
    }
    if (StringUtils.isNotBlank(dto.getRemark())) {
      Validate.isTrue(dto.getRemark().length()<=400, "备注最多400位字符串");
    }
  }
}
