package com.biz.crm.mdm.admin.web.report.customer.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.mdm.admin.web.report.customer.dto.AbstractCustomerPageDto;
import com.biz.crm.mdm.admin.web.report.customer.repository.CustomerReportRepository;
import com.biz.crm.mdm.admin.web.report.customer.service.CustomerPageVoService;
import com.biz.crm.mdm.admin.web.report.customer.strategy.CustomerReportStrategy;
import com.biz.crm.mdm.admin.web.report.customer.vo.CustomerPageVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerTagVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerTagVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 客户分页VO接口实现类
 *
 * @author ning.zhang
 * @date 2021/11/4
 */
@Service
@Slf4j
public class CustomerPageVoServiceImpl implements CustomerPageVoService {

  @Autowired(required = false)
  private CustomerReportRepository customerReportRepository;

  @Autowired(required = false)
  private CustomerTagVoService customerTagVoService;

  private Map<String, CustomerReportStrategy> strategyMap;

  public CustomerPageVoServiceImpl(List<CustomerReportStrategy> itemStrategies) {
    strategyMap = Maps.newHashMap();
    if (org.springframework.util.CollectionUtils.isEmpty(itemStrategies)) {
      return;
    }
    itemStrategies.forEach(strategy -> {
      CustomerReportStrategy itemStrategy = strategyMap.get(strategy.getPageSource());
      Validate.isTrue(Objects.isNull(itemStrategy), String.format("存在相同的报表数据策略[%s]", strategy.getPageSource()));
      strategyMap.put(strategy.getPageSource(), strategy);
    });
  }

  @Override
  public Page<CustomerPageVo> findByConditions(Pageable pageable, AbstractCustomerPageDto dto) {
    CustomerReportStrategy strategy = strategyMap.get(dto.getPageSource());
    //1.策略不存在获取请求参数封装结果不通过,直接返回null
    if (Objects.isNull(strategy)) {
      log.info("客户信息报表策略不存在....");
      return null;
    }
    if (!strategy.onRequest(dto)) {
      log.info("客户信息报表参数封装校验不通过....");
      return null;
    }
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto.setTenantCode(TenantUtils.getTenantCode());
    //2.获取报表数据
    Page<CustomerPageVo> resultPage = this.customerReportRepository.findByConditions(pageable, dto);
    List<CustomerPageVo> records = resultPage.getRecords();
    if (CollectionUtils.isEmpty(records)) {
      new Page<>();
    }
    if (org.springframework.util.CollectionUtils.isEmpty(resultPage.getRecords())) {
      return resultPage;
    }
    //3.数据返回封装
    strategy.onReturn(resultPage.getRecords());

    //组装标签数据
    this.buildTagInfo(records);
    resultPage.setRecords(records);
    return resultPage;
  }

  /**
   * 构建标签数据
   *
   * @param records
   */
  private void buildTagInfo(List<CustomerPageVo> records) {
    List<String> customerCodes = records.stream().map(CustomerPageVo::getCustomerCode).collect(Collectors.toList());
    List<CustomerTagVo> byCustomerCodes = this.customerTagVoService.findByCustomerCodes(customerCodes);
    if (CollectionUtils.isEmpty(byCustomerCodes)) {
      return;
    }
    Map<String, List<CustomerTagVo>> map = byCustomerCodes.stream().collect(Collectors.groupingBy(CustomerTagVo::getCustomerCode));
    for (CustomerPageVo record : records) {
      List<CustomerTagVo> customerTagVos = map.get(record.getCustomerCode());
      if (CollectionUtils.isEmpty(customerTagVos)) {
        continue;
      }
      record.setTags(customerTagVos.stream().map(CustomerTagVo::getTagName).collect(Collectors.toSet()));
    }
  }


}
