package com.biz.crm.mdm.admin.web.report.customer.strategy.helper;

import com.biz.crm.mdm.admin.web.report.customer.vo.CustomerPageVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerTagVoService;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerDockingVo;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerRelateOrgVo;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerTagVo;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 客户信息报表策略辅助类
 *
 * @author ning.zhang
 * @date 2022/3/29
 */
@Component
public class CustomerReportStrategyHelper {

  @Autowired(required = false)
  private CustomerVoService customerVoService;

  @Autowired(required = false)
  private CustomerTagVoService customerTagVoService;

  /**
   * 封装客户信息自身扩展数据
   *
   * @param voList 客户信息
   */
  public void buildCustomerSelfInfo(List<CustomerPageVo> voList) {
    if (CollectionUtils.isEmpty(voList)) {
      return;
    }
    List<String> customerCodes = voList.stream().map(CustomerPageVo::getCustomerCode).collect(Collectors.toList());
    List<CustomerVo> customerVos = this.customerVoService.findByCustomerCodes(customerCodes);
    if (CollectionUtils.isEmpty(customerVos)) {
      return;
    }
    Map<String, CustomerVo> customerVoMap = customerVos.stream()
        .collect(Collectors.toMap(CustomerVo::getCustomerCode, t -> t, (a, b) -> b));
    voList.forEach(customerPageVo -> {
      CustomerVo customerVo = customerVoMap.get(customerPageVo.getCustomerCode());
      if (Objects.isNull(customerVo)) {
        return;
      }
      this.buildOrgInfo(customerVo, customerPageVo);
    });
  }

  /**
   * 封装客户信息组织数据
   *
   * @param customerVo     客户详细信息
   * @param customerPageVo 客户分页信息
   */
  private void buildOrgInfo(CustomerVo customerVo, CustomerPageVo customerPageVo) {
    List<CustomerRelateOrgVo> orgList = customerVo.getOrgList();
    if (CollectionUtils.isEmpty(orgList)) {
      return;
    }
    Set<String> orgCodeSet = Sets.newHashSet();
    Set<String> orgNameSet = Sets.newHashSet();
    orgList.forEach(customerRelateOrgVo -> {
      if (StringUtils.isNotBlank(customerRelateOrgVo.getOrgCode())) {
        orgCodeSet.add(customerRelateOrgVo.getOrgCode());
      }
      if (StringUtils.isNotBlank(customerRelateOrgVo.getOrgName())) {
        orgNameSet.add(customerRelateOrgVo.getOrgName());
      }
    });
    customerPageVo.setOrgCode(CollectionUtils.isEmpty(orgCodeSet) ? null : String.join(",", orgCodeSet));
    customerPageVo.setOrgName(CollectionUtils.isEmpty(orgNameSet) ? null : String.join(",", orgNameSet));
  }

  /**
   * 封装客户信息职位数据
   *
   * @param customerVo     客户详细信息
   * @param customerPageVo 客户分页信息
   */
  private void buildPositionInfo(CustomerVo customerVo, CustomerPageVo customerPageVo) {
    List<CustomerDockingVo> dockingList = customerVo.getDockingList();
    if (CollectionUtils.isEmpty(dockingList)) {
      return;
    }
    Set<String> positionCodeSet = Sets.newHashSet();
    Set<String> positionNameSet = Sets.newHashSet();
    dockingList.forEach(dockingVo -> {
      if (StringUtils.isNotBlank(dockingVo.getPositionCode())) {
        positionCodeSet.add(dockingVo.getPositionCode());
      }
      if (StringUtils.isNotBlank(dockingVo.getPositionName())) {
        positionNameSet.add(dockingVo.getPositionName());
      }
    });
    customerPageVo.setPositionCode(CollectionUtils.isEmpty(positionCodeSet) ? null : String.join(",", positionCodeSet));
    customerPageVo.setPositionName(CollectionUtils.isEmpty(positionNameSet) ? null : String.join(",", positionNameSet));
  }

  /**
   * 封装标签数据
   *
   * @param voList 客户信息
   */
  public void buildTagInfo(List<CustomerPageVo> voList) {
    List<String> customerCodes = voList.stream().map(CustomerPageVo::getCustomerCode).collect(Collectors.toList());
    List<CustomerTagVo> byCustomerCodes = this.customerTagVoService.findByCustomerCodes(customerCodes);
    if (CollectionUtils.isEmpty(byCustomerCodes)) {
      return;
    }
    Map<String, List<CustomerTagVo>> map = byCustomerCodes.stream().collect(Collectors.groupingBy(CustomerTagVo::getCustomerCode));
    for (CustomerPageVo record : voList) {
      List<CustomerTagVo> customerTagVos = map.get(record.getCustomerCode());
      if (CollectionUtils.isEmpty(customerTagVos)) {
        continue;
      }
      record.setTags(customerTagVos.stream().map(CustomerTagVo::getTagName).collect(Collectors.toSet()));
    }
  }
}
