package com.biz.crm.mdm.admin.web.report.terminal.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mdm.admin.web.report.terminal.dto.NoRelaCurTerminalDto;
import com.biz.crm.mdm.admin.web.report.terminal.dto.NoRelaTerminalDto;
import com.biz.crm.mdm.admin.web.report.terminal.dto.RelaCurTerminalDto;
import com.biz.crm.mdm.admin.web.report.terminal.dto.TerminalReportPaginationDto;
import com.biz.crm.mdm.admin.web.report.terminal.repository.TerminalReportVoRepository;
import com.biz.crm.mdm.admin.web.report.terminal.service.TerminalReportVoService;
import com.biz.crm.mdm.admin.web.report.terminal.vo.TerminalContactReportVo;
import com.biz.crm.mdm.admin.web.report.terminal.vo.TerminalReportVo;
import com.biz.crm.mdm.business.org.local.entity.Org;
import com.biz.crm.mdm.business.org.local.service.OrgService;
import com.biz.crm.mdm.business.terminal.local.entity.TerminalContact;
import com.biz.crm.mdm.business.terminal.local.entity.TerminalRelaCustomerOrg;
import com.biz.crm.mdm.business.terminal.local.entity.TerminalRelaOrg;
import com.biz.crm.mdm.business.terminal.local.service.TerminalContactService;
import com.biz.crm.mdm.business.terminal.sdk.enums.TerminalSupplyTypeEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 终端供货关系
 *
 * @author sunx
 * @date 2021/11/5
 */
@Slf4j
@Service
public class TerminalReportVoServiceImpl implements TerminalReportVoService {

  @Autowired(required = false)
  private OrgService orgService;
  @Autowired(required = false)
  private TerminalReportVoRepository terminalReportVoRepository;
  @Autowired(required = false)
  private TerminalContactService terminalContactService;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public Page<TerminalReportVo> findByConditions(
      Pageable pageable, TerminalReportPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    Page<TerminalReportVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<TerminalReportVo> pageResult = terminalReportVoRepository.findByConditions(page, dto);
    this.initTerminalPageExtInfo(pageResult);
    return pageResult;
  }

  @Override
  public Page<TerminalReportVo> findByNoRelaCurTerminalDto(
      Pageable pageable, NoRelaCurTerminalDto dto) {
    Validate.notNull(dto, "查询参数缺失");
    Boolean f =
        StringUtils.isBlank(dto.getUserName())
            && StringUtils.isBlank(dto.getCustomerUserName())
            && StringUtils.isBlank(dto.getCustomerCode());
    Validate.isTrue(!f, "客户编码、客户用户账户、企业用户账户参数不能都为空");

    // 如果客户编码不为空对应是客户关联终端查询逻辑
    if (StringUtils.isNotBlank(dto.getCustomerCode())) {
      dto.setSupplyType(StringUtils.EMPTY);
      dto.setCustomerUserName(StringUtils.EMPTY);
      dto.setUserName(StringUtils.EMPTY);
    }

    // 如果客户用户账号不为对应客户用户关联终端查询逻辑
    if (StringUtils.isNotBlank(dto.getCustomerUserName())) {
      dto.setSupplyType(TerminalSupplyTypeEnum.CUSTOMER_USER.getValue());
      dto.setCustomerCode(StringUtils.EMPTY);
      dto.setUserName(StringUtils.EMPTY);
    }
    // 如果企业用户账号不为对应客户用户关联终端查询逻辑
    if (StringUtils.isNotBlank(dto.getUserName())) {
      dto.setSupplyType(TerminalSupplyTypeEnum.DEFAULT.getValue());
      dto.setCustomerCode(StringUtils.EMPTY);
      dto.setCustomerUserName(StringUtils.EMPTY);
    }
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    Page<TerminalReportVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<TerminalReportVo> pageResult =
        terminalReportVoRepository.findByNoRelaCurTerminalDto(page, dto);
    this.initTerminalPageExtInfo(pageResult);
    this.buildPositionMsg(pageResult.getRecords());
    return pageResult;
  }

  @Override
  public Page<TerminalReportVo> findByNoRelaTerminalDto(Pageable pageable, NoRelaTerminalDto dto) {
    Validate.notNull(dto, "查询参数缺失");
    Boolean f =
        StringUtils.isBlank(dto.getUserName())
            && StringUtils.isBlank(dto.getCustomerUserName())
            && StringUtils.isBlank(dto.getCustomerCode());
    Validate.isTrue(!f, "客户编码、客户用户账户、企业用户账户参数不能都为空");

    // 如果客户编码不为空对应是客户关联终端查询逻辑
    if (StringUtils.isNotBlank(dto.getCustomerCode())) {
      dto.setSupplyType(StringUtils.EMPTY);
      dto.setCustomerUserName(StringUtils.EMPTY);
      dto.setUserName(StringUtils.EMPTY);
    }

    // 如果客户用户账号不为对应客户用户关联终端查询逻辑
    if (StringUtils.isNotBlank(dto.getCustomerUserName())) {
      dto.setSupplyType(TerminalSupplyTypeEnum.CUSTOMER_USER.getValue());
      dto.setCustomerCode(StringUtils.EMPTY);
      dto.setUserName(StringUtils.EMPTY);
    }
    // 如果企业用户账号不为对应客户用户关联终端查询逻辑
    if (StringUtils.isNotBlank(dto.getUserName())) {
      dto.setSupplyType(TerminalSupplyTypeEnum.DEFAULT.getValue());
      dto.setCustomerCode(StringUtils.EMPTY);
      dto.setCustomerUserName(StringUtils.EMPTY);
    }
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    Page<TerminalReportVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<TerminalReportVo> pageResult =
        terminalReportVoRepository.findByNoRelaTerminalDto(page, dto);
    this.initTerminalPageExtInfo(pageResult);
    return pageResult;
  }

  @Override
  public Page<TerminalReportVo> findByRelaCurTerminalDto(
      Pageable pageable, RelaCurTerminalDto dto) {
    Validate.notNull(dto, "查询参数缺失");
    Boolean f =
        StringUtils.isBlank(dto.getUserName())
            && StringUtils.isBlank(dto.getCustomerUserName())
            && StringUtils.isBlank(dto.getCustomerCode());
    Validate.isTrue(!f, "客户编码、客户用户账户、企业用户账户参数不能都为空");

    // 如果客户编码不为空对应是客户关联终端查询逻辑
    if (StringUtils.isNotBlank(dto.getCustomerCode())) {
      dto.setSupplyType(StringUtils.EMPTY);
      dto.setCustomerUserName(StringUtils.EMPTY);
      dto.setUserName(StringUtils.EMPTY);
    }

    // 如果客户用户账号不为对应客户用户关联终端查询逻辑
    if (StringUtils.isNotBlank(dto.getCustomerUserName())) {
      dto.setSupplyType(TerminalSupplyTypeEnum.CUSTOMER_USER.getValue());
      dto.setCustomerCode(StringUtils.EMPTY);
      dto.setUserName(StringUtils.EMPTY);
    }
    // 如果企业用户账号不为对应客户用户关联终端查询逻辑
    if (StringUtils.isNotBlank(dto.getUserName())) {
      dto.setSupplyType(TerminalSupplyTypeEnum.DEFAULT.getValue());
      dto.setCustomerCode(StringUtils.EMPTY);
      dto.setCustomerUserName(StringUtils.EMPTY);
    }
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());

    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    Page<TerminalReportVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<TerminalReportVo> pageResult =
        terminalReportVoRepository.findByRelaCurTerminalDto(page, dto);
    this.initTerminalPageExtInfo(pageResult);
    this.buildPositionMsg(pageResult.getRecords());
    this.buildTerminalContactInfo(pageResult.getRecords());
    return pageResult;
  }

  /**
   * 组装职位信息
   *
   * @param records
   */
  private List<TerminalReportVo> buildPositionMsg(List<TerminalReportVo> records) {
    if (CollectionUtils.isEmpty(records)) {
      return new ArrayList<>(0);
    }
    List<String> terminalCodes =
        records.stream()
            .distinct()
            .map(TerminalReportVo::getTerminalCode)
            .collect(Collectors.toList());
    List<TerminalReportVo> terminalReportVos =
        terminalReportVoRepository.findByTerminalCodes(terminalCodes);
    if (CollectionUtils.isEmpty(terminalReportVos)) {
      return records;
    }
    Map<String, TerminalReportVo> terminalReportVoMap = new HashMap<>();
    Map<String, List<TerminalReportVo>> collect = terminalReportVos.stream()
        .filter(o -> StringUtils.isNotBlank(o.getPositionCode()))
        .filter(o -> StringUtils.isNotBlank(o.getPositionName()))
        .collect(Collectors.groupingBy(TerminalReportVo::getTerminalCode));
    collect.forEach((k, v) -> {
      String code = "";
      String name = "";
      TerminalReportVo terminalReportVo = new TerminalReportVo();
      if (!CollectionUtils.isEmpty(v)) {
        for (TerminalReportVo vo : v) {
          if (StringUtils.isBlank(code)) {
            code = code.concat(vo.getPositionCode());
            name = name.concat(vo.getPositionName());
          } else {
            code = code.concat(",").concat(vo.getPositionCode());
            name = name.concat(",").concat(vo.getPositionName());
          }
        }
      }
      terminalReportVo.setPositionCode(code);
      terminalReportVo.setPositionName(name);
      terminalReportVoMap.put(k, terminalReportVo);
    });
    for (TerminalReportVo record : records) {
      if (terminalReportVoMap.containsKey(record.getTerminalCode())) {
        TerminalReportVo vo = terminalReportVoMap.get(record.getTerminalCode());
        record.setPositionCode(vo.getPositionCode());
        record.setPositionName(vo.getPositionName());
      }
    }
    return records;
  }

  /**
   * 组装终端组织和客户组织信息
   *
   * @param pageResult
   */
  private void initTerminalPageExtInfo(Page<TerminalReportVo> pageResult) {
    if (Objects.isNull(pageResult) || CollectionUtils.isEmpty(pageResult.getRecords())) {
      return;
    }

    Set<String> terminalCodeSet =
        pageResult.getRecords().stream()
            .filter(a -> StringUtils.isNotBlank(a.getTerminalCode()))
            .map(TerminalReportVo::getTerminalCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(terminalCodeSet)) {
      return;
    }
    List<TerminalRelaOrg> terminalRelaOrgList =
        this.terminalReportVoRepository.findTerminalRelaOrgListByTerminalCodes(terminalCodeSet);

    List<TerminalRelaCustomerOrg> terminalRelaCustomerOrgList =
        this.terminalReportVoRepository.findTerminalRelaCustomerOrgListByTerminalCodes(
            terminalCodeSet);

    Map<String, String> mapOrg = Maps.newHashMap();
    Map<String, String> mapCustomerOrg = Maps.newHashMap();

    if (!CollectionUtils.isEmpty(terminalRelaOrgList)) {
      mapOrg =
          terminalRelaOrgList.stream()
              .filter(a -> StringUtils.isNoneBlank(a.getTerminalCode(), a.getOrgName()))
              .collect(
                  Collectors.groupingBy(
                      TerminalRelaOrg::getTerminalCode,
                      Collectors.mapping(TerminalRelaOrg::getOrgName, Collectors.joining(","))));
    }
    if (!CollectionUtils.isEmpty(terminalRelaCustomerOrgList)) {
      mapCustomerOrg =
          terminalRelaCustomerOrgList.stream()
              .filter(a -> StringUtils.isNoneBlank(a.getTerminalCode(), a.getOrgName()))
              .collect(
                  Collectors.groupingBy(
                      TerminalRelaCustomerOrg::getTerminalCode,
                      Collectors.mapping(
                          TerminalRelaCustomerOrg::getOrgName, Collectors.joining(","))));
    }
    for (TerminalReportVo record : pageResult.getRecords()) {
      record.setOrgName(mapOrg.get(record.getTerminalCode()));
      record.setCustomerOrgName(mapCustomerOrg.get(record.getTerminalCode()));
    }
  }

  /**
   * 构建终端联系人信息
   *
   * @param records 终端列表
   */
  private void buildTerminalContactInfo(List<TerminalReportVo> records) {
    if (CollectionUtils.isEmpty(records)) {
      return;
    }
    Map<String, TerminalReportVo> terminalMap = records.stream().collect(Collectors.toMap(TerminalReportVo::getTerminalCode, t -> t, (a, b) -> b));
    List<TerminalContact> terminalContacts = this.terminalContactService.findByTerminalCodes(Lists.newArrayList(terminalMap.keySet()));
    if (CollectionUtils.isEmpty(terminalContacts)) {
      return;
    }
    terminalContacts.forEach(terminalContact -> {
      TerminalReportVo terminalReportVo = terminalMap.get(terminalContact.getTerminalCode());
      if (Objects.isNull(terminalReportVo)) {
        return;
      }
      TerminalContactReportVo terminalContactVo = this.nebulaToolkitService.copyObjectByWhiteList(terminalContact, TerminalContactReportVo.class, HashSet.class, ArrayList.class);
      if (CollectionUtils.isEmpty(terminalReportVo.getContacts())) {
        terminalReportVo.setContacts(Lists.newArrayList());
      }
      terminalReportVo.getContacts().add(terminalContactVo);
    });
  }
}
