package com.biz.crm.mdm.admin.web.dataview.customer;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mdm.business.customer.org.local.entity.CustomerOrg;
import com.biz.crm.mdm.business.customer.org.local.repository.CustomerOrgRepository;
import com.biz.crm.mdm.business.region.local.entity.Region;
import com.biz.crm.mdm.business.region.local.repository.RegionRepository;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.europa.database.sdk.strategy.ExternalQueryInterceptor;
import com.bizunited.nebula.europa.sdk.context.execute.ExecuteContent;
import com.bizunited.nebula.europa.sdk.context.matedata.MetaData;
import com.bizunited.nebula.europa.sdk.vo.EuropaInfoVo;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import javax.persistence.EntityManager;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author XXLsansui
 * @date 2023-02-01 9:27
 * @description： MDM客户信息数据视图补充
 */
@Component
public class CustomerQueryInterceptor implements ExternalQueryInterceptor {

  @Autowired
  private RegionRepository regionRepository;
  @Autowired
  private CustomerOrgRepository customerOrgRepository;

  private static final String DISTRICT_CODE = "district_code";
  private static final String CITY_CODE = "city_code";
  private static final String PROVINCE_CODE = "province_code";
  private static final String CUSTOMER_ORG_CODE = "customer_org_code";


    /**
     * 扩展查询拦截器的编码
     *
     * @return
     */
  @Override
  public String code() {
      return "mdm_customer_queryInterceptor";
  }

    /**
     * 扩展查询拦截器的
     */
  @Override
  public String name() {
    return "MDM客户信息-关联省市区和组织信息补充";
  }

    /**
     * 该方法正是查询拦截器被触发的第二个场景，即在执行完数据视图正式查询过程，并正确产生了“元数据”结构（MetaData）、查询内容（ExecuteContent）后</p>
     * 这个时候，开发人员不但可以干预元数据结构的信息，还可以对查询内容进行干预，
     * 例如添加查询列，或者修改某个查询单元格中的返回信息，等等
     *
     * @param currentEntityManager 当前数据库执行正在使用的连接管理器。情尽可能使用该连接管理器进行查询。
     * @param metadata
     * @param europaInfoVo
     * @param queryContent
     * @param fieldNames           数据视图正式执行时，运行该扩展查询拦截器的字段名。为什么会是数组呢？因为在一个数据视图上，可能有两个或者多个扩展字段绑定了同一个扩展查询拦截器
     * @return 请注意返回值，是一个对象数组集合。这个每一个对象数组的查询 等于  fieldNames 数组的长度；
     * 也就是说如果fieldNames为A，B；两个字段名，那么每一个Object[]长度就应该为2，其中0号索引位的值就是某行A字段的值；1号索引位的值就是某行B字段的值</p>
     * <p>
     * 另外还要注意，返回的扩展字段查询结果的每一行数据的顺序，应该和queryContent中已有行数据的顺序保持一致
     */
  @Override
  public List<Object[]> process(EntityManager currentEntityManager, MetaData metadata, EuropaInfoVo europaInfoVo, ExecuteContent queryContent, String... fieldNames) {
    List<Map<String, Object>> results = queryContent.getResults();
    if (CollectionUtils.isEmpty(results)) {
      return null;
    }
    if (0 == results.parallelStream().filter(row -> row != null).count()) {
      return Lists.newLinkedList();
    }
    List<String> districtCodes = Lists.newArrayList();
    List<String> cityCodes = Lists.newArrayList();
    List<String> provinceCodes = Lists.newArrayList();
    List<String> customerOrgCodes = Lists.newArrayList();
    results.forEach(map -> {
      //拿到当前页面数据 省市区 所属组织编码
      if(map.containsKey(PROVINCE_CODE)){
        provinceCodes.add((String) map.get(PROVINCE_CODE));
      }
      if(map.containsKey(CITY_CODE)){
        cityCodes.add((String) map.get(CITY_CODE));
      }
      if(map.containsKey(DISTRICT_CODE)){
            districtCodes.add((String) map.get(DISTRICT_CODE));
      }
      if(map.containsKey(CUSTOMER_ORG_CODE)){
        customerOrgCodes.add((String) map.get(CUSTOMER_ORG_CODE));
      }
    });

    // 查询省市区 customerCode 通过省市区编码查询名字
    List<Region> provinceList = regionRepository.lambdaQuery()
            .eq(Region::getTenantCode, TenantUtils.getTenantCode())
            .eq(Region::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .in(Region::getRegionCode, provinceCodes)
            .list();
    Map<String, String> provinceNameMap = provinceList.stream().collect(Collectors.toMap(Region::getRegionCode, Region::getRegionName, (k1, k2)->k2));
    List<Region> cityList = regionRepository.lambdaQuery()
            .eq(Region::getTenantCode, TenantUtils.getTenantCode())
            .eq(Region::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .in(Region::getRegionCode, cityCodes)
            .list();
    Map<String, String> cityNameMap = cityList.stream().collect(Collectors.toMap(Region::getRegionCode, Region::getRegionName, (k1, k2)->k2));
    List<Region> districtList = regionRepository.lambdaQuery()
            .eq(Region::getTenantCode, TenantUtils.getTenantCode())
            .eq(Region::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .in(Region::getRegionCode, districtCodes)
            .list();
    Map<String, String> districtNameMap = districtList.stream().collect(Collectors.toMap(Region::getRegionCode, Region::getRegionName, (k1, k2)->k2));

    List<CustomerOrg> customerOrgList = customerOrgRepository.findListByCodes(customerOrgCodes, TenantUtils.getTenantCode());
    Map<String, String> customerOrgMap = customerOrgList.stream().collect(Collectors.toMap(CustomerOrg::getCustomerOrgCode, CustomerOrg::getCustomerOrgName, (k1, k2)->k2));

    //组装客户信息
    List<Object[]> externalContents = Lists.newArrayList();
    for (int index = 0; index < results.size(); index++) {
      List<Object> itemList = Lists.newArrayList();
      //拿取当前行标识
      String provinceCode = (String) results.get(index).get(PROVINCE_CODE);
      String cityCode = (String) results.get(index).get(CITY_CODE);
      String districtCode = (String) results.get(index).get(DISTRICT_CODE);
      String customerOrgCode = (String) results.get(index).get(CUSTOMER_ORG_CODE);

      // 匹配map数据
      String provinceRegion = provinceNameMap.get(provinceCode);
      String cityRegion = cityNameMap.get(cityCode);
      String districtRegion = districtNameMap.get(districtCode);
      String customerOrgName = customerOrgMap.get(customerOrgCode);

      for (String externalFieldName : fieldNames) {
        if (StringUtils.equals(externalFieldName, "province_name")) {
          itemList.add(ObjectUtils.isEmpty(provinceRegion) ? "" : provinceRegion);
        }
        if (StringUtils.equals(externalFieldName, "city_name")) {
          itemList.add(ObjectUtils.isEmpty(cityRegion) ? "" : cityRegion);
        }
        if (StringUtils.equals(externalFieldName, "district_name")) {
          itemList.add(ObjectUtils.isEmpty(districtRegion) ? "" : districtRegion);
        }
        if (StringUtils.equals(externalFieldName, "customer_org_name")) {
          itemList.add(ObjectUtils.isEmpty(customerOrgName) ? "" : customerOrgName);
        }
      }
      externalContents.add(itemList.toArray(new Object[]{}));
    }

    return externalContents;
  }
}
