package com.biz.crm.mdm.admin.web.imports.channel.org.service;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.admin.web.imports.channel.org.model.ChannelOrgRelationTerminalImportVo;
import com.biz.crm.mdm.business.channel.org.relation.local.entity.ChannelOrgRelationTerminal;
import com.biz.crm.mdm.business.channel.org.relation.local.repository.ChannelOrgRelationTerminalRepository;
import com.biz.crm.mdm.business.channel.org.sdk.service.ChannelOrgVoService;
import com.biz.crm.mdm.business.channel.org.sdk.vo.ChannelOrgVo;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

/**
 * 客户组织关联门店导入功能
 *
 * @author pengxi
 */
@Component
public class CustomerOrgRelationTerminalImportProcess implements ImportProcess<ChannelOrgRelationTerminalImportVo> {

  @Autowired(required = false)
  private ChannelOrgVoService channelOrgVoService;
  @Autowired(required = false)
  private TerminalVoService terminalVoService;
  @Autowired(required = false)
  private ChannelOrgRelationTerminalRepository terminalRepository;

  @Override
  public Integer getBatchCount() {
    return 10000;
  }

  @Override
  @Transactional
  public Map<Integer, String> execute(LinkedHashMap<Integer, ChannelOrgRelationTerminalImportVo> data,
      TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
    if (MapUtils.isEmpty(data)) {
      return null;
    }
    Map<Integer, String> resultMap = Maps.newConcurrentMap();
    // 1、检查表格原始数据是否为空
    this.validateOriginalData(resultMap, data);
    if (MapUtils.isNotEmpty(resultMap)) {
      return resultMap;
    }
    // 2、检查数据有效性
    this.validateEffective(resultMap, data);
    if (MapUtils.isNotEmpty(resultMap)) {
      return resultMap;
    }
    // 3、
    this.saveNotExist(data);
    return null;
  }

  /**
   * 检查表格原始数据是否为空
   */
  private void validateOriginalData(Map<Integer, String> resultMap, LinkedHashMap<Integer, ChannelOrgRelationTerminalImportVo> data) {
    data.forEach((index, importVo)->{
      StringBuilder item = new StringBuilder();
      if (StringUtils.isBlank(importVo.getChannelOrgCode())) {
        item.append("L").append(index).append("渠道组织编码不能为空！");
      }
      if (StringUtils.isBlank(importVo.getChannelOrgName())) {
        //item.append("L").append(index).append("渠道组织名称不能为空！");
      }
      if (StringUtils.isBlank(importVo.getTerminalCode())) {
        item.append("L").append(index).append("终端编码不能为空！");
      }
      if (StringUtils.isBlank(importVo.getTerminalName())) {
        //item.append("L").append(index).append("终端名称不能为空！");
      }
      if (StringUtils.isBlank(item.toString())) {
        return;
      }
      resultMap.put(index, item.toString());
    });
  }

  /**
   * 检查表格原始数据是否有效
   */
  private void validateEffective(Map<Integer, String> resultMap, LinkedHashMap<Integer, ChannelOrgRelationTerminalImportVo> data) {
    Collection<ChannelOrgRelationTerminalImportVo> relationVos = data.values();
    List<String> customerOrgCodes = relationVos.stream().map(ChannelOrgRelationTerminalImportVo::getChannelOrgCode).distinct().collect(Collectors.toList());
    List<String> terminalCodes = relationVos.stream().map(ChannelOrgRelationTerminalImportVo::getTerminalCode).distinct().collect(Collectors.toList());
    // 查询渠道组织
    List<ChannelOrgVo> channelOrgVos = this.channelOrgVoService.findByChannelOrgCodes(customerOrgCodes);
    Validate.notEmpty(channelOrgVos, "渠道组织不存在！");
    Map<String, ChannelOrgVo> channelOrgVoMap = Optional.of(channelOrgVos).orElse(Lists.newArrayList())
            .stream().collect(Collectors.toMap(ChannelOrgVo::getChannelOrgCode, Function.identity()));
    // 查询终端信息
    List<TerminalVo> terminalVos = this.terminalVoService.findTerminalAndContactByTerminalCodes(terminalCodes);
    Validate.notEmpty(terminalVos, "终端信息不存在！");
    Map<String, TerminalVo> terminalVoMap = Optional.of(terminalVos).orElse(Lists.newArrayList())
            .stream().collect(Collectors.toMap(TerminalVo::getTerminalCode, Function.identity()));
    data.forEach((index, importVo)->{
      StringBuilder item = new StringBuilder();
      ChannelOrgVo channelOrgVo = channelOrgVoMap.get(importVo.getChannelOrgCode());
      if (null == channelOrgVo) {
        item.append("L").append(index).append("渠道组织不存在！");
      } else {
        importVo.setChannelOrgName(channelOrgVo.getChannelOrgName());
        importVo.setChannelOrgRuleCode(channelOrgVo.getRuleCode());
      }
      TerminalVo terminalVo = terminalVoMap.get(importVo.getTerminalCode());
      if (null == terminalVo) {
        item.append("L").append(index).append("终端信息不存在！");
      } else {
        importVo.setTerminalName(terminalVo.getTerminalName());
      }
      if (StringUtils.isBlank(item.toString())) {
        return;
      }
      resultMap.put(index, item.toString());
    });
  }

  /**
   * 只保存不存在的
   */
  private void saveNotExist(LinkedHashMap<Integer, ChannelOrgRelationTerminalImportVo> data) {
    Collection<ChannelOrgRelationTerminalImportVo> relationVos = data.values();
    List<ChannelOrgRelationTerminal> oldEntities = this.terminalRepository.findByTenantCodeAndTerminalCodes(TenantUtils.getTenantCode(), relationVos.stream().map(ChannelOrgRelationTerminalImportVo::getTerminalCode).distinct().collect(Collectors.toList()));
    List<ChannelOrgRelationTerminal> entities = new ArrayList<>(data.size());
    String tenantCode = TenantUtils.getTenantCode();
    relationVos.forEach(vo->{
      if (CollectionUtils.isNotEmpty(oldEntities)) {
        Optional<ChannelOrgRelationTerminal> optional = oldEntities.stream()
                .filter(o->tenantCode.equals(o.getTenantCode())
                        && o.getChannelOrgCode().equals(vo.getChannelOrgCode())
                        && o.getTerminalCode().equals(vo.getTerminalCode()))
                .findAny();
        if (optional.isPresent()) {
          // 已存在关联跳过
          return;
        }
      }
      ChannelOrgRelationTerminal entity = new ChannelOrgRelationTerminal();
      entity.setTenantCode(tenantCode);
      entity.setChannelOrgCode(vo.getChannelOrgCode());
      entity.setTerminalCode(vo.getTerminalCode());
      entity.setTerminalName(vo.getTerminalName());
      entity.setChannelOrgName(vo.getChannelOrgName());
      entity.setChannelOrgRuleCode(vo.getChannelOrgRuleCode());
      entities.add(entity);
    });
    this.terminalRepository.saveBatch(entities);
  }

  @Override
  public Class<ChannelOrgRelationTerminalImportVo> findCrmExcelVoClass() {
    return ChannelOrgRelationTerminalImportVo.class;
  }

  @Override
  public String getTemplateCode() {
    return "MDM_CHANNEL_ORG_RELATION_TERMINAL_IMPORT";
  }

  @Override
  public String getTemplateName() {
    return "MDM渠道组织关联门店导入";
  }
}
