package com.biz.crm.mdm.admin.web.imports.price.service;

import java.util.*;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.admin.web.imports.price.modle.PriceProCrmImportVo;
import com.biz.crm.mdm.business.price.local.entity.Price;
import com.biz.crm.mdm.business.price.local.entity.PriceDimension;
import com.biz.crm.mdm.business.price.local.entity.PriceType;
import com.biz.crm.mdm.business.price.local.service.PriceService;
import com.biz.crm.mdm.business.price.local.service.PriceTypeService;
import com.biz.crm.mdm.business.price.sdk.constant.PriceConstant;
import com.biz.crm.mdm.business.price.sdk.enums.PriceDimensionEnum;
import com.biz.crm.mdm.business.price.sdk.register.PriceDimensionRegister;
import com.biz.crm.mdm.business.product.local.entity.Product;
import com.biz.crm.mdm.business.product.local.service.ProductService;
import com.bizunited.nebula.common.service.NebulaToolkitService;

import lombok.extern.slf4j.Slf4j;

/**
 * 价格维护导入 维度：商品
 * @author HanJiaJun
 * @describe:
 * @createTime 2022年05月24日 14:30:00
 */
@Component
@Slf4j
public class PriceProImportProcess implements ImportProcess<PriceProCrmImportVo> {

  @Autowired(required = false)
  private PriceService priceService;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private List<PriceDimensionRegister> priceDimensionRegisters;

  @Autowired(required = false)
  private ProductService productService;

  @Autowired(required = false)
  private PriceTypeService priceTypeService;

  @Override
  @Transactional
  public Map<Integer, String> execute(
      LinkedHashMap<Integer, PriceProCrmImportVo> data,
      TaskGlobalParamsVo paramsVo,
      Map<String, Object> params) {
    final Optional<PriceProCrmImportVo> first = data.values().stream().findFirst();
    if (!first.isPresent()) {
      return null;
    }
    PriceProCrmImportVo vo = first.get();
    this.validate(vo, params);
    this.execute(vo, params);
    return null;
  }

  /**
   * 校验
   * @param data
   * @param params
   */
  private void validate(PriceProCrmImportVo vo, Map<String, Object> params) {
    boolean f = StringUtils.isNoneBlank(vo.getTypeCode(),  vo.getProductCode());
    Validate.isTrue(f,"价格类型编码、商品编码不能为空");
    Validate.notNull(vo.getPrice(),"价格不能为空");
    Validate.notNull(vo.getBeginTime(),"开始时间不能为空");
    Validate.notNull(vo.getEndTime(),"结束时间不能为空");
    Validate.isTrue(vo.getBeginTime().compareTo(vo.getEndTime())<0,"开始时间必须大于结束时间");
    //校验价格类型
    PriceType detailByTypeCode = priceTypeService.findDetailByTypeCode(vo.getTypeCode());
    Validate.notNull(detailByTypeCode,"尚无当前价格类型编码");
    //设置价格维度编码
    HashMap<Integer, PriceDimensionEnum> map = new HashMap<>();
    for (PriceDimensionRegister priceDimensionRegister : priceDimensionRegisters) {
      String code = priceDimensionRegister.getCode();
      if (PriceDimensionEnum.PRODUCT.getDictCode().equals(code)) {
        map.put(priceDimensionRegister.sort(), PriceDimensionEnum.PRODUCT);
      }
    }
    Validate.notEmpty(map, "未查询到价格维度");
    List<Integer> collect = map.keySet().stream().sorted().collect(Collectors.toList());
    String typeCode = vo.getTypeCode();
    String typeName = "";
    //拼接
    for (Integer integer : collect) {
      PriceDimensionEnum priceDimensionEnum = map.get(integer);
      typeCode = typeCode.concat(PriceConstant.SEPARATOR).concat(priceDimensionEnum.getDictCode());
      if (StringUtils.isBlank(typeName)){
        typeName=typeName.concat(priceDimensionEnum.getValue());
      }else {
        typeName = typeName.concat(PriceConstant.NAME_SEPARATOR).concat(priceDimensionEnum.getValue());
      }
    }
    //设置价格维度编码
    vo.setTypeDetailCode(typeCode);
    vo.setTypeDetailName(typeName);
  }

  /**
   * 执行
   * @param params
   */
  private void execute(PriceProCrmImportVo vo, Map<String, Object> params) {
    Price price = this.nebulaToolkitService.copyObjectByWhiteList(vo, Price.class, HashSet.class, ArrayList.class);
    //设置价格维度
    List<PriceDimension> dimensionList =new ArrayList<>();
    PriceDimension product =new PriceDimension();
    product.setDimensionCode(PriceDimensionEnum.PRODUCT.getDictCode());
    product.setRelateCode(vo.getProductCode());
    //追加商品信息
    addProductMsg(product);
    dimensionList.add(product);
    price.setDimensionList(dimensionList);
    this.priceService.create(price);
  }

  /**
   * 添加商品信息
   * @param product
   */
  private void addProductMsg(PriceDimension product) {
    Product byProductCode = this.productService.findByProductCode(product.getRelateCode());
    Validate.notNull(byProductCode,"["+product.getRelateCode()+"未查询到商品信息");
    product.setRelateName(byProductCode.getProductName());
  }


  @Override
  public Class findCrmExcelVoClass() {
    return PriceProCrmImportVo.class;
  }

  @Override
  public String getBusinessCode() {
    return "MDM_PRICE_IMPORT";
  }

  @Override
  public String getBusinessName() {
    return "MDM价格维护导入";
  }

  @Override
  public String getTemplateCode() {
    return "MDM_PRICE_PRODUCT_IMPORT";
  }

  @Override
  public String getTemplateName() {
    return "MDM价格维护导入，商品维度";
  }
}
