package com.biz.crm.mdm.admin.web.filter;

import com.biz.crm.mdm.admin.web.strategy.LoginLimitStrategy;
import com.bizunited.nebula.common.controller.model.ResponseCode;
import com.bizunited.nebula.common.controller.model.ResponseModel;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.security.local.loginform.NebulaSecurityLoginFormDetailsSource;
import com.bizunited.nebula.security.sdk.config.SimpleSecurityProperties;
import com.bizunited.nebula.security.sdk.loginform.LoginDetails;
import com.google.common.collect.Lists;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.security.web.util.matcher.OrRequestMatcher;
import org.springframework.security.web.util.matcher.RequestMatcher;
import org.springframework.web.filter.OncePerRequestFilter;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;

import java.util.List;

/**
 * 描述：登录限制拦截去</br>
 * 提供对登录错误次数的显示以及锁定时间的限制
 * 提供在锁定时间后进行状态基接触的功能
 *
 * @author keller
 * @date 2022/10/19
 */
public class LoginLimitFilter extends OncePerRequestFilter {
  /**
   * 用户身份转换
   */
  @Autowired
  private NebulaSecurityLoginFormDetailsSource nebulaSecurityLoginFormDetailsSource;
  /**
   * 登录限制策略
   */
  @Autowired(required = false)
  private List<LoginLimitStrategy> loginLimitStrategies;
  @Autowired
  private SimpleSecurityProperties simpleSecurityProperties;
  /**
   * jwt header头
   */
  private final String JWT_HEADER = "jwt";

  @Override
  protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
    String jwt = request.getHeader(JWT_HEADER);
    // 已经存在登录信息直接忽略
    if (StringUtils.isNotBlank(jwt)) {
      filterChain.doFilter(request, response);
      return;
    }
    // 未配置策略情况下直接忽略
    if (CollectionUtils.isEmpty(this.loginLimitStrategies)) {
      filterChain.doFilter(request, response);
      return;
    }
    if (!CollectionUtils.isEmpty(this.loginLimitStrategies)) {
      /*
       * 进行登录限制检查可以抽象出来作为各种检查策略，如进行登录次数错误验证、黑名单等
       */
      LoginDetails loginDetails = nebulaSecurityLoginFormDetailsSource.buildDetails(request);
      try {
        this.loginLimitStrategies.forEach(loginLimitStrategy -> loginLimitStrategy.handle(loginDetails));
      } catch (RuntimeException e) {
        this.write(response, e.getMessage());
        return;
      }
      filterChain.doFilter(request, response);
    }
  }

  /**
   * 直接输出错误信息
   *
   * @param response
   * @param msg
   * @throws IOException
   */
  private void write(HttpServletResponse response, String msg) throws IOException {
    if (StringUtils.isBlank(msg)) {
      ResponseModel result = new ResponseModel(System.currentTimeMillis(), null, ResponseCode.E603, new IllegalAccessException("账号已经被锁定，请联系管理员！"));
      response.setContentType("application/json;charset=UTF-8");
      response.getWriter().write(JsonUtils.obj2JsonString(result));
    } else {
      ResponseModel result = new ResponseModel(System.currentTimeMillis(), null, ResponseCode.E603, new IllegalAccessException(msg));
      response.setContentType("application/json;charset=UTF-8");
      response.getWriter().write(JsonUtils.obj2JsonString(result));
    }
  }

  @Override
  protected boolean shouldNotFilter(HttpServletRequest request) throws ServletException {
    String[] ignoreUrls = simpleSecurityProperties.getIgnoreUrls();
    if (!ArrayUtils.isEmpty(ignoreUrls)) {
      List<RequestMatcher> ignoreUrlmatchers = Lists.newArrayList();
      for (String pattern : ignoreUrls) {
        ignoreUrlmatchers.add(new AntPathRequestMatcher(pattern, null));
        OrRequestMatcher matcher = new OrRequestMatcher(ignoreUrlmatchers);
        if (matcher.matches(request)) {
          return true;
        }
      }
    }
    return false;
  }
}