package com.biz.crm.mdm.admin.web.report.spu.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.mdm.admin.web.report.spu.dto.VisitorProductSpuDto;
import com.biz.crm.mdm.admin.web.report.spu.dto.VisitorProductSpuPaginationDto;
import com.biz.crm.mdm.admin.web.report.spu.repository.VisitorProductSpuVoRepository;
import com.biz.crm.mdm.admin.web.report.spu.service.VisitorProductSpuVoService;
import com.biz.crm.mdm.business.price.sdk.dto.FindPriceDto;
import com.biz.crm.mdm.business.price.sdk.enums.FindPriceUserTypeEnum;
import com.biz.crm.mdm.business.price.sdk.enums.PriceDimensionEnum;
import com.biz.crm.mdm.business.price.sdk.service.PriceModelVoService;
import com.biz.crm.mdm.business.price.sdk.vo.PriceModelVo;
import com.biz.crm.mdm.business.product.level.sdk.service.ProductLevelVoSdkService;
import com.biz.crm.mdm.business.product.level.sdk.vo.ProductLevelVo;
import com.biz.crm.mdm.business.product.spu.local.entity.ProductSpu;
import com.biz.crm.mdm.business.product.spu.local.entity.ProductSpuMedia;
import com.biz.crm.mdm.business.product.spu.local.entity.ProductSpuRelateSku;
import com.biz.crm.mdm.business.product.spu.local.repository.ProductSpuMediaRepository;
import com.biz.crm.mdm.business.product.spu.local.repository.ProductSpuRelateSkuRepository;
import com.biz.crm.mdm.business.product.spu.local.repository.ProductSpuRepository;
import com.biz.crm.mdm.business.product.spu.sdk.enums.MediaTypeEnum;
import com.biz.crm.mdm.business.product.spu.sdk.vo.ProductSpuMediaVo;
import com.biz.crm.mdm.business.product.spu.sdk.vo.ProductSpuRelateSkuVo;
import com.biz.crm.mdm.business.product.spu.sdk.vo.ProductSpuVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * spu查询服务实现类
 *
 * @author pengxi
 * @date 2022/9/9
 */
@Slf4j
@Service
public class VisitorProductSpuVoServiceImpl implements VisitorProductSpuVoService {

  @Autowired(required = false)
  private VisitorProductSpuVoRepository visitorProductSpuVoRepository;

  @Autowired(required = false)
  private ProductLevelVoSdkService productLevelVoSdkService;

  @Autowired(required = false)
  private ProductSpuRepository productSpuRepository;

  @Autowired(required = false)
  private ProductSpuRelateSkuRepository productSpuRelateSkuRepository;

  @Autowired(required = false)
  private ProductSpuMediaRepository productSpuMediaRepository;

  @Autowired(required = false)
  private PriceModelVoService priceModelVoService;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public Page<ProductSpuVo> findByConditions(Pageable pageable, VisitorProductSpuPaginationDto dto) {
    Set<String> productCodeSet = this.findProductCodeSet(dto.getKeyword(), dto.getLikeKeyword(), dto.getProductLevelCode(), dto.getProductCodeList());
    if (CollectionUtils.isEmpty(productCodeSet)) {
      return new Page<>();
    }
    // 优先取dto对象中的分页参数
    pageable = PageRequest.of(
        Optional.ofNullable(dto.getPage()).orElse(pageable.getPageNumber()),
        Optional.ofNullable(dto.getPageSize()).orElse(pageable.getPageSize()));
    Page<VisitorProductSpuDto> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    VisitorProductSpuDto visitorProductSpuDto = new VisitorProductSpuDto();
    visitorProductSpuDto.setKeyword(dto.getKeyword());
    visitorProductSpuDto.setLikeKeyword(dto.getLikeKeyword());
    visitorProductSpuDto.setSpuTagList(dto.getSpuTagList());
    visitorProductSpuDto.setProductCodeList(Lists.newArrayList(productCodeSet));
    Page<String> spuCodePage = this.visitorProductSpuVoRepository.findSpuCodePage(page, visitorProductSpuDto);
    Page<ProductSpuVo> productSpuVoPage = new Page<>();
    if (CollectionUtils.isEmpty(spuCodePage.getRecords())) {
      return productSpuVoPage;
    }
    List<ProductSpuVo> spuVoList = this.findProductSpu(dto, spuCodePage.getRecords(), productCodeSet);
    productSpuVoPage.setRecords(spuVoList);
    productSpuVoPage.setSize(spuCodePage.getSize());
    productSpuVoPage.setTotal(spuCodePage.getTotal());
    return productSpuVoPage;
  }

  /**
   * 获取spu信息、spu对应sku询价、促销标记
   *
   * @param dto
   * @param spuCodeList
   * @return
   */
  private List<ProductSpuVo> findProductSpu(VisitorProductSpuPaginationDto dto, List<String> spuCodeList, Set<String> productCodeSet) {
    if (Objects.isNull(dto)
        || CollectionUtils.isEmpty(spuCodeList)
        || CollectionUtils.isEmpty(productCodeSet)) {
      log.error("dto、spuCodeList、productCodeSet参数为空");
      return Lists.newLinkedList();
    }
    List<ProductSpu> spuList = this.productSpuRepository.findBySpuCodes(spuCodeList);
    if (CollectionUtils.isEmpty(spuList)) {
      log.error("根据[{}]获取到的spuList为空", spuList);
      return Lists.newLinkedList();
    }

    Map<String, ProductSpu> mapSpu = spuList.stream()
        .filter(a -> StringUtils.isNotBlank(a.getSpuCode()))
        .collect(Collectors.toMap(ProductSpu::getSpuCode, Function.identity(), (a, b) -> a));
    List<ProductSpuRelateSku> skuList1 =
        this.productSpuRelateSkuRepository.findBySpuCodes(spuCodeList);
    if (CollectionUtils.isEmpty(skuList1)) {
      log.error("根据[{}]获取spu关联上架的sku信息为空", spuCodeList);
      return Lists.newLinkedList();
    }
    List<ProductSpuRelateSku> skuList =
        skuList1.stream()
            .filter(a -> productCodeSet.contains(a.getProductCode()))
            .collect(Collectors.toList());
    if (CollectionUtils.isEmpty(skuList)) {
      log.error("根据[{}]获取spu关联上架的sku信息为空", spuCodeList);
      return Lists.newLinkedList();
    }
    List<ProductSpuMedia> mediaList =
        this.productSpuMediaRepository.findBySpuCodesAndType(
            spuCodeList, MediaTypeEnum.PICTURE.getCode());
    Map<String, List<ProductSpuMedia>> mapMedia =
        mediaList.stream()
            .filter(a -> StringUtils.isNotBlank(a.getSpuCode()))
            .collect(Collectors.groupingBy(ProductSpuMedia::getSpuCode));
    Set<String> skuCodeSet =
        skuList.stream()
            .filter(a -> StringUtils.isNotBlank(a.getProductCode()))
            .map(ProductSpuRelateSku::getProductCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(skuCodeSet)) {
      log.error("skuCodeSet为空");
      return Lists.newLinkedList();
    }
    // k-spuCode,v-商品编码集合
    Map<String, List<String>> mapSpuSku =
        skuList.stream()
            .filter(a -> StringUtils.isNoneBlank(a.getSpuCode(), a.getProductCode()))
            .collect(
                Collectors.groupingBy(
                    ProductSpuRelateSku::getSpuCode,
                    Collectors.mapping(ProductSpuRelateSku::getProductCode, Collectors.toList())));
    // 询价
    Map<String, PriceModelVo> priceModelVoMap = this.findPriceModel(skuCodeSet, dto.getBusinessCode());
    // 促销标记
    return this.initSpu(spuCodeList, mapSpu, mapMedia, mapSpuSku, priceModelVoMap);
  }

  /**
   * 设置spu扩展信息、spu信息、图片信息、商品价格
   *
   * @param spuCodeList
   * @param mapSpu
   * @param mapMedia
   * @param mapSpuSku
   * @param priceModelVoMap
   * @return
   */
  private List<ProductSpuVo> initSpu(
      List<String> spuCodeList,
      Map<String, ProductSpu> mapSpu,
      Map<String, List<ProductSpuMedia>> mapMedia,
      Map<String, List<String>> mapSpuSku,
      Map<String, PriceModelVo> priceModelVoMap) {
    if (CollectionUtils.isEmpty(spuCodeList) || mapSpu.isEmpty() || mapSpuSku.isEmpty()) {
      log.error("initSpu的spuCodeList、mapSpu、mapSpuSku为空");
      return Lists.newLinkedList();
    }
    List<ProductSpuVo> list = Lists.newLinkedList();
    for (String spuCode : spuCodeList) {
      ProductSpuVo spuVo = new ProductSpuVo();
      ProductSpu spu = mapSpu.getOrDefault(spuCode, new ProductSpu());
      spuVo.setSpuCode(spuCode);
      spuVo.setSpuName(spu.getSpuName());
      spuVo.setIsShelf(spu.getIsShelf());
      spuVo.setBeginDateTime(spu.getBeginDateTime());
      spuVo.setEndDateTime(spu.getEndDateTime());
      spuVo.setSort(spu.getSort());
      List<ProductSpuMedia> mediaList = mapMedia.get(spuCode);
      if (CollectionUtils.isNotEmpty(mediaList)) {
        spuVo.setPictureList(
            (List<ProductSpuMediaVo>)
                this.nebulaToolkitService.copyCollectionByBlankList(
                    mediaList,
                    ProductSpuMedia.class,
                    ProductSpuMediaVo.class,
                    HashSet.class,
                    ArrayList.class));
      }
      List<String> productCodeList = mapSpuSku.get(spuCode);
      if (CollectionUtils.isNotEmpty(productCodeList)) {
        List<ProductSpuRelateSkuVo> productList = Lists.newLinkedList();
        for (String productCode : productCodeList) {
          PriceModelVo priceModelVo = priceModelVoMap.getOrDefault(productCode, new PriceModelVo());
          ProductSpuRelateSkuVo skuVo = new ProductSpuRelateSkuVo();
          skuVo.setProductCode(productCode);
          skuVo.setPrice(priceModelVo.getPrice());
          productList.add(skuVo);
        }
        spuVo.setProductList(productList);
        spuVo.setPriceShowList(this.findPriceShow(productList));
      }
      list.add(spuVo);
    }
    return list;
  }

  /**
   * 价格展示
   *
   * @param productList
   * @return
   */
  private List<BigDecimal> findPriceShow(List<ProductSpuRelateSkuVo> productList) {
    if (CollectionUtils.isEmpty(productList)) {
      return Lists.newLinkedList();
    }
    return productList.stream()
        .filter(a -> Objects.nonNull(a.getPrice()))
        .map(ProductSpuRelateSkuVo::getPrice)
        .distinct()
        .sorted()
        .collect(Collectors.toList());
  }

  /**
   * 询价
   *
   * @param skuCodeSet
   * @param customerCode
   * @return
   */
  private Map<String, PriceModelVo> findPriceModel(Set<String> skuCodeSet, String customerCode) {
    if (CollectionUtils.isEmpty(skuCodeSet) || StringUtils.isBlank(customerCode)) {
      return Maps.newHashMap();
    }
    FindPriceDto dto = new FindPriceDto();
    dto.setUserCode(customerCode);
    dto.setUserType(FindPriceUserTypeEnum.CUSTOMER.getDictCode());
    dto.setDimensionCode(PriceDimensionEnum.PRODUCT.getDictCode());
    dto.setProductCodeSet(skuCodeSet);
    return this.priceModelVoService.findPrice(dto);
  }

  /**
   * 根据keyword+productLevelCode+productCodeList筛选符合的商品编码集合信息
   *
   * @param keyword
   * @param productLevelCode
   * @param productCodeList
   * @return
   */
  private Set<String> findProductCodeSet(String keyword, String likeKeyword, String productLevelCode, List<String> productCodeList) {
    // 有可能dto.getProductCodeList()是空的情况
    Set<String> productCodeSet = Sets.newHashSet();
    String ruleCode = StringUtils.EMPTY;
    if (StringUtils.isNotBlank(productLevelCode)) {
      ProductLevelVo productLevelVo = this.productLevelVoSdkService.findDetailsByCode(productLevelCode);
      if (Objects.nonNull(productCodeList)) {
        ruleCode = productLevelVo.getRuleCode();
      }
    }
    // 支持productCodeList为空的情况
    VisitorProductSpuDto productSpuDto = new VisitorProductSpuDto();
    productSpuDto.setKeyword(keyword);
    productSpuDto.setLikeKeyword(likeKeyword);
    productSpuDto.setRuleCode(ruleCode);
    if (CollectionUtils.isNotEmpty(productCodeList)) {
      for (List<String> item : Lists.partition(productCodeList, 500)) {
        productSpuDto.setProductCodeList(item);
        Set<String> curSet = this.visitorProductSpuVoRepository.findProductCodeSet(productSpuDto);
        if (CollectionUtils.isNotEmpty(curSet)) {
          productCodeSet.addAll(curSet);
        }
      }
    } else {
      Set<String> curSet = this.visitorProductSpuVoRepository.findProductCodeSet(productSpuDto);
      if (CollectionUtils.isNotEmpty(curSet)) {
        productCodeSet.addAll(curSet);
      }
    }
    return productCodeSet;
  }

}
