package com.biz.crm.mdm.admin.web.strategy.internal;

import com.biz.crm.mdm.admin.web.strategy.LoginLimitStrategy;

import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.security.sdk.config.SimpleSecurityProperties;
import com.bizunited.nebula.security.sdk.loginform.LoginDetails;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Component;

/**
 * 描述：</br>登录错误次数限制登录策略
 *
 * @author keller
 * @date 2022/10/19
 */
@Component
@Slf4j
public class LoginFailedTimesLimitStrategy implements LoginLimitStrategy {
  /**
   * 触发错误次数
   */
  @Value("${security.login.failedTimes:5}")
  private int failedTimes;
  @Autowired
  private StringRedisTemplate stringRedisTemplate;
  /**
   * 登录错误redis记录key
   */
  private static final String LOGIN_FAILED_KEY = "crm:login:failed:%s:%s";

  @Autowired
  private SimpleSecurityProperties simpleSecurityProperties;

  @Override
  public void handle(LoginDetails loginDetails) {
    if (StringUtils.isBlank(loginDetails.getAccount())) {
      log.warn("login limit found user account null");
      return;
    }
    Integer type = loginDetails.getType();
    if (type == null) {
      type = this.simpleSecurityProperties.getDefaultLoginType();
    }
    String account = StringUtils.join(type, "_", loginDetails.getAccount());
    String redisKey = String.format(LOGIN_FAILED_KEY, TenantUtils.getTenantCode(), account);
    String loginFailedTimeStr = this.stringRedisTemplate.opsForValue().get(redisKey);
    Integer loginFailedTime = 0;
    if (StringUtils.isNotBlank(loginFailedTimeStr)) {
      loginFailedTime = Integer.parseInt(loginFailedTimeStr);
      Validate.isTrue(loginFailedTime < failedTimes, "登录错误次数超过限制，账号已被锁定，请联系管理员");
    }
  }

  @Override
  public int getOrder() {
    return Integer.MIN_VALUE + 100;
  }
}
