package com.biz.crm.mdm.business.party.local.service.impl;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;

import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mdm.business.party.local.entity.DeliveryWay;
import com.biz.crm.mdm.business.party.local.repository.DeliveryWayRepository;
import com.biz.crm.mdm.business.party.sdk.dto.DeliveryWayDto;
import com.biz.crm.mdm.business.party.sdk.dto.QueryDeliveryWayDto;
import com.biz.crm.mdm.business.party.sdk.dto.SubdueDto;
import com.biz.crm.mdm.business.party.sdk.service.DeliveryWayService;
import com.biz.crm.mdm.business.party.sdk.vo.DeliveryWayVo;
import com.biz.crm.mdm.business.party.sdk.vo.QueryDeliveryWayVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;

import lombok.extern.slf4j.Slf4j;

/**
 * (DeliveryWayVo)表服务实现类11
 *
 * @author makejava
 * @since 2022-11-03 17:06:02
 */
@Slf4j
@Service("deliveryWayService")
public class DeliveryWayServiceImpl implements DeliveryWayService {

    @Autowired
    private DeliveryWayRepository deliveryWayRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Override
    public Page<DeliveryWayVo> findByConditions(Pageable pageable, DeliveryWayDto dto) {
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        dto = Optional.ofNullable(dto).orElse(new DeliveryWayDto());
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        Page<DeliveryWayVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return deliveryWayRepository.findByConditions(page, dto);
    }

    /**
     * 送达方搜索接口
     *
     * @param dto 对象
     */
    @Override
    public DeliveryWayVo searchInterface(SubdueDto dto) {
        DeliveryWayVo deliveryWayVo = deliveryWayRepository.searchInterface(dto);
        if (ObjectUtils.isEmpty(deliveryWayVo)) {
            throw new RuntimeException("查询数据不不存在！");
        }
        return deliveryWayVo;
    }

    /**
     * 根据送达方名称，通过送达方名称获取对应送达方编码
     *
     * @param deliveryWayName
     */
    @Override
    public String findByDeliveryWayCode(String deliveryWayName) {
        Validate.notNull(deliveryWayName, "送达方名称不能为空");
        DeliveryWay deliveryWay = deliveryWayRepository.findByDeliveryWayCode(deliveryWayName);
        Validate.notNull(deliveryWayName, "查询数据不存在");
        System.out.println("查询数据不存在" + deliveryWay.getDeliveryWayCode());
        return deliveryWay.getDeliveryWayCode();
    }

    /**
     * 根据送达方名称，通过送达方名称获取对应送达方编码
     *
     * @param deliveryWayCode
     */
    @Override
    public String findBySalesOrganization(String deliveryWayCode) {
        Validate.notNull(deliveryWayCode, "送达方名称不能为空");
        DeliveryWay deliveryWay = deliveryWayRepository.findBySalesOrganization(deliveryWayCode);
        Validate.notNull(deliveryWay, "查询数据不存在");
        return deliveryWay.getSalesInstitutionName();
    }

    /**
     * 通过售达方所属编码和名称查询售达方编码和名称
     *
     * @param queryDeliveryWayDto 查询实体
     * @return 所有数据
     */
    @Override
    public List<QueryDeliveryWayVo> findByDeliveryWayDto(QueryDeliveryWayDto queryDeliveryWayDto) {
        Validate.isTrue(StringUtils.isNotBlank(queryDeliveryWayDto.getSoldToPartyName()), "所属售达方名称不能为空");
        Validate.isTrue(StringUtils.isNotBlank(queryDeliveryWayDto.getSoldToPartyName()), "所属售达方编码不能为空");
        List<DeliveryWay> deliveryWay = deliveryWayRepository.queryByQueryDeliveryWayDto(queryDeliveryWayDto);
        List<QueryDeliveryWayVo> deliveryWayVoList = new ArrayList<>();
        if (deliveryWay == null) {
            return new ArrayList<>();
        }
        deliveryWay.forEach(aa -> {
            QueryDeliveryWayVo deliveryWayVo = new QueryDeliveryWayVo();
            deliveryWayVo.setDeliveryWayName(aa.getSoldToPartyName());
            deliveryWayVo.setDeliveryWayCode(aa.getSoldToPartyName());
            deliveryWayVoList.add(deliveryWayVo);
        });
        return deliveryWayVoList;
    }

    /**
     * 更新送达方
     *
     * @param deliveryWayDto 请求参数dto
     * @return 送达方
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public DeliveryWayVo update(DeliveryWayDto deliveryWayDto) {
        this.updateValidation(deliveryWayDto);
        String deliveryWayDtoId = deliveryWayDto.getId();
        // 根据送达方配置信息获取历史数据
        DeliveryWay deliveryWay = deliveryWayRepository.findByWayId(deliveryWayDtoId);
        Validate.notNull(deliveryWay, "修改信息不存在");
        deliveryWay.setTenantCode(TenantUtils.getTenantCode());
        Validate.isTrue(deliveryWayDto.getDeliveryWayCode().equals(deliveryWay.getDeliveryWayCode()), "送达方编码信息不能修改");
        DeliveryWay deliveryWay1 = this.nebulaToolkitService.copyObjectByWhiteList(deliveryWayDto, DeliveryWay.class,
            HashSet.class, ArrayList.class);
        this.deliveryWayRepository.saveOrUpdate(deliveryWay1);
        return this.nebulaToolkitService.copyObjectByWhiteList(deliveryWay1, DeliveryWayVo.class, HashSet.class,
            ArrayList.class);
    }

    /**
     * 通过ID获取送达方信息
     *
     * @param id 送达方ID
     * @return 操作结果
     */
    @Override
    public DeliveryWayVo findById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        DeliveryWayVo deliveryWayVo = this.deliveryWayRepository.findById(id, TenantUtils.getTenantCode());
        Validate.notNull(deliveryWayVo, "修改信息不存在");
        return deliveryWayVo;
    }

    private void updateValidation(DeliveryWayDto deliveryWayDto) {
        this.validation(deliveryWayDto);
    }

    private void validation(DeliveryWayDto deliveryWayDto) {
        Validate.notBlank(deliveryWayDto.getDeliveryWayCode(), "请选择送达方类型编码");
        Validate.notBlank(deliveryWayDto.getDeliveryWayName(), "请选择送达方名称");
        Validate.notNull(deliveryWayDto.getTypeName(), "送达方类型不能为空");
        Validate.notNull(deliveryWayDto.getTypeCode(), "送达方类型编码不能为空");
    }
}
