package com.biz.crm.mdm.business.party.local.service.impl;

import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.party.local.entity.DeliveryWay;
import com.biz.crm.mdm.business.party.local.repository.DeliveryWayRepository;
import com.biz.crm.mdm.business.party.local.service.DeliveryWayVoService;
import com.biz.crm.mdm.business.party.sdk.constant.DeliveryWayConstant;
import com.biz.crm.mn.common.base.constant.CommonConstant;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.common.base.util.UuidCrmUtil;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.constants.MasterDataMdgConstants;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.dto.MasterDataMdgBaseDto;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.service.MasterDataMdgService;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.vo.MasterDataMdgClientVo;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

/**
 * 1 拉去MDG主数据的http接口
 *
 * @author makejava
 * @since 2022-11-03 17:06:02
 */
@Slf4j
@Service("deliveryWayVoService")
public class DeliveryWayVoServiceImpl implements DeliveryWayVoService {

    @Autowired(required = false)
    private RedisMutexService redisMutexService;

    @Autowired(required = false)
    private MasterDataMdgService masterDataMdgService;

    @Autowired(required = false)
    private DeliveryWayRepository deliveryWayRepository;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    /**
     * 批量拉取送达方数据(ECC)
     *
     * @param dto
     * @author dutaotao
     * @date 2022/12/6 22:04
     **/
    @Override
    public void pullClientList(MasterDataMdgBaseDto dto) {
        if (ObjectUtils.isEmpty(dto)) {
            dto = new MasterDataMdgBaseDto();
        }
        if (StringUtils.isEmpty(dto.getPageNum())) {
            dto.setPageNum("1");
        }
        if (StringUtils.isEmpty(dto.getPageSize())) {
            dto.setPageSize(CommonConstant.MAX_PAGE_SIZE_STR);
        }

        String lockKey = DateUtil.format(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY_NO_CH);
        boolean lock = true;
        try {
            lock = this.lock(lockKey);
            if (!lock) {
                return;
            }
            AtomicLong failCount = new AtomicLong(0);
            List<MasterDataMdgClientVo> masterDataMdgClientVos = null;
            int maxPage = 0;
            if (BooleanEnum.TRUE.getCapital().equals(dto.getFullPullFlag())) {
                maxPage = this.countPageMax(dto);
            } else {
                maxPage = Integer.parseInt(dto.getPageNum());
            }
            for (int pageNum = Integer.parseInt(dto.getPageNum()); pageNum <= maxPage; pageNum++) {
                dto.setPageNum(Integer.toString(pageNum));
                masterDataMdgClientVos = masterDataMdgService.pullClientList(dto);
                if (CollectionUtils.isEmpty(masterDataMdgClientVos)) {
                    return;
                }
                // 数据校验
                List<DeliveryWay> pullList = this.supplierValidateClient(masterDataMdgClientVos, failCount);
                log.info("当前拉取进度：" + pageNum + "/" + maxPage + "，每页" + dto.getPageSize() + "条数据，拉取失败的数据共"
                        + failCount.get() + "条");
                failCount.set(0);
                List<String> deliveryWayCodeList =
                        pullList.stream().map(DeliveryWay::getDeliveryWayCode).collect(Collectors.toList());
                // 区分更新、新增操作
                List<DeliveryWay> deliveryWays = deliveryWayRepository.findByDeliveryWayCodingLists(deliveryWayCodeList);
                if (CollectionUtils.isEmpty(deliveryWays)) {
                    this.saveOrUpdateBatch(pullList, null);
                    continue;
                }
                // 区分更新、新增
                List<DeliveryWay> saveList = new ArrayList<>();
                List<DeliveryWay> updateList = new ArrayList<>();
                Map<String, DeliveryWay> map = deliveryWays.stream().collect(
                        Collectors.toMap(DeliveryWay::getDeliveryWayCode, v -> v,
                                (oldValue, newValue) -> newValue));
                pullList.forEach(pull -> {
                    DeliveryWay oldEntity = map.get(pull.getDeliveryWayCode());
                    if (Objects.nonNull(oldEntity)) {
                        pull.setId(oldEntity.getId());
                        pull.setCreateAccount(oldEntity.getCreateAccount());
                        pull.setCreateName(oldEntity.getCreateName());
                        pull.setCreateTime(oldEntity.getCreateTime());
                        updateList.add(pull);
                    } else {
                        saveList.add(pull);
                    }
                });
                this.saveOrUpdateBatch(saveList, updateList);
            }
        } finally {
            if (lock) {
                this.unlockClient(lockKey);
            }
        }
    }

    /**
     * 最大分页数
     *
     * @param dto
     * @return
     * @date 2023/1/8 1:33
     */
    private int countPageMax(MasterDataMdgBaseDto dto) {
        Integer totalNum = masterDataMdgService.countList(MasterDataMdgConstants.ECC_CLIENT_TPM, dto.getDs(), dto.getUdate());
        int maxPage = 1;
        int sizeInt = Integer.parseInt(dto.getPageSize());
        if (totalNum > sizeInt) {
            maxPage = totalNum % sizeInt > 0 ? (totalNum / sizeInt) + 1 : totalNum / sizeInt;
        }
        return maxPage;
    }

    /**
     * 批量新增数据（MD物料单位 G用）
     *
     * @param saveList
     * @param updateList
     * @author dutaotao
     * @date 2023/1/3 17:38
     **/
    @Transactional(rollbackFor = Exception.class)
    void saveOrUpdateBatch(List<DeliveryWay> saveList, List<DeliveryWay> updateList) {
        // 新增操作
        if (!CollectionUtils.isEmpty(saveList)) {
            AbstractCrmUserIdentity userIdentity = loginUserService.getAbstractLoginUser();
            String username = userIdentity.getUsername();
            String realName = userIdentity.getRealName();
            Date nowDate = new Date();
            String tenantCode = TenantUtils.getTenantCode();
            saveList.forEach(entity -> {
                if (StringUtils.isEmpty(entity.getId())) {
                    entity.setId(UuidCrmUtil.general());
                }
                entity.setCreateAccount(username);
                entity.setCreateName(realName);
                entity.setCreateTime(nowDate);
                entity.setTenantCode(tenantCode);
                if (StringUtils.isEmpty(entity.getDelFlag())) {
                    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                }
                if (StringUtils.isEmpty(entity.getEnableStatus())) {
                    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                }
            });
            this.deliveryWayRepository.saveBatchForXml(saveList);
        }
        //更新操作
        if (!CollectionUtils.isEmpty(updateList)) {
            this.deliveryWayRepository.updateBatchById(updateList);
        }
    }

    /**
     * 按年月日解锁ECC
     *
     * @param yearMonthDay
     * @author huojia
     * @date 2022/12/6 21:52
     **/
    private void unlockClient(String yearMonthDay) {

        if (StringUtils.isEmpty(yearMonthDay)) {
            throw new RuntimeException("拉取销售办公室关系主数据解锁失败，日期不能为空！");
        }
        redisMutexService.unlock(DeliveryWayConstant.CLIENT_LOCK + yearMonthDay);
    }

    /**
     * 按年月日加锁 ECC
     *
     * @param yearMonthDay
     * @return boolean
     * @author huojia
     * @date 2022/12/6 21:52
     **/
    private boolean lock(String yearMonthDay) {

        if (StringUtils.isEmpty(yearMonthDay)) {
            throw new RuntimeException("拉取销售办公室关系主数据解锁失败，日期不能为空！");
        }
        return this.redisMutexService.tryLock(DeliveryWayConstant.CLIENT_LOCK + yearMonthDay, TimeUnit.HOURS, 12);
    }

    /**
     * 数据校验 ECC客户主数据
     *
     * @param mdgVoList
     * @return java.util.List<com.biz.crm.mdm.business.supplier.local.entity.SupplierEntity>
     * @author huojia
     * @date 2022/12/6 21:37
     **/
    private List<DeliveryWay> supplierValidateClient(List<MasterDataMdgClientVo> mdgVoList,
                                                     AtomicLong failCount) {
        // 编码去重
        List<DeliveryWay> pullList = new ArrayList<>();

        String tenantCode = TenantUtils.getTenantCode();
        String enableStatus = DelFlagStatusEnum.NORMAL.getCode();
        String delFlag = DelFlagStatusEnum.NORMAL.getCode();
        mdgVoList.forEach(mdgVo -> {
            if (StringUtils.isEmpty(mdgVo.getKunn2())) {
                log.error("本次拉取数据送达方编码为空：" + mdgVo);
                failCount.getAndIncrement();
                return;
            }
            if (StringUtils.isEmpty(mdgVo.getVkorg())) {
                log.error("本次拉取数据销售机构编码为空：" + mdgVo);
                failCount.getAndIncrement();
                return;
            }
            if (StringUtils.isEmpty(mdgVo.getVtweg())) {
                log.error("本次拉取数据渠道编码为空：" + mdgVo);
                failCount.getAndIncrement();
                return;
            }
            if (StringUtils.isEmpty(mdgVo.getSpart())) {
                log.error("本次拉取数据业态编码为空：" + mdgVo);
                failCount.getAndIncrement();
                return;
            }
            //ERP编码+机构+渠道+业态
            mdgVo.setKunn2(StringUtils.trimToEmpty(mdgVo.getKunn2()));
            mdgVo.setKunnr(StringUtils.trimToEmpty(mdgVo.getKunnr()));
            mdgVo.setVkorg(StringUtils.trimToEmpty(mdgVo.getVkorg()));
            mdgVo.setVtweg(StringUtils.trimToEmpty(mdgVo.getVtweg()));
            mdgVo.setSpart(StringUtils.trimToEmpty(mdgVo.getSpart()));
            mdgVo.setVkbur(StringUtils.trimToEmpty(mdgVo.getVkbur()));
            mdgVo.setVkgrp(StringUtils.trimToEmpty(mdgVo.getVkgrp()));
            String deliveryWayCode = mdgVo.getKunn2() + mdgVo.getVkorg() + mdgVo.getVtweg() + mdgVo.getSpart();
            /**
             * 根据售达方编码获取售达方名字
             *
             * @param industryDeliveryWayCode
             * @return
             */
            DeliveryWay deliveryWay = new DeliveryWay();
            // 送达方编码
            deliveryWay.setDeliveryWayCode(deliveryWayCode);
            deliveryWay.setErpCode(mdgVo.getKunn2());
            // 送达方名称
            deliveryWay.setDeliveryWayName(mdgVo.getName1());
            // 售达方编码
            if (StringUtils.isNotBlank(mdgVo.getKunnr())) {
                String soldToPartyCode = mdgVo.getKunnr() + mdgVo.getVkorg() + mdgVo.getVtweg() + mdgVo.getSpart();
                deliveryWay.setSoldToPartyCode(soldToPartyCode);
            }
            deliveryWay.setSoldToPartyErpCode(mdgVo.getKunnr());
            //TODO 不能循环
//            List<String> customerName =
//                    customerVoService.findByIndustryDeliveryWayName(deliveryWay.getSoldToPartyCode());
//            if (!Objects.isNull(customerName)) {
//                deliveryWay.setSoldToPartyName(customerName.get(0));
//            }
            // 所属客户渠道编码
            deliveryWay.setCustomerChannelCode(mdgVo.getVtweg());
            // 根据所属客户渠道编码查询所属客户渠道区域

            deliveryWay.setBusinessFormatCode(mdgVo.getSpart());
            String salesOrgCode = deliveryWay.getCustomerChannelCode() + deliveryWay.getBusinessFormatCode() + mdgVo.getVkorg();
            // 所属销售机构编码
            deliveryWay.setSalesInstitutionErpCode(mdgVo.getVkorg());
            deliveryWay.setSalesInstitutionCode(salesOrgCode);
            //TODO 不能循环
            // 根据所属销售机构编码查询所属销售机构名称
//            List<String> salesInstitutionName =
//                    salesOrgVoService.findBySalesOrgName(deliveryWay.getSalesInstitutionCode());
//            if (!Objects.isNull(salesInstitutionName)) {
//                deliveryWay.setSalesInstitutionName(salesInstitutionName.get(0));
//            }

            // 所属销售部门编码
            salesOrgCode = salesOrgCode + mdgVo.getVkbur();
            deliveryWay.setSalesRegionErpCode(mdgVo.getVkbur());
            if (StringUtils.isNotBlank(mdgVo.getVkbur())) {
                deliveryWay.setSalesRegionCode(salesOrgCode);
            } else {
                deliveryWay.setSalesRegionCode("");
            }
            //TODO 不能循环
            // 所属销售机构部门编码查询所属销售机构部门名称
//            List<String> salesRegionName = salesOrgVoService.findBySalesOrgName(deliveryWay.getSalesRegionCode());
//            if (!Objects.isNull(salesRegionName)) {
//                deliveryWay.setSalesRegionName(salesRegionName.get(0));
//            }
            // 所属销售组编码
            deliveryWay.setSalesOrgErpCode(mdgVo.getVkgrp());
            if (StringUtils.isNotBlank(mdgVo.getVkgrp())
                    && StringUtils.isNotBlank(mdgVo.getVkbur())) {
                deliveryWay.setSalesOrgCode(salesOrgCode + mdgVo.getVkgrp());
            } else {
                deliveryWay.setSalesRegionCode("");
            }
            //TODO 不能循环
            // 所属销售组编码查询销售组名称
//            List<String> salesOrgName = salesOrgVoService.findBySalesOrgName(deliveryWay.getSalesOrgCode());
//            if (!Objects.isNull(salesOrgName)) {
//                deliveryWay.setSalesOrgName(salesOrgName.get(0));
//            }
            // 收获地址
            deliveryWay.setHarvestAddress(mdgVo.getStras());
            // 收获联系人
            deliveryWay.setHarvestContactPersons(mdgVo.getName());
            // 收获联系方式
            deliveryWay.setHarvestContactPhone(mdgVo.getTelf1());
            deliveryWay.setTenantCode(tenantCode);
            deliveryWay.setEnableStatus(enableStatus);
            deliveryWay.setDelFlag(delFlag);
            pullList.add(deliveryWay);
        });
        return pullList.stream()
                .filter(k -> StringUtils.isNotBlank(k.getDeliveryWayCode()))
                .collect(Collectors.collectingAndThen(Collectors.toCollection(()
                        -> new TreeSet<>(Comparator.comparing(DeliveryWay::getDeliveryWayCode))), ArrayList::new));

    }
}
