package com.biz.crm.mdm.business.price.form.local.service.impl;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.price.form.local.constant.PriceFormConstant;
import com.biz.crm.mdm.business.price.form.local.entity.PriceForm;
import com.biz.crm.mdm.business.price.form.local.repository.PriceFormRepository;
import com.biz.crm.mdm.business.price.form.sdk.dto.PriceFormDto;
import com.biz.crm.mdm.business.price.form.sdk.dto.PriceFormLogEventDto;
import com.biz.crm.mdm.business.price.form.sdk.dto.PricingDmensionDto;
import com.biz.crm.mdm.business.price.form.sdk.dto.TypePriceDataDto;
import com.biz.crm.mdm.business.price.form.sdk.enmus.PricingDimensionEnum;
import com.biz.crm.mdm.business.price.form.sdk.event.log.PriceFormLogEventListener;
import com.biz.crm.mdm.business.price.form.sdk.service.PriceFormService;
import com.biz.crm.mdm.business.price.form.sdk.service.PricingDmensionService;
import com.biz.crm.mdm.business.price.form.sdk.vo.*;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

/**
 * (PriceFormServiceImpl)表服务实现类
 *
 * @author makejava
 * @since 2022-11-03 17:06:02
 */
@Slf4j
@Service("priceFormService")
public class PriceFormServiceImpl implements PriceFormService {

    @Autowired
    private PriceFormRepository priceFormRepository;

    @Autowired
    private GenerateCodeService generateCodeService;

    @Autowired
    private NebulaToolkitService nebulaToolkitService;

    @Autowired
    private PricingDmensionService pricingDmensionService;


    @Autowired
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    /**
     * 分页查询所有数据
     *
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据
     */
    @Override
    public Page<PriceFormVo> findByConditions(Pageable pageable, PriceFormDto dto) {
        if (pageable == null) {
            pageable = PageRequest.of(0, 50);
        }
        if (dto == null) {
            dto = new PriceFormDto();
        }

        /**获取租户编号*/
        if (StringUtils.isBlank(dto.getTenantCode())) {
            dto.setTenantCode(TenantUtils.getTenantCode());
        }

        /**获取删除标志*/
        if (StringUtils.isBlank(dto.getDelFlag())) {
            dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        }
        return this.priceFormRepository.findByConditions(pageable, dto);
    }


    /**
     * 新增价格类型
     *
     * @param dto 查询实体
     * @return 所有数据
     */
    @Override
    public void create(TypePriceDataDto dto) {
        Date nowDate = new Date();
        //价格类型置表
        PriceFormDto priceFormDto = dto.getPriceFormDto();
        //定价维度配置表
        List<PricingDmensionDto> pricingDmensionDto = dto.getPricingDmensionDto();
        Validate.isTrue(StringUtils.isNotBlank(priceFormDto.getPriceFormName()), "价格类型名称不能为空");
        Validate.isTrue(priceFormDto.getPriceFormName().length() < 128, "价格类型名称，在进行输入时填入值超过了限定长度(128)，请检查!");
        //通过价格类型编码查询用户数据
        PriceForm priceForm = this.priceFormRepository.findByPriceFormCode(priceFormDto.getPriceFormCode());
        Validate.isTrue(Objects.isNull(priceForm), "已存在当前价格类型编码的数据");
        //设置公共字段
        priceFormDto.setTenantCode(TenantUtils.getTenantCode());
        priceFormDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        priceFormDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        priceFormDto.setCreateName(loginUserService.getLoginAccountName());
        priceFormDto.setCreateTime(nowDate);
        //设置价格类型编码
        String ruleCode = PriceFormConstant.PRICE_TYPE_ENCODING_PREFIX;
        String promotionCode = generateCodeService.generateCode(ruleCode, 1).get(0);
        priceFormDto.setPriceFormCode(promotionCode);
        //保存主表价格类型字段
        PriceForm priceForm1 = this.nebulaToolkitService.copyObjectByWhiteList(priceFormDto, PriceForm.class, HashSet.class, ArrayList.class);
        priceFormRepository.save(priceForm1);
        //保存副表定价维度配置表数据
        for (PricingDmensionDto dmensionDto : pricingDmensionDto) {
            Validate.isTrue(StringUtils.isNotBlank(dmensionDto.getPricingDimension()), "定价维度不能为空");
            pricingDmensionService.sava(dmensionDto, promotionCode);
        }
        //新增业务日志
        PriceFormLogEventDto logEventDto = new PriceFormLogEventDto();
        //设置原始的值
        logEventDto.setOriginal(null);
        logEventDto.setNewest(priceFormDto);
        //设置新的值
        //创建可序列化的消费者在创建对象
        SerializableBiConsumer<PriceFormLogEventListener, PriceFormLogEventDto> onCreate = PriceFormLogEventListener::onCreate;
        this.nebulaNetEventClient.publish(logEventDto, PriceFormLogEventListener.class, onCreate);
    }

    /**
     * 更新价格类型
     *
     * @param dto 请求参数dto
     * @return 职位实体
     */
    @Override
    public void update(TypePriceDataDto dto) {
        Date date = new Date();
        this.updateValidation(dto);
        PriceFormDto priceFormDto = dto.getPriceFormDto();

        PriceForm byPriceFormCode = this.priceFormRepository.findByPriceFormCode(priceFormDto.getPriceFormCode());
        Validate.notNull(byPriceFormCode, "更新数据不存在");
        priceFormDto.setCurrentUpdatePeople(loginUserService.getLoginAccountName());
        priceFormDto.setCurrentUpdateDate(date);
        //保存主表价格类型字段
        PriceForm priceForm = this.nebulaToolkitService.copyObjectByWhiteList(priceFormDto, PriceForm.class, HashSet.class, ArrayList.class);
        this.priceFormRepository.saveOrUpdate(priceForm);
        //保存
        //根据编码类型删除选择预算项目
        this.pricingDmensionService.deleteByControlsConfigCode(priceFormDto.getPriceFormCode());
        //批量新定价维度配置表
        this.pricingDmensionService.savaBatch(dto.getPricingDmensionDto());
    }


    /**
     * 在修改positionLevel模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
     *
     * @param dto 检查对象
     */
    private void updateValidation(TypePriceDataDto dto) {
        Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
        PriceFormDto priceFormDto = dto.getPriceFormDto();
        priceFormDto.setTenantCode(TenantUtils.getTenantCode());
        Validate.notBlank(priceFormDto.getPriceFormName(), "价格类型名称不能为空");
        Validate.isTrue(StringUtils.isBlank(priceFormDto.getPriceFormName()) || priceFormDto.getPriceFormName().length() < 128, "价格类型名称，在进行编辑时填入值超过了限定长度(128)，请检查!");
        List<PricingDmensionDto> listDto = dto.getPricingDmensionDto();
        listDto.forEach(l -> {
            Validate.notBlank(l.getPricingDimension(), "定价维度不能为空");
        });
    }


    /**
     * 通过价格类型名查询价格详情
     *
     * @param priceFormName 对象
     */
    @Override
    public PriceManagementVo findByPriceFormName(String priceFormName) {
        List<PricingDimensionVo> arrayList = new ArrayList<>();
        Validate.notBlank(priceFormName, "价格类型名称不能为空");
        Validate.isTrue(priceFormName.length() < 255, "价格类型名称，在进行编辑时填入值超过了限定长度(255)，请检查!");
        //通过价格类型名查询价格详情
        PriceForm priceForm = this.priceFormRepository.findByPriceDetails(priceFormName);
        PriceFormVo priceFormVo = this.nebulaToolkitService.copyObjectByWhiteList(priceForm, PriceFormVo.class, HashSet.class, ArrayList.class);
        Validate.notNull(priceForm, "当前价格类型的数据不存在");
        //通过价格类型编码定价维度
        List<PricingDmensionVo> pricingDmensionVo = this.pricingDmensionService.findByPricingDimension(priceForm.getPriceFormCode());
        Validate.notNull(priceForm, "查询出来的定价维度配置体表数据不存在");
        for (PricingDmensionVo pricingDmensionVos : pricingDmensionVo) {
            String pricingDimension = pricingDmensionVos.getPricingDimension();
            String[] split = pricingDimension.split(",");
            String result = "";
            for (String s : split) {
                if (s.equals(PricingDimensionEnum.CHANNEL.getPriceFormCode())) {
                    result = result + "/" + PricingDimensionEnum.CHANNEL.getPriceFormName();
                } else if (s.equals(PricingDimensionEnum.CUSTOMER.getPriceFormCode())) {
                    result = result + "/" + PricingDimensionEnum.CUSTOMER.getPriceFormName();
                } else if (s.equals(PricingDimensionEnum.PRODUCT.getPriceFormCode())) {
                    result = result + "/" + PricingDimensionEnum.PRODUCT.getPriceFormName();
                } else if (s.equals(PricingDimensionEnum.MARKETING_ORGANIZING.getPriceFormCode())) {
                    result = result + "/" + PricingDimensionEnum.MARKETING_ORGANIZING.getPriceFormName();
                } else if (s.equals(PricingDimensionEnum.ORG.getPriceFormCode())) {
                    result = result + "/" + PricingDimensionEnum.ORG.getPriceFormName();
                } else if (s.equals(PricingDimensionEnum.PRICE_GROUP.getPriceFormCode())) {
                    result = result + "/" + PricingDimensionEnum.PRICE_GROUP.getPriceFormName();
                } else if (s.equals(PricingDimensionEnum.TERMINAL.getPriceFormCode())) {
                    result = result + "/" + PricingDimensionEnum.TERMINAL.getPriceFormName();
                }
            }
            PricingDimensionVo dimensionVo = new PricingDimensionVo(pricingDimension, result);
            arrayList.add(dimensionVo);
        }
        //获取价格 维度
        return new PriceManagementVo(priceFormVo, arrayList);
    }

    /**
     * 批量删除
     */
    @Override
    @Transactional
    public void deleteBatch(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            throw new RuntimeException("请选择要删除的数据！");
        }
        List<PriceForm> dimensionControls = this.priceFormRepository.listByIds(ids);
        if (!CollectionUtils.isEmpty(dimensionControls)) {
            dimensionControls.forEach(priceForm -> {
                priceForm.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
            });
            this.priceFormRepository.updateBatchById(dimensionControls);
            ids.forEach(controlsById -> {
                //执行删除核销采集（主表逻辑删除，明细项信息不做变更）
                this.priceFormRepository.deleteByControlsById(controlsById);
            });
        }
        List<PriceFormVo> priceFormVos = this.findByIds(ids);
        //删除业务日志
        Collection<PriceFormDto> dimensionBudgetDtos = this.nebulaToolkitService.copyCollectionByWhiteList(priceFormVos, PriceFormVo.class, PriceFormDto.class, HashSet.class, ArrayList.class);
        SerializableBiConsumer<PriceFormLogEventListener, PriceFormLogEventDto> onDelete = PriceFormLogEventListener::onDelete;
        for (PriceFormDto dimensionControlsDto : dimensionBudgetDtos) {
            PriceFormLogEventDto logEventDto = new PriceFormLogEventDto();
            logEventDto.setOriginal(dimensionControlsDto);
            this.nebulaNetEventClient.publish(logEventDto, PriceFormLogEventListener.class, onDelete);
        }
    }

    /**
     * 查详情
     */
    @Override
    public TypePriceDataVo findById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        //重构查询方法
        PriceForm result = priceFormRepository.findByIdAndTenantCode(id, TenantUtils.getTenantCode());
        //保存主表价格类型字段
        PriceFormVo priceFormVo = this.nebulaToolkitService.copyObjectByWhiteList(result, PriceFormVo.class, HashSet.class, ArrayList.class);
        List<PricingDmensionVo> vo = this.pricingDmensionService.findByIdAnddelFlagde(result.getPriceFormCode(), result.getDelFlag());
        return new TypePriceDataVo(priceFormVo, vo);
    }


    /**
     * 通过价格类型名称查询价格详情
     *
     * @param priceFormCode 请求参数
     * @return 价格类型详情
     */
    @Override
    public PriceManagementDataVo findByPriceManagementData(String priceFormCode) {
        List<String> list = new ArrayList<>();
        //查询表价格数据
        Validate.notBlank(priceFormCode, "价格类型名称不能为空");
        //通过价格类型名查询价格详情
        PriceForm priceForm = this.priceFormRepository.findByCodeData(priceFormCode);
        Validate.notNull(priceForm, "当前价格价格类型编码数据不存在");
        //通过价格类型编码定价维度
        List<PricingDmensionVo> pricingDmensionVo = this.pricingDmensionService.findByPricingDimension(priceForm.getPriceFormCode());
        Validate.notNull(pricingDmensionVo, "查询出来的定价维度配置表数据不存在");

        for (PricingDmensionVo dmensionVo : pricingDmensionVo) {
            list.add(dmensionVo.getPricingDimension());
        }
        return new PriceManagementDataVo(priceForm.getPriceFormCode(), list, priceForm.getPriceFormName());
    }


    /**
     * 批量查询
     */
    private List<PriceFormVo> findByIds(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return Lists.newArrayList();
        }
        List<PriceForm> priceForms = priceFormRepository.listByIds(ids);
        if (CollectionUtils.isEmpty(priceForms)) {
            return Lists.newArrayList();
        }
        return Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(priceForms, PriceForm.class, PriceFormVo.class, HashSet.class, ArrayList.class));
    }

}
