package com.biz.crm.mdm.business.price.management.local.repository;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.price.management.local.entity.PriceDirectData;
import com.biz.crm.mdm.business.price.management.local.mapper.PriceManagementMapper;
import com.biz.crm.mdm.business.price.management.sdk.dto.PriceDirectDataDto;
import com.biz.crm.mdm.business.price.management.sdk.vo.PriceDirectDataVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.List;

/**
 * 价格管理的数据库访问类 {@link PriceDirectData}
 *
 * @author ning.zhang
 * @date 2021/9/28
 */
@Component
public class PriceManagementRepository extends ServiceImpl<PriceManagementMapper, PriceDirectData> {

    @Autowired
    private PriceManagementMapper priceManagementMapper;

    /**
     * 分页查询数据
     *
     * @param pageable 分页对象
     * @param dto      实体对象
     * @return
     */
    public Page<PriceDirectDataVo> findByConditions(Pageable pageable, PriceDirectDataDto dto) {
        if (pageable == null) {
            pageable = PageRequest.of(0, 50);
        }
        dto.setTenantCode(TenantUtils.getTenantCode());
        Page<PriceDirectDataVo> page = new Page<PriceDirectDataVo>(pageable.getPageNumber(), pageable.getPageSize());
        return priceManagementMapper.findByConditions(page, dto);
    }

    /**
     * 通过定价记录号查询价格类型数据
     *
     * @param pricingRecordNumber 实体对象
     * @return
     */
    public PriceDirectData findByPricingRecordNumber(String pricingRecordNumber) {
        return this.lambdaQuery().eq(PriceDirectData::getPricingRecordNumber, pricingRecordNumber).one();
    }

    /**
     * 查找上一次更新价格
     */
    public PriceDirectDataVo getByPriceUpdatedLastTime(String pricingDimension) {
        return priceManagementMapper.getByPriceUpdatedLastTime(pricingDimension);
    }

    /**
     * 批量更新启用禁用状态
     *
     * @param ids    ID集合
     * @param enable 启用禁用状态
     */
    public void updateEnableStatusByIds(List<String> ids, EnableStatusEnum enable) {
        this.lambdaUpdate()
                .in(PriceDirectData::getId, ids)
                .eq(PriceDirectData::getTenantCode, TenantUtils.getTenantCode())
                .set(PriceDirectData::getEnableStatus, enable.getCode())
                .update();
    }

    /**
     * 根据定价记录号，通过产品编码获取对价格管理基本信息
     *
     * @param pricingRecordNumber
     */
    public PriceDirectData listByPricingRecordNumber(String pricingRecordNumber) {
        return this.lambdaQuery().eq(PriceDirectData::getPricingRecordNumber, pricingRecordNumber).one();
    }

    /**
     * 重构查询方法
     *
     * @param id
     * @param tenantCode
     * @return
     */
    public PriceDirectData findByIdAndTenantCode(String id, String tenantCode) {
        return this.lambdaQuery()
                .eq(PriceDirectData::getTenantCode, tenantCode)
                .in(PriceDirectData::getId, id)
                .one();
    }

    /**
     * 获取删除标记
     *
     * @param delFlag
     * @return
     */
    public PriceDirectData deleteById(String delFlag,String id) {
        return this.lambdaQuery().eq(PriceDirectData::getDelFlag, delFlag).eq(PriceDirectData::getId, id).one();
    }


    /**
     * 根据主键集合，修改 enable_status
     *
     * @param enable
     * @param ids
     */
    public void updateEnableStatusByIdIn(EnableStatusEnum enable, List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.lambdaUpdate().in(PriceDirectData::getId, ids)
                .set(PriceDirectData::getEnableStatus, enable.getCode())
                .eq(PriceDirectData::getTenantCode, TenantUtils.getTenantCode()) // 过滤多租户
                .update();
    }

}