package com.biz.crm.mdm.business.price.management.local.service.impl;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.price.management.local.constant.PriceDirectDataConstant;
import com.biz.crm.mdm.business.price.management.local.entity.PriceDirectData;
import com.biz.crm.mdm.business.price.management.local.repository.PriceManagementRepository;
import com.biz.crm.mdm.business.price.management.sdk.dto.PriceDirectDataDto;
import com.biz.crm.mdm.business.price.management.sdk.dto.PriceDirectDataLogEventDto;
import com.biz.crm.mdm.business.price.management.sdk.enmus.ValidStateEnum;
import com.biz.crm.mdm.business.price.management.sdk.event.log.PriceDirectLogEventListener;
import com.biz.crm.mdm.business.price.management.sdk.service.PriceManagementService;
import com.biz.crm.mdm.business.price.management.sdk.vo.PriceDirectDataVo;
import com.biz.crm.mn.common.base.eunm.DataFromEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;

/**
 * (PriceManagementServiceImpl)表服务实现类
 *
 * @author Administrator
 * @since 2022-11-19 11:06:02
 */
@Slf4j
@Service("priceManagementService")
public class PriceManagementServiceImpl implements PriceManagementService {

    @Autowired
    private PriceManagementRepository priceManagementRepository;

    @Autowired
    private GenerateCodeService generateCodeService;

    @Autowired
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired
    private NebulaToolkitService nebulaToolkitService;


    /**
     * 分页查询所有数据
     *
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据
     */
    @Override
    public Page<PriceDirectDataVo> findByConditions(Pageable pageable, PriceDirectDataDto dto) {
        if (pageable == null) {
            pageable = PageRequest.of(0, 50);
        }
        if (dto == null) {
            dto = new PriceDirectDataDto();
        }

        //获取租户编号
        if (StringUtils.isBlank(dto.getTenantCode())) {
            dto.setTenantCode(TenantUtils.getTenantCode());
        }

        //获取删除标志
        if (StringUtils.isBlank(dto.getDelFlag())) {
            dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        }
        return this.priceManagementRepository.findByConditions(pageable, dto);
    }

    /**
     * 新增价格管理
     *
     * @param dto 查询实体
     * @return 所有数据
     */
    @Override
    public PriceDirectDataVo create(PriceDirectDataDto dto) {
        //当前时间
        Date now = new Date();
        PriceDirectDataVo PriceUpdatedLastTime = null;
        //参数效验
        extracted(dto);
        //通过定价记录号查询价格类型数据
        PriceDirectData priceDirectData = this.priceManagementRepository.findByPricingRecordNumber(dto.getPricingRecordNumber());
        Validate.isTrue(Objects.isNull(priceDirectData), "已存在当前价格管理编码的数据");
        //设置公共字段
        dto.setTenantCode(TenantUtils.getTenantCode());
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());

        dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        //设置定价记录号
        String ruleCode = PriceDirectDataConstant.PRICEFORMCODE;
        String promotionCode = generateCodeService.generateCode(ruleCode, 1).get(0);
        dto.setPricingRecordNumber(promotionCode);
        dto.setCreateTime(new Date());

        //设置有效状态
        if (dto.getStartTime().getTime() > now.getTime()) {
            dto.setValidState(ValidStateEnum.NOT_IN_EFFECT.getCode());
        } else if (now.getTime() > dto.getStartTime().getTime() || now.getTime() < dto.getEndTime().getTime()) {
            dto.setValidState(ValidStateEnum.IN_EFFECT.getCode());
        } else if (now.getTime() > dto.getEndTime().getTime()) {
            dto.setValidState(ValidStateEnum.OUT_OF_DATE.getCode());
        } else {
            dto.setValidState(ValidStateEnum.DISABLED.getCode());
        }

        //保存主表价格类型字段
        PriceDirectData result = this.nebulaToolkitService.copyObjectByWhiteList(dto,
                PriceDirectData.class,
                HashSet.class,
                ArrayList.class);

        priceManagementRepository.save(result);

        //新增业务日志
        PriceDirectDataLogEventDto logEventDto = new PriceDirectDataLogEventDto();
        //设置原始的值
        logEventDto.setOriginal(null);
        logEventDto.setNewest(dto);
        //设置新的值
        //创建可序列化的消费者在创建对象
        SerializableBiConsumer<PriceDirectLogEventListener, PriceDirectDataLogEventDto> onCreate =
                PriceDirectLogEventListener::onCreate;
        this.nebulaNetEventClient.publish(logEventDto, PriceDirectLogEventListener.class, onCreate);
        return PriceUpdatedLastTime;
    }

    /**
     * 参数效验
     */
    private void extracted(PriceDirectDataDto dto) {
        Validate.isTrue(StringUtils.isNotBlank(dto.getPricingDimension()), "定价维度不能为空");
        Validate.isTrue(StringUtils.isNotBlank(dto.getPriceFormCode()), "价格类型编码不能为空");
        Validate.isTrue(StringUtils.isNotBlank(dto.getClient()), "客户不能为空");
        Validate.isTrue(StringUtils.isNotBlank(dto.getGoodsName()), "商品名称不能为空");
        Validate.isTrue(StringUtils.isNotBlank(dto.getGoodsCode()), "商品编码不能为空");
        Validate.notNull(dto.getPrice(), "价格不能为空");
        Validate.notNull(dto.getStartTime(), "有效开始时间不能为空");
        Validate.notNull(dto.getEndTime(), "有效结束时间不能为空");
        Validate.isTrue(StringUtils.isNotBlank(dto.getPriceFormName()), "价格类型类型不能为空");
        Validate.isTrue(dto.getPriceFormCode().length() < 64, "价格类型编码，在进行输入时填入值超过了限定长度(64)，请检查!");
        Validate.isTrue(dto.getPriceFormName().length() < 128, "价格类型名称，在进行输入时填入值超过了限定长度(128)，请检查!");

    }

    /**
     * 更新价格管理
     *
     * @param dto 请求参数dto
     * @return 职位实体
     */
    @Override
    public String update(PriceDirectDataDto dto) {
        this.updateValidation(dto);
        PriceDirectData priceDirectData = this.priceManagementRepository.findByPricingRecordNumber(dto.getPricingRecordNumber());
        Validate.notNull(priceDirectData, "更新数据不存在");
        Validate.isTrue(priceDirectData.getPricingRecordNumber().equals(dto.getPricingRecordNumber()), "定价记录号不能修改");
        //保存主表价格类型字段
        PriceDirectData priceDirectData1 = this.nebulaToolkitService.copyObjectByWhiteList(dto,
                PriceDirectData.class,
                HashSet.class,
                ArrayList.class);
        //维度预算
        this.priceManagementRepository.saveOrUpdate(priceDirectData1);

        //编写业务日志
        PriceDirectDataDto priceDirectDataDto = this.nebulaToolkitService.copyObjectByWhiteList(priceDirectData, PriceDirectDataDto.class, LinkedHashSet.class, ArrayList.class);
        PriceDirectDataLogEventDto logEventDto = new PriceDirectDataLogEventDto();
        //设置原始的值
        logEventDto.setOriginal(priceDirectDataDto);
        //设置新的值
        logEventDto.setNewest(dto);
        //创建可序列化的消费者在创建对象
        SerializableBiConsumer<PriceDirectLogEventListener, PriceDirectDataLogEventDto> onUpdate =
                PriceDirectLogEventListener::onUpdate;
        this.nebulaNetEventClient.publish(logEventDto, PriceDirectLogEventListener.class, onUpdate);
        return "更新成功";
    }


    /**
     * 根据定价记录号，通过产品编码获取对价格管理基本信息
     *
     * @param pricingRecordNumber
     */
    @Override
    public PriceDirectDataVo listByPricingRecordNumber(String pricingRecordNumber) {
        PriceDirectData priceDirectData = this.priceManagementRepository.listByPricingRecordNumber(pricingRecordNumber);
        PriceDirectDataVo result = this.nebulaToolkitService.copyObjectByWhiteList(priceDirectData,
                PriceDirectDataVo.class,
                HashSet.class,
                ArrayList.class);
        return result;
    }


    /**
     * 在修改positionLevel模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
     *
     * @param dto 检查对象
     */
    private void updateValidation(PriceDirectDataDto dto) {
        Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
        dto.setTenantCode(TenantUtils.getTenantCode());
        Validate.notBlank(dto.getId(), "修改信息时，id不能为空！");
        Validate.notBlank(dto.getPriceFormCode(), "价格类型编码");
        Validate.notBlank(dto.getPricingRecordNumber(), "定价记录号编码");
        Validate.isTrue(StringUtils.isBlank(dto.getPriceFormName()) || dto.getGoodsName().length() < 128
                , "商品和价格名称，在进行编辑时填入值超过了限定长度(128)，请检查!");
    }


    /**
     * 批量导入
     *
     * @param importList
     */
    @Override
    public void importSave(List<PriceDirectDataDto> importList) {
        if (CollectionUtils.isEmpty(importList)) {
            return;
        }
        importList.forEach(dto -> {
            dto.setTenantCode(TenantUtils.getTenantCode());
            dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            dto.setPriceFormCode(DataFromEnum.IMPORT.getCode());
        });
        Collection<PriceDirectData> salesGoalEntities = this.nebulaToolkitService.copyCollectionByWhiteList(importList, PriceDirectDataDto.class, PriceDirectData.class, LinkedHashSet.class, ArrayList.class);
        this.priceManagementRepository.saveBatch(salesGoalEntities);
        // 业务日志创建
        importList.forEach(dto -> {
            PriceDirectDataLogEventDto logEventDto = new PriceDirectDataLogEventDto();
            logEventDto.setOriginal(null);
            logEventDto.setNewest(dto);
            SerializableBiConsumer<PriceDirectLogEventListener, PriceDirectDataLogEventDto> onCreate =
                    PriceDirectLogEventListener::onCreate;
            this.nebulaNetEventClient.publish(logEventDto, PriceDirectLogEventListener.class, onCreate);
        });
    }

    /**
     * 查详情
     */
    @Override
    public PriceDirectDataVo findById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        //重构查询方法
        PriceDirectData result = priceManagementRepository.findByIdAndTenantCode(id, TenantUtils.getTenantCode());
        //获取删除标记
        PriceDirectData priceDirectData = this.priceManagementRepository.deleteById(result.getDelFlag(),id);
        //保存主表价格类型字段
        PriceDirectDataVo priceFormVo = this.nebulaToolkitService.copyObjectByWhiteList(priceDirectData,
                PriceDirectDataVo.class,
                HashSet.class,
                ArrayList.class);

        return priceFormVo;
    }


    /**
     * 批量启用职位
     *
     * @param ids 职位id集合
     */
    @Override
    public void enableBatch(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids), "待修正的数据主键不能为空");
        priceManagementRepository.updateEnableStatusByIdIn(EnableStatusEnum.ENABLE, ids);
    }

    @Override
    public void disableBatch(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids), "待修正的数据主键不能为空");
        priceManagementRepository.updateEnableStatusByIdIn(EnableStatusEnum.DISABLE, ids);
    }

}
