package com.biz.crm.mdm.business.format.local.service.internal;


import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.format.local.entity.BusinessFormat;
import com.biz.crm.mdm.business.format.local.repository.BusinessFormatRepository;
import com.biz.crm.mdm.business.format.local.service.BusinessFormatService;
import com.biz.crm.mdm.business.format.sdk.dto.BusinessFormatQueryPaginationDto;
import com.biz.crm.mdm.business.format.sdk.enums.DataSourceEnum;
import com.biz.crm.mn.common.base.dto.CommonSelectDto;
import com.biz.crm.mn.common.base.vo.CommonSelectVo;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.stream.Collectors;

/**
 * 业态主表实体(BusinessFormat)表服务实现类
 *
 * @author makejava
 * @since 2023-02-24 17:33:20
 */
@Service("businessFormatService")
public class BusinessFormatServiceImpl implements BusinessFormatService {

  @Autowired
  private BusinessFormatRepository businessFormatRepository;

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public BusinessFormat findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.businessFormatRepository.getById(id);
  }

  /**
   * 新增数据
   *
   * @param businessFormat 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public BusinessFormat create(BusinessFormat businessFormat) {
    this.createValidate(businessFormat);
    this.businessFormatRepository.saveOrUpdate(businessFormat);
    return businessFormat;
  }

  /**
   * 修改新据
   *
   * @param businessFormat 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public BusinessFormat update(BusinessFormat businessFormat) {
    this.updateValidate(businessFormat);
    this.businessFormatRepository.saveOrUpdate(businessFormat);
    return businessFormat;
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    List<BusinessFormat> businessFormats = this.businessFormatRepository.findByIds(idList);
    Validate.isTrue(!CollectionUtils.isEmpty(businessFormats), "删除数据时，未查询到需要删除的数据！");
    businessFormats.stream().forEach(businessFormat -> businessFormat.setDelFlag(DelFlagStatusEnum.DELETE.getCode()));
    this.businessFormatRepository.saveOrUpdateBatch(businessFormats);
  }

  /**
   * 启用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 启用结果
   */
  @Transactional
  @Override
  public void enable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "启用数据时，主键集合不能为空！");
    List<BusinessFormat> businessFormats = this.businessFormatRepository.findEnableStatusByIds(EnableStatusEnum.DISABLE, idList);
    if (CollectionUtils.isEmpty(businessFormats)) {
      return;
    }
    businessFormats.stream().forEach(businessFormat -> businessFormat.setEnableStatus(EnableStatusEnum.ENABLE.getCode()));
    this.businessFormatRepository.saveOrUpdateBatch(businessFormats);
  }

  /**
   * 禁用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 禁用结果
   */
  @Transactional
  @Override
  public void disable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "禁用数据时，主键集合不能为空！");
    List<BusinessFormat> businessFormats = this.businessFormatRepository.findEnableStatusByIds(EnableStatusEnum.ENABLE, idList);
    if (CollectionUtils.isEmpty(businessFormats)) {
      return;
    }
    businessFormats.stream().forEach(businessFormat -> businessFormat.setEnableStatus(EnableStatusEnum.DISABLE.getCode()));
    this.businessFormatRepository.saveOrUpdateBatch(businessFormats);
  }

  @Override
  public Page<BusinessFormat> findByConditions(Pageable pageable, BusinessFormatQueryPaginationDto paginationDto) {
    Page<BusinessFormat> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());

    return businessFormatRepository.lambdaQuery()
            .eq(BusinessFormat::getDelFlag, paginationDto.getDelFlag())
            .like(StringUtils.isNotBlank(paginationDto.getBusinessFormatName()),
                    BusinessFormat::getBusinessFormatName, paginationDto.getBusinessFormatName())
            .like(StringUtils.isNotBlank(paginationDto.getBusinessFormatCode()),
                    BusinessFormat::getBusinessFormatCode, paginationDto.getBusinessFormatCode())
            //状态
            .eq(StringUtils.isNotBlank(paginationDto.getEnableStatus()),
                    BusinessFormat::getEnableStatus, paginationDto.getEnableStatus())
            //来源
            .eq(StringUtils.isNotBlank(paginationDto.getSource()),
                    BusinessFormat::getSource, paginationDto.getSource())
            //更新人
            .like(StringUtils.isNotBlank(paginationDto.getModifyName()),
                    BusinessFormat::getModifyName, paginationDto.getModifyName())
            //更新开始时间
            .ge(paginationDto.getModifyStartTime() != null,
                    BusinessFormat::getModifyTime, paginationDto.getModifyStartTime())
            //更新结束时间
            .le(paginationDto.getModifyEndTime() != null,
                    BusinessFormat::getModifyTime, paginationDto.getModifyEndTime())
            .page(page);
  }

  @Override
  public List<CommonSelectVo> findBrandFormSelectList(CommonSelectDto dto) {
    QueryWrapper<BusinessFormat> queryWrapper = new QueryWrapper<>();
    if (ObjectUtils.isNotEmpty(dto)) {
      if (StringUtils.isNotBlank(dto.getCodeOrName())) {
        queryWrapper.lambda().like(BusinessFormat::getBusinessFormatName, dto.getCodeOrName())
                .or().like(BusinessFormat::getBusinessFormatCode, dto.getCodeOrName());
      }
      if (StringUtils.isNotBlank(dto.getSelectedCode())) {
        queryWrapper.lambda().ne(BusinessFormat::getBusinessFormatCode, dto.getSelectedCode());
      }
      if (!CollectionUtils.isEmpty(dto.getSelectedCodeList())) {
        queryWrapper.lambda().notIn(BusinessFormat::getBusinessFormatCode, dto.getSelectedCodeList());
      }
    }
    List<BusinessFormat> list = this.businessFormatRepository.list(queryWrapper);
    if (CollectionUtils.isEmpty(list)) {
      return null;
    }
    return list.stream().map(s -> {
      CommonSelectVo vo = new CommonSelectVo();
      vo.setValue(s.getBusinessFormatName());
      vo.setCode(s.getBusinessFormatCode());
      return vo;
    }).collect(Collectors.toList());
  }

  /**
   * 创建验证
   *
   * @param businessFormat
   */
  private void createValidate(BusinessFormat businessFormat) {
    Validate.notNull(businessFormat, "新增时，对象信息不能为空！");
    Validate.notEmpty(businessFormat.getBusinessFormatName(), "新增时，业态名称不能为空");
    BusinessFormat one = businessFormatRepository.lambdaQuery().eq(BusinessFormat::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .eq(BusinessFormat::getBusinessFormatName, businessFormat.getBusinessFormatName()).one();
    Validate.isTrue(one == null, "业态名称已存在");
    businessFormat.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    businessFormat.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    businessFormat.setSource(DataSourceEnum.SELF_CREATE_SOURCE.getCode());
    businessFormat.setId(null);

  }

  /**
   * 修改验证
   *
   * @param businessFormat
   */
  private void updateValidate(BusinessFormat businessFormat) {
    Validate.notNull(businessFormat, "修改时，对象信息不能为空！");
    Validate.notBlank(businessFormat.getId(), "修改数据时，id不能为空！");
    Validate.notEmpty(businessFormat.getBusinessFormatName(), "修改数据时，业态名称不能为空");
    BusinessFormat one = businessFormatRepository.lambdaQuery().eq(BusinessFormat::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .eq(BusinessFormat::getBusinessFormatName, businessFormat.getBusinessFormatName().trim())
            .ne(BusinessFormat::getId, businessFormat.getId()).one();
    Validate.isTrue(one == null, "业态名称已存在");
    businessFormat.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    businessFormat.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    businessFormat.setSource(DataSourceEnum.SELF_CREATE_SOURCE.getCode());
  }
}

