package com.biz.crm.mdm.business.product.unit.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.mdm.business.product.unit.local.entity.MdmUnitEntity;
import com.biz.crm.mdm.business.product.unit.local.repository.UnitRepository;


import com.biz.crm.mdm.business.product.unit.local.service.UnitService;
import com.biz.crm.mdm.business.product.unit.sdk.dto.UnitPaginationQueryDto;
import com.biz.crm.mdm.business.product.unit.sdk.enums.DataSourceEnum;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.beans.factory.annotation.Autowired;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;

import javax.sql.DataSource;
import java.util.List;


/**
 * 单位表(Unit)表服务实现类
 *
 * @author makejava
 * @since 2023-03-01 14:49:24
 */
@Service("unitService")
public class UnitServiceImpl implements UnitService {

  @Autowired
  private UnitRepository unitRepository;

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public MdmUnitEntity findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.unitRepository.getById(id);
  }

  /**
   * 新增数据
   *
   * @param unit 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public MdmUnitEntity create(MdmUnitEntity unit) {
    this.createValidate(unit);
    this.unitRepository.saveOrUpdate(unit);
    return unit;
  }

  /**
   * 修改新据
   *
   * @param unit 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public MdmUnitEntity update(MdmUnitEntity unit) {
    this.updateValidate(unit);
    this.unitRepository.saveOrUpdate(unit);
    return unit;
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    List<MdmUnitEntity> units = this.unitRepository.findByIds(idList);
    Validate.isTrue(!CollectionUtils.isEmpty(units), "删除数据时，未查询到需要删除的数据！");
    units.stream().forEach(unit -> unit.setDelFlag(DelFlagStatusEnum.DELETE.getCode()));
    this.unitRepository.saveOrUpdateBatch(units);
  }

  /**
   * 启用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 启用结果
   */
  @Transactional
  @Override
  public void enable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "启用数据时，主键集合不能为空！");
    List<MdmUnitEntity> units = this.unitRepository.findEnableStatusByIds(EnableStatusEnum.DISABLE, idList);
    if(CollectionUtils.isEmpty(units)){
      return;
    }
    units.stream().forEach(unit -> unit.setEnableStatus(EnableStatusEnum.ENABLE.getCode()));
    this.unitRepository.saveOrUpdateBatch(units);
  }

  /**
   * 禁用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 禁用结果
   */
  @Transactional
  @Override
  public void disable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "禁用数据时，主键集合不能为空！");
    List<MdmUnitEntity> units = this.unitRepository.findEnableStatusByIds(EnableStatusEnum.ENABLE, idList);
    if(CollectionUtils.isEmpty(units)){
      return;
    }
    units.stream().forEach(unit -> unit.setEnableStatus(EnableStatusEnum.DISABLE.getCode()));
    this.unitRepository.saveOrUpdateBatch(units);
  }

  @Override
  public Page<MdmUnitEntity> findByConditions(Pageable pageable, UnitPaginationQueryDto paginationDto) {
    Page<MdmUnitEntity> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.unitRepository.lambdaQuery()
            //租户编码
            .eq(StringUtils.isNotBlank(paginationDto.getTenantCode()),
                    MdmUnitEntity::getTenantCode,
                    paginationDto.getTenantCode())
            //数据来源
            .eq(StringUtils.isNotBlank(paginationDto.getSource()),
                    MdmUnitEntity::getSource,
                    paginationDto.getSource())
            //单位编码
            .like(StringUtils.isNotBlank(paginationDto.getUnitCode()),
                    MdmUnitEntity::getUnitCode,
                    paginationDto.getUnitCode())
            //单位名称
            .like(StringUtils.isNotBlank(paginationDto.getUnitName()),
                    MdmUnitEntity::getUnitName,
                    paginationDto.getUnitName())
            //更新人
            .like(StringUtils.isNotBlank(paginationDto.getModifyName()),
                    MdmUnitEntity::getModifyName,
                    paginationDto.getModifyName())
            //禁用状态
            .eq(StringUtils.isNotBlank(paginationDto.getEnableStatus()),
                    MdmUnitEntity::getEnableStatus,
                    paginationDto.getEnableStatus())
            //删除状态
            .eq(StringUtils.isNotBlank(paginationDto.getDelFlag()),
                    MdmUnitEntity::getDelFlag,
                    paginationDto.getDelFlag())
            //删除状态
            .eq(StringUtils.isBlank(paginationDto.getDelFlag()),
                    MdmUnitEntity::getDelFlag,
                    DelFlagStatusEnum.NORMAL.getCode())
            .page(page);
  }

  @Override
  public List<MdmUnitEntity> getByCodes(List<String> codes) {
    return unitRepository
            .lambdaQuery()
            .eq(MdmUnitEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .eq(MdmUnitEntity::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
            .in(MdmUnitEntity::getUnitCode, codes).list();

  }

  /**
   * 创建验证
   *
   * @param unit
   */
  private void createValidate(MdmUnitEntity unit) {
    Validate.notNull(unit, "新增时，对象信息不能为空！");
    unit.setId(null);
//    Validate.notBlank(unit.getTenantCode(), "新增数据时，租户编号不能为空！");
    Validate.notBlank(unit.getUnitCode(), "新增数据时，单位编码不能为空！");
    Validate.notBlank(unit.getUnitName(), "新增数据时，单位名称不能为空！");
    MdmUnitEntity codeSameUnit = this.unitRepository
            .lambdaQuery()
            .eq(MdmUnitEntity::getUnitCode, unit.getUnitCode())
            .eq(MdmUnitEntity::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
            .one();
    Validate.isTrue(ObjectUtils.isEmpty(codeSameUnit), "单位编码已存在");
    MdmUnitEntity nameSameUnit = this.unitRepository
            .lambdaQuery().eq(MdmUnitEntity::getUnitName, unit.getUnitName()).one();
    Validate.isTrue(ObjectUtils.isEmpty(nameSameUnit), "单位名称已存在");
    unit.setTenantCode(TenantUtils.getTenantCode());
    unit.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    if (StringUtils.isBlank(unit.getEnableStatus())) {
      unit.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    }
    unit.setSource(DataSourceEnum.SELF_CREATE_SOURCE.getCode());
  }

  /**
   * 修改验证
   *
   * @param unit
   */
  private void updateValidate(MdmUnitEntity unit) {
    Validate.notNull(unit, "修改时，对象信息不能为空！");
    Validate.notBlank(unit.getId(), "修改数据时，不能为空！");
    Validate.notBlank(unit.getTenantCode(), "修改数据时，租户编号不能为空！");
    Validate.notBlank(unit.getUnitCode(), "修改数据时，单位编码不能为空！");
    Validate.notBlank(unit.getUnitName(), "修改数据时，单位名称不能为空！");

    Validate.isTrue(DataSourceEnum.SELF_CREATE_SOURCE.getCode().equals(unit.getSource()), "修改数据时，非自建数据不能修改！");
    MdmUnitEntity codeSameUnit = this.unitRepository.lambdaQuery().ne(MdmUnitEntity::getId, unit.getId())
            .eq(MdmUnitEntity::getUnitCode, unit.getUnitCode()).one();
    Validate.isTrue(ObjectUtils.isEmpty(codeSameUnit), "单位编码已存在");
    MdmUnitEntity nameSameUnit = this.unitRepository.lambdaQuery().ne(MdmUnitEntity::getId, unit.getId())
            .eq(MdmUnitEntity::getUnitName, unit.getUnitName()).one();
    Validate.isTrue(ObjectUtils.isEmpty(nameSameUnit), "单位名称已存在");
    unit.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    if (StringUtils.isBlank(unit.getEnableStatus())) {
      unit.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    }
  }
}

