package com.biz.crm.mdm.business.businessunit.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.businessunit.local.entity.MdmBusinessUnit;
import com.biz.crm.mdm.business.businessunit.local.entity.MdmBusinessUnitDetail;
import com.biz.crm.mdm.business.businessunit.local.repository.MdmBusinessUnitDetailRepository;
import com.biz.crm.mdm.business.businessunit.local.repository.MdmBusinessUnitRepository;
import com.biz.crm.mdm.business.businessunit.sdk.constant.BusinessUnitConstant;
import com.biz.crm.mdm.business.businessunit.sdk.dto.BusinessUnitEventDto;
import com.biz.crm.mdm.business.businessunit.sdk.dto.MdmBusinessUnitDetailDto;
import com.biz.crm.mdm.business.businessunit.sdk.dto.MdmBusinessUnitDto;
import com.biz.crm.mdm.business.businessunit.sdk.dto.MdmFormatCodeDto;
import com.biz.crm.mdm.business.businessunit.sdk.event.BusinessUnitEventListener;
import com.biz.crm.mdm.business.businessunit.sdk.service.MdmBusinessUnitVoService;
import com.biz.crm.mdm.business.businessunit.sdk.vo.MdmBusinessUnitDetailVo;
import com.biz.crm.mdm.business.businessunit.sdk.vo.MdmBusinessUnitVo;
import com.biz.crm.mdm.business.dictionary.sdk.dto.DictDataDto;
import com.biz.crm.mdm.business.dictionary.sdk.dto.DictTypeDto;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictTypeVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictTypeVo;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

/**
 * 业务单元(MdmBusinessUnit)
 *
 * @author yaoyongming
 * @since 2022-10-18 17:28:36
 */
@Service
public class MdmBusinessUnitServiceImpl implements MdmBusinessUnitVoService {

    @Autowired(required = false)
    private MdmBusinessUnitRepository mdmBusinessUnitRepository;

    @Autowired(required = false)
    private MdmBusinessUnitDetailRepository mdmBusinessUnitDetailRepository;

    @Autowired(required = false)
    private GenerateCodeService generateCodeService;

    @Autowired(required = false)
    private DictDataVoService dictDataVoService;

    @Autowired(required = false)
    private DictTypeVoService dictTypeVoService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;
    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private SalesOrgVoService salesOrgVoService;

    /**
     * 多条件分页查询
     *
     * @param pageable 分页参数
     * @param dto      查询参数
     * @return com.baomidou.mybatisplus.extension.plugins.pagination.Page<com.biz.crm.mdm.business.businessunit.sdk.vo.MdmBusinessUnitVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.5 14:15
     */
    @Override
    public Page<MdmBusinessUnitVo> findByConditions(Pageable pageable, MdmBusinessUnitDto dto) {
        ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));

        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        dto = Optional.ofNullable(dto).orElse(new MdmBusinessUnitDto());
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        Page<MdmBusinessUnitVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.mdmBusinessUnitRepository.findByConditions(page, dto);
    }


    /**
     * 根据ID查询详情
     *
     * @param id ID
     * @return com.biz.crm.mdm.business.businessunit.sdk.vo.MdmBusinessUnitVo
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.5 0:25
     */
    @Override
    public MdmBusinessUnitVo findById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        MdmBusinessUnit entity = this.mdmBusinessUnitRepository.getById(id);
        if (Objects.isNull(entity)) {
            return null;
        }
        MdmBusinessUnitVo mdmBusinessUnitVo = this.nebulaToolkitService.copyObjectByBlankList(
                entity, MdmBusinessUnitVo.class,
                HashSet.class, ArrayList.class);
        List<MdmBusinessUnitDetail> mdmBusinessUnitDetailList = this.mdmBusinessUnitDetailRepository.listByUnitCodes(Lists.newArrayList(entity.getBusinessUnitCode()));
        if (!CollectionUtils.isEmpty(mdmBusinessUnitDetailList)) {
            List<MdmBusinessUnitDetailVo> mdmBusinessUnitDetailVos = (List<MdmBusinessUnitDetailVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
                    mdmBusinessUnitDetailList, MdmBusinessUnitDetail.class, MdmBusinessUnitDetailVo.class, LinkedHashSet.class, ArrayList.class
            );
            mdmBusinessUnitVo.setUnitDetailList(mdmBusinessUnitDetailVos);
        }
        return mdmBusinessUnitVo;
    }

    /**
     * 根据ID集合查询详情
     *
     * @param idList ID集合
     * @return java.util.List<com.biz.crm.mdm.business.businessunit.sdk.vo.MdmBusinessUnitVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.5 0:26
     */
    @Override
    public List<MdmBusinessUnitVo> findByIds(List<String> idList) {
        if (CollectionUtils.isEmpty(idList)) {
            return Collections.emptyList();
        }
        List<MdmBusinessUnit> entityList = this.mdmBusinessUnitRepository.listByIds(idList);
        if (CollectionUtils.isEmpty(entityList)) {
            return Collections.emptyList();
        }
        List<MdmBusinessUnitVo> mdmBusinessUnitVos = (List<MdmBusinessUnitVo>) this.nebulaToolkitService.copyCollectionByBlankList(
                entityList, MdmBusinessUnit.class, MdmBusinessUnitVo.class,
                HashSet.class, ArrayList.class);
        List<String> codeList = entityList.stream().map(MdmBusinessUnit::getBusinessUnitCode).collect(Collectors.toList());
        List<MdmBusinessUnitDetail> mdmBusinessUnitDetailList = this.mdmBusinessUnitDetailRepository.listByUnitCodes(codeList);
        // 设置明细数据
        if (!CollectionUtils.isEmpty(mdmBusinessUnitDetailList)) {
            List<MdmBusinessUnitDetailVo> mdmBusinessUnitDetailVos = (List<MdmBusinessUnitDetailVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
                    mdmBusinessUnitDetailList, MdmBusinessUnitDetail.class, MdmBusinessUnitDetailVo.class, LinkedHashSet.class, ArrayList.class
            );
            Map<String, List<MdmBusinessUnitDetailVo>> detailMap = mdmBusinessUnitDetailVos.stream()
                    .collect(Collectors.groupingBy(MdmBusinessUnitDetailVo::getBusinessUnitCode));
            mdmBusinessUnitVos.forEach(mdmBusinessUnitVo -> {
                mdmBusinessUnitVo.setUnitDetailList(detailMap.get(mdmBusinessUnitVo.getBusinessUnitCode()));
            });
        }
        return mdmBusinessUnitVos;
    }

    /**
     * 根据编码查询详情
     *
     * @param code 编码
     * @return com.biz.crm.mdm.business.businessunit.sdk.vo.MdmBusinessUnitVo
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.5 0:27
     */
    @Override
    public MdmBusinessUnitVo findByCode(String code) {
        if (StringUtils.isBlank(code)) {
            return null;
        }

        MdmBusinessUnit entity = this.mdmBusinessUnitRepository.findByCode(code);
        if (Objects.isNull(entity)) {
            return null;
        }
        MdmBusinessUnitVo mdmBusinessUnitVo = this.nebulaToolkitService.copyObjectByBlankList(
                entity, MdmBusinessUnitVo.class,
                HashSet.class, ArrayList.class);
        List<MdmBusinessUnitDetail> mdmBusinessUnitDetailList = this.mdmBusinessUnitDetailRepository.listByUnitCodes(Lists.newArrayList(entity.getBusinessUnitCode()));
        if (!CollectionUtils.isEmpty(mdmBusinessUnitDetailList)) {
            List<MdmBusinessUnitDetailVo> mdmBusinessUnitDetailVos = (List<MdmBusinessUnitDetailVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
                    mdmBusinessUnitDetailList, MdmBusinessUnitDetail.class, MdmBusinessUnitDetailVo.class, LinkedHashSet.class, ArrayList.class
            );
            mdmBusinessUnitVo.setUnitDetailList(mdmBusinessUnitDetailVos);
        }
        return mdmBusinessUnitVo;
    }

    /**
     * 根据编码集合查询详情
     *
     * @param codeList 编码集合
     * @return java.util.List<com.biz.crm.mdm.business.businessunit.sdk.vo.MdmBusinessUnitVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.5 0:27
     */
    @Override
    public List<MdmBusinessUnitVo> findByCodes(List<String> codeList) {
        if (CollectionUtils.isEmpty(codeList)) {
            return Collections.emptyList();
        }
        List<MdmBusinessUnit> entityList = this.mdmBusinessUnitRepository.findByCodes(codeList);
        if (CollectionUtils.isEmpty(entityList)) {
            return Collections.emptyList();
        }
        List<MdmBusinessUnitVo> mdmBusinessUnitVos = (List<MdmBusinessUnitVo>) this.nebulaToolkitService.copyCollectionByBlankList(
                entityList, MdmBusinessUnit.class, MdmBusinessUnitVo.class,
                HashSet.class, ArrayList.class);
        List<MdmBusinessUnitDetail> mdmBusinessUnitDetailList = this.mdmBusinessUnitDetailRepository.listByUnitCodes(codeList);
        // 设置明细数据
        if (!CollectionUtils.isEmpty(mdmBusinessUnitDetailList)) {
            List<MdmBusinessUnitDetailVo> mdmBusinessUnitDetailVos = (List<MdmBusinessUnitDetailVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
                    mdmBusinessUnitDetailList, MdmBusinessUnitDetail.class, MdmBusinessUnitDetailVo.class, LinkedHashSet.class, ArrayList.class
            );
            Map<String, List<MdmBusinessUnitDetailVo>> detailMap = mdmBusinessUnitDetailVos.stream().collect(Collectors.groupingBy(MdmBusinessUnitDetailVo::getBusinessUnitCode));
            mdmBusinessUnitVos.forEach(mdmBusinessUnitVo -> {
                mdmBusinessUnitVo.setUnitDetailList(detailMap.get(mdmBusinessUnitVo.getBusinessUnitCode()));
            });
        }
        return mdmBusinessUnitVos;
    }

    /**
     * 新增
     *
     * @param dto
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public MdmBusinessUnitVo create(MdmBusinessUnitDto dto) {
        this.createValidate(dto);
        // redis生成料编码code
        List<String> codeList = this.generateCodeService.generateCode(BusinessUnitConstant.UNIT_CODE, 1);
        dto.setBusinessUnitCode(codeList.get(0));
        dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        dto.setTenantCode(TenantUtils.getTenantCode());
        MdmBusinessUnit entity = this.nebulaToolkitService.copyObjectByBlankList(
                dto, MdmBusinessUnit.class,
                HashSet.class, ArrayList.class);
        this.mdmBusinessUnitRepository.saveOrUpdate(entity);
        // 更新明细表数据
        List<MdmBusinessUnitDetailDto> unitDetailList = dto.getUnitDetailList();
        unitDetailList.forEach(unitDetail -> {
            unitDetail.setBusinessUnitCode(entity.getBusinessUnitCode());
            unitDetail.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            unitDetail.setTenantCode(TenantUtils.getTenantCode());
        });
        this.mdmBusinessUnitDetailRepository.saveOrUpdateBatch(unitDetailList);
        //更新数据字典信息
        this.saveOrUpdateDictListByBusinessUnit(Collections.singletonList(dto));
        BusinessUnitEventDto eventDto = new BusinessUnitEventDto();
        eventDto.setOriginal(null);
        eventDto.setNewest(this.nebulaToolkitService.copyObjectByBlankList(
                entity, MdmBusinessUnitDto.class,
                HashSet.class, ArrayList.class));
        SerializableBiConsumer<BusinessUnitEventListener, BusinessUnitEventDto> consumer = BusinessUnitEventListener::onCreate;
        this.nebulaNetEventClient.publish(eventDto, BusinessUnitEventListener.class, consumer);
        return this.nebulaToolkitService.copyObjectByBlankList(
                entity, MdmBusinessUnitVo.class,
                HashSet.class, ArrayList.class);

    }

    /**
     * 修改
     *
     * @param dto
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public MdmBusinessUnitVo update(MdmBusinessUnitDto dto) {
        this.updateValidate(dto);

        BusinessUnitEventDto eventDto = new BusinessUnitEventDto();
        eventDto.setOriginal(dto);
        MdmBusinessUnit entity = this.nebulaToolkitService.copyObjectByBlankList(
                dto, MdmBusinessUnit.class,
                HashSet.class, ArrayList.class);
        this.mdmBusinessUnitRepository.saveOrUpdate(entity);
        // 更新明细表数据
        List<MdmBusinessUnitDetailDto> unitDetailList = dto.getUnitDetailList();
        unitDetailList.forEach(unitDetail -> {
            unitDetail.setBusinessUnitCode(entity.getBusinessUnitCode());
            unitDetail.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            unitDetail.setTenantCode(TenantUtils.getTenantCode());
        });
        this.mdmBusinessUnitDetailRepository.saveOrUpdateBatch(unitDetailList);
        //更新数据字典信息
        this.saveOrUpdateDictListByBusinessUnit(Collections.singletonList(dto));

        //更新销售组织上的业态信息
        Set<String> salesOrgCodeSet = new HashSet<>();
        if (!CollectionUtils.isEmpty(unitDetailList)) {
            salesOrgCodeSet = unitDetailList.stream().filter(i -> StringUtils.isNotBlank(i.getSalesOrgCode())).map(MdmBusinessUnitDetailDto::getSalesOrgCode).collect(Collectors.toSet());
        }
        if(!salesOrgCodeSet.isEmpty()){
            this.salesOrgVoService.editSalesOrgBusinessUnit(entity.getBusinessUnitCode(),new ArrayList<>(salesOrgCodeSet));
        }

        eventDto.setNewest(this.nebulaToolkitService.copyObjectByBlankList(
                entity, MdmBusinessUnitDto.class,
                HashSet.class, ArrayList.class));
        SerializableBiConsumer<BusinessUnitEventListener, BusinessUnitEventDto> consumer = BusinessUnitEventListener::onUpdate;
        this.nebulaNetEventClient.publish(eventDto, BusinessUnitEventListener.class, consumer);
        return this.nebulaToolkitService.copyObjectByBlankList(
                entity, MdmBusinessUnitVo.class,
                HashSet.class, ArrayList.class);
    }

    /**
     * 删除
     *
     * @param idList
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<String> idList) {
        Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
        List<MdmBusinessUnit> businessUnitList = this.mdmBusinessUnitRepository.findByIds(idList);
        Set<String> dictCodeSet = new HashSet<>();
        businessUnitList.forEach(entity -> {
            String businessUnitCode = entity.getBusinessUnitCode();
            dictCodeSet.add(businessUnitCode);
            entity.setBusinessUnitCode(businessUnitCode + entity.getId());
            entity.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
        });
        this.mdmBusinessUnitRepository.updateBatchById(businessUnitList);
        // 删除明细
        this.mdmBusinessUnitDetailRepository.delByUnitCode(Lists.newArrayList(dictCodeSet));
        businessUnitList.forEach(entity -> {
            BusinessUnitEventDto eventDto = new BusinessUnitEventDto();
            eventDto.setOriginal(this.nebulaToolkitService.copyObjectByBlankList(
                    entity, MdmBusinessUnitDto.class,
                    HashSet.class, ArrayList.class));
            eventDto.setNewest(null);
            SerializableBiConsumer<BusinessUnitEventListener, BusinessUnitEventDto> consumer = BusinessUnitEventListener::onDelete;
            this.nebulaNetEventClient.publish(eventDto, BusinessUnitEventListener.class, consumer);
        });
        this.delDictByDictCode(dictCodeSet);
    }

    /**
     * 新建校验
     *
     * @param mdmUnit
     */
    private void createValidate(MdmBusinessUnitDto mdmUnit) {
        this.commonValidate(mdmUnit);
        mdmUnit.setId(null);

    }

    /**
     * 修改校验
     *
     * @param mdmUnit
     */
    private void updateValidate(MdmBusinessUnitDto mdmUnit) {
        this.commonValidate(mdmUnit);
        Validate.notBlank(mdmUnit.getId(), "更新时数据主键不能为空！");
        MdmBusinessUnit businessUnit = this.mdmBusinessUnitRepository.findById(mdmUnit.getId());
        Validate.notNull(businessUnit, "更新数据不存在！");
        Validate.isTrue(businessUnit.getBusinessUnitCode().equals(mdmUnit.getBusinessUnitCode()),
                "编码不可修改！");
    }

    /**
     * 基础共同校验
     *
     * @param mdmUnit
     */
    private void commonValidate(MdmBusinessUnitDto mdmUnit) {
        Validate.notBlank(mdmUnit.getBusinessUnitName(), "业务名称不能为空！");
        Validate.notEmpty(mdmUnit.getUnitDetailList(), "明细不能为空！");
        List<MdmBusinessUnitDetailDto> unitDetailList = mdmUnit.getUnitDetailList();
        // 不能重复填
        Map<String, List<MdmBusinessUnitDetailDto>> map =
                unitDetailList.stream().collect(Collectors.groupingBy(vo -> vo.getBusinessFormatCode() + vo.getSalesOrgCode() + vo.getChannelCode()));
        Validate.isTrue(map.size() == unitDetailList.size(), "不能填相同维度明细！");
        // 至少要填一项
        unitDetailList.forEach(unitDetail -> {
            if (StringUtils.isEmpty(unitDetail.getBusinessFormatCode())
                    && StringUtils.isEmpty(unitDetail.getSalesOrgCode())
                    && StringUtils.isEmpty(unitDetail.getChannelCode())) {
                throw new RuntimeException("业态、销售组织、渠道不能全部为空！");
            }
            if (StringUtils.isNotEmpty(unitDetail.getSalesOrgCode())
                    && StringUtils.isEmpty(unitDetail.getSalesOrgName())) {
                throw new RuntimeException("销售组织名称不能为空！");
            }
            if (StringUtils.isNotEmpty(unitDetail.getChannelCode())
                    && StringUtils.isEmpty(unitDetail.getChannelName())) {
                throw new RuntimeException("渠道名称不能为空！");
            }
        });
    }

    /**
     * 批量启用
     *
     * @param ids
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        Validate.notEmpty(ids, "待启用的数据主键不能为空");
        List<MdmBusinessUnit> businessUnitList = this.mdmBusinessUnitRepository.findByIds(ids);
        Validate.notEmpty(businessUnitList, "待启用的数据不存在或已删除!");
        this.mdmBusinessUnitRepository.updateEnableStatusByIdIn(EnableStatusEnum.ENABLE, ids);
        List<MdmBusinessUnitDto> list = (List<MdmBusinessUnitDto>) this.nebulaToolkitService.copyCollectionByBlankList(
                businessUnitList, MdmBusinessUnit.class, MdmBusinessUnitDto.class,
                HashSet.class, ArrayList.class);
        list.forEach(dto -> {
            BusinessUnitEventDto eventDto = new BusinessUnitEventDto();
            eventDto.setOriginal(dto);
            eventDto.setNewest(null);
            SerializableBiConsumer<BusinessUnitEventListener, BusinessUnitEventDto> consumer = BusinessUnitEventListener::onEnable;
            this.nebulaNetEventClient.publish(eventDto, BusinessUnitEventListener.class, consumer);
        });
        //更新数据字典信息
        this.saveOrUpdateDictListByBusinessUnit(list);
    }

    /**
     * 批量禁用
     *
     * @param ids
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        Validate.notEmpty(ids, "待禁用的数据主键不能为空");
        List<MdmBusinessUnit> businessUnitList = this.mdmBusinessUnitRepository.findByIds(ids);
        Validate.notEmpty(businessUnitList, "待禁用的数据不存在或已删除!");
        Set<String> dictCodeSet = businessUnitList.stream()
                .filter(k -> StringUtils.isNotBlank(k.getBusinessUnitCode()))
                .map(MdmBusinessUnit::getBusinessUnitCode)
                .collect(Collectors.toSet());
        this.mdmBusinessUnitRepository.updateEnableStatusByIdIn(EnableStatusEnum.DISABLE, ids);
        List<MdmBusinessUnitDto> list = (List<MdmBusinessUnitDto>) this.nebulaToolkitService.copyCollectionByBlankList(
                businessUnitList, MdmBusinessUnit.class, MdmBusinessUnitDto.class,
                HashSet.class, ArrayList.class);
        list.forEach(entity -> {
            BusinessUnitEventDto eventDto = new BusinessUnitEventDto();
            eventDto.setOriginal(entity);
            eventDto.setNewest(null);
            SerializableBiConsumer<BusinessUnitEventListener, BusinessUnitEventDto> consumer = BusinessUnitEventListener::onDisable;
            this.nebulaNetEventClient.publish(eventDto, BusinessUnitEventListener.class, consumer);
        });
        this.delDictByDictCode(dictCodeSet);
    }

    /**
     * 查询业态分页列表（通过数据字典转换，数据权限使用请勿修改）
     *
     * @param pageable 分页参数
     * @param codeDto  筛选参数
     * @return Page<DictDataVo>
     */
    @Override
    public Page<DictDataVo> findFormatByConditions(Pageable pageable, MdmFormatCodeDto codeDto) {
        //定义一个集合
        if (pageable == null) {
            pageable = PageRequest.of(0, 50);
        }
        Validate.notNull(codeDto, "请求参数不能为空！");

        // 手动分页
        List<DictDataVo> allList = dictDataVoService.findByDictTypeCode(BusinessUnitConstant.MDM_BUSINESS_FORMAT);
        List<DictDataVo> newList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(allList)) {
            if (StringUtils.isNotBlank(codeDto.getDictValue())) {
                for (DictDataVo vo : allList) {
                    if (vo.getDictValue().contains(codeDto.getDictValue())) {
                        newList.add(vo);
                    }
                }
            } else {
                newList = allList;
            }
        }
        List<DictDataVo> pageList = newList.stream()
                .skip((long) (pageable.getPageNumber() - 1) * pageable.getPageSize())
                .limit(pageable.getPageSize()).collect(Collectors.toList());
        Page<DictDataVo> page = new Page<>();
        page.setRecords(pageList);
        page.setSize(pageable.getPageSize());
        page.setTotal(newList.size());
        return page;
    }

    /**
     * 根据数据字典编码删除数据字典
     *
     * @param dictCodeSet
     */
    private void delDictByDictCode(Set<String> dictCodeSet) {
        if (CollectionUtils.isEmpty(dictCodeSet)) {
            return;
        }
        //查询已关联数据字典
        DictTypeVo dictTypeVo = dictTypeVoService.findByDictTypeCode(BusinessUnitConstant.MDM_BUSINESS_UNIT);
        Map<String, DictDataVo> dictMap = new HashMap<>(8);
        if (Objects.nonNull(dictTypeVo)) {
            List<DictDataVo> dictDataVoList = dictDataVoService.findByDictTypeCode(BusinessUnitConstant.MDM_BUSINESS_UNIT);
            if (!CollectionUtils.isEmpty(dictDataVoList)) {
                dictDataVoList.forEach(entity -> {
                    dictMap.put(entity.getDictCode(), entity);
                });
            }
        }
        List<String> dictIdList = new ArrayList<>();
        dictCodeSet.forEach(dictCode -> {
            if (dictMap.containsKey(dictCode)) {
                dictIdList.add(dictMap.get(dictCode).getId());
            }
        });
        if (!CollectionUtils.isEmpty(dictIdList)) {
            dictDataVoService.deleteBatch(dictIdList);
        }
    }

    /**
     * 根据业务单元信息新增或更新数据字典
     *
     * @param businessUnitList
     */
    private void saveOrUpdateDictListByBusinessUnit(List<MdmBusinessUnitDto> businessUnitList) {
        if (CollectionUtils.isEmpty(businessUnitList)) {
            return;
        }
        //查询已关联数据字典
        DictTypeVo dictTypeVo = dictTypeVoService.findByDictTypeCode(BusinessUnitConstant.MDM_BUSINESS_UNIT);
        if (Objects.isNull(dictTypeVo)) {
            DictTypeDto dictTypeDto = new DictTypeDto();
            dictTypeDto.setDictTypeCode(BusinessUnitConstant.MDM_BUSINESS_UNIT);
            dictTypeDto.setDictTypeName("业务单元");
            dictTypeDto.setDictTypeModule("MDM");
            dictTypeVoService.save(dictTypeDto);
        }
        List<DictDataVo> dictDataVoList = dictDataVoService.findByDictTypeCode(BusinessUnitConstant.MDM_BUSINESS_UNIT);
        Set<String> dictDataVoSet = dictDataVoList
                .stream().map(DictDataVo::getDictCode).collect(Collectors.toSet());
        AtomicInteger dictSortAtomic = new AtomicInteger(dictDataVoList.stream()
                .filter(k -> Objects.nonNull(k.getDictSort()))
                .map(DictDataVo::getDictSort)
                .mapToInt(i -> i).max().orElse(0));
        businessUnitList.stream()
                .filter(k -> !dictDataVoSet.contains(k.getBusinessUnitCode()))
                .forEach(businessUnit -> {
                    DictDataDto dataDto = new DictDataDto();
                    dataDto.setDictCode(businessUnit.getBusinessUnitCode());
                    dataDto.setDictValue(businessUnit.getBusinessUnitName());
                    dataDto.setDictDesc(businessUnit.getBusinessUnitName());
                    dataDto.setDictSort(dictSortAtomic.incrementAndGet());
                    dataDto.setDictTypeCode(BusinessUnitConstant.MDM_BUSINESS_UNIT);
                    dictDataVoService.save(dataDto);
                });

    }
}

