package com.biz.crm.mdm.business.channel.org.employee.local.service.internal;


import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.channel.org.employee.local.entity.ChannelOrgEmployee;
import com.biz.crm.mdm.business.channel.org.employee.local.repository.ChannelOrgEmployeeRepository;
import com.biz.crm.mdm.business.channel.org.employee.local.service.ChannelOrgEmployeeService;
import com.biz.crm.mdm.business.channel.org.employee.sdk.event.ChannelOrgEmployeeEventDto;
import com.biz.crm.mdm.business.channel.org.employee.sdk.event.ChannelOrgEmployeeEventListener;
import com.biz.crm.mdm.business.channel.org.employee.sdk.event.ChannelOrgEmployeeInfoEventDto;
import com.biz.crm.mdm.business.user.sdk.service.UserVoService;
import com.biz.crm.mdm.business.user.sdk.vo.UserVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

/**
 * 渠道组织员工表(ChannelOrgEmployee)表服务实现类
 *
 * @author songjingen
 * @since 2022-11-23 10:32:48
 */
@Service("channelOrgEmployeeService")
public class ChannelOrgEmployeeServiceImpl implements ChannelOrgEmployeeService {

  @Autowired
  private ChannelOrgEmployeeRepository channelOrgEmployeeRepository;
  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;
  @Autowired(required = false)
  private UserVoService userVoService;
  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public ChannelOrgEmployee findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    ChannelOrgEmployee orgEmployee = this.channelOrgEmployeeRepository.getById(id);
    return orgEmployee;
  }

  /**
   * 新增数据
   *
   * @param channelOrgEmployees 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public List<ChannelOrgEmployee> create(List<ChannelOrgEmployee> channelOrgEmployees) {
    Validate.isTrue(!CollectionUtils.isEmpty(channelOrgEmployees), "新增时，数据为空！");
    Map<String, ChannelOrgEmployee> employeeMap = channelOrgEmployees.stream().collect(Collectors.toMap(o -> StringUtils.joinWith(o.getChannelOrgCode(), o.getUserCode()), o -> o));
    Validate.isTrue(employeeMap.size() == channelOrgEmployees.size(), "新增数据时，存在同一渠道组织关联多个相同的企业用户！");
    //验收数据是否存在
    Set<String> channelOrgCodes = channelOrgEmployees.stream().map(ChannelOrgEmployee::getChannelOrgCode).collect(Collectors.toSet());
    Set<String> userCodes = channelOrgEmployees.stream().map(ChannelOrgEmployee::getUserCode).collect(Collectors.toSet());
    List<ChannelOrgEmployee> currentEmployeeList = this.channelOrgEmployeeRepository.findByUserCodesAndChannelOrgCodes(userCodes, channelOrgCodes);
    Map<String, ChannelOrgEmployee> currentMap = new HashMap<>();
    if (!CollectionUtils.isEmpty(currentEmployeeList)) {
      currentMap = currentEmployeeList.stream().collect(Collectors.toMap(o -> StringUtils.joinWith(o.getChannelOrgCode(), o.getUserCode()), o -> o));
    }
    for (ChannelOrgEmployee channelOrgEmployee : channelOrgEmployees) {
      this.createValidate(channelOrgEmployee);
      Validate.isTrue(currentMap.get(StringUtils.joinWith(channelOrgEmployee.getChannelOrgCode(), channelOrgEmployee.getUserCode())) == null, "新增数据时，存在同一渠道组织关联多个相同的企业用户！");
    }
    this.channelOrgEmployeeRepository.saveBatch(channelOrgEmployees);
    //推送事件,查询到用户数据才会推送
    ChannelOrgEmployeeEventDto channelOrgEmployeeEventDto = new ChannelOrgEmployeeEventDto();
    List<UserVo> userVos = this.userVoService.findByUserCodes(userCodes);
    if (!CollectionUtils.isEmpty(userVos)) {
      List<ChannelOrgEmployeeInfoEventDto> channelOrgEmployeeInfoEventDtos = (List<ChannelOrgEmployeeInfoEventDto>) this.nebulaToolkitService.copyCollectionByWhiteList(userVos, UserVo.class,
          ChannelOrgEmployeeInfoEventDto.class,
          HashSet.class, ArrayList.class);
      ChannelOrgEmployee channelOrgEmployee = channelOrgEmployees.get(0);
      channelOrgEmployeeEventDto.setChannelOrgCode(channelOrgEmployee.getChannelOrgCode());
      channelOrgEmployeeEventDto.setChannelOrgName(channelOrgEmployee.getChannelOrgName());
      channelOrgEmployeeEventDto.setChannelOrgEmployeeInfoEventDtoList(channelOrgEmployeeInfoEventDtos);
      SerializableBiConsumer<ChannelOrgEmployeeEventListener, ChannelOrgEmployeeEventDto> onCreate = ChannelOrgEmployeeEventListener::onCreate;
      this.nebulaNetEventClient.publish(channelOrgEmployeeEventDto, ChannelOrgEmployeeEventListener.class, onCreate);
    }
    return channelOrgEmployees;
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    List<ChannelOrgEmployee> channelOrgEmployees = this.channelOrgEmployeeRepository.findByIds(idList);
    Validate.isTrue(!CollectionUtils.isEmpty(channelOrgEmployees), "删除数据时，未查询到需要删除的数据！");
    channelOrgEmployees.stream().forEach(channelOrgEmployee -> channelOrgEmployee.setDelFlag(DelFlagStatusEnum.DELETE.getCode()));
    this.channelOrgEmployeeRepository.saveOrUpdateBatch(channelOrgEmployees);
    //推送事件
    ChannelOrgEmployeeEventDto channelOrgEmployeeEventDto = new ChannelOrgEmployeeEventDto();
    ChannelOrgEmployee channelOrgEmployee = channelOrgEmployees.get(0);
    channelOrgEmployeeEventDto.setChannelOrgCode(channelOrgEmployee.getChannelOrgCode());
    channelOrgEmployeeEventDto.setChannelOrgName(channelOrgEmployee.getChannelOrgName());
    channelOrgEmployeeEventDto.setUserCodes(channelOrgEmployees.stream().map(ChannelOrgEmployee::getUserCode).collect(Collectors.toSet()));
    SerializableBiConsumer<ChannelOrgEmployeeEventListener, ChannelOrgEmployeeEventDto> onDelete = ChannelOrgEmployeeEventListener::onDelete;
    this.nebulaNetEventClient.publish(channelOrgEmployeeEventDto, ChannelOrgEmployeeEventListener.class, onDelete);

  }

  /**
   * 启用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 启用结果
   */
  @Transactional
  @Override
  public void enable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "启用数据时，主键集合不能为空！");
    List<ChannelOrgEmployee> channelOrgEmployees = this.channelOrgEmployeeRepository.findEnableStatusByIds(EnableStatusEnum.DISABLE, idList);
    Validate.isTrue(!CollectionUtils.isEmpty(channelOrgEmployees), "启用数据时，未查询到需要启用的数据！");
    channelOrgEmployees.stream().forEach(channelOrgEmployee -> channelOrgEmployee.setEnableStatus(EnableStatusEnum.ENABLE.getCode()));
    this.channelOrgEmployeeRepository.saveOrUpdateBatch(channelOrgEmployees);
  }

  /**
   * 禁用（单个或者批量）
   *
   * @param idList 主键结合
   * @return 禁用结果
   */
  @Transactional
  @Override
  public void disable(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "禁用数据时，主键集合不能为空！");
    List<ChannelOrgEmployee> channelOrgEmployees = this.channelOrgEmployeeRepository.findEnableStatusByIds(EnableStatusEnum.ENABLE, idList);
    Validate.isTrue(!CollectionUtils.isEmpty(channelOrgEmployees), "禁用数据时，未查询到需要禁用的数据！");
    channelOrgEmployees.stream().forEach(channelOrgEmployee -> channelOrgEmployee.setEnableStatus(EnableStatusEnum.DISABLE.getCode()));
    this.channelOrgEmployeeRepository.saveOrUpdateBatch(channelOrgEmployees);
  }

  /**
   * 创建验证
   *
   * @param channelOrgEmployee
   */
  private void createValidate(ChannelOrgEmployee channelOrgEmployee) {
    Validate.notNull(channelOrgEmployee, "新增时，对象信息不能为空！");
    channelOrgEmployee.setId(null);
    channelOrgEmployee.setTenantCode(TenantUtils.getTenantCode());
    channelOrgEmployee.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    channelOrgEmployee.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    Validate.notBlank(channelOrgEmployee.getTenantCode(), "新增数据时，租户编号不能为空！");
    Validate.notBlank(channelOrgEmployee.getChannelOrgCode(), "新增数据时，渠道组织编码不能为空！");
    Validate.notBlank(channelOrgEmployee.getChannelOrgName(), "新增数据时，渠道组织名称不能为空！");
    Validate.notBlank(channelOrgEmployee.getUserCode(), "新增数据时，企业用户编码不能为空！");

  }

}

