package com.biz.crm.mdm.business.channel.org.local.repository;
/**
 * Created by Bao Hongbin on 2021-10-27 18:01.
 */


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.vo.LazyTreeVo;
import com.biz.crm.mdm.business.channel.org.local.entity.ChannelOrg;
import com.biz.crm.mdm.business.channel.org.local.mapper.ChannelOrgMapper;
import com.biz.crm.mdm.business.channel.org.sdk.dto.ChannelOrgPaginationDto;
import com.biz.crm.mdm.business.channel.org.sdk.vo.ChannelOrgVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

/**
 * @program: crm
 * @description: 渠道组织的数据库访问类
 * @author: Bao Hongbin
 * @create: 2021-10-27 18:01
 **/
@Component
public class ChannelOrgRepository extends ServiceImpl<ChannelOrgMapper, ChannelOrg> {

  /**
   * 根据渠道组织名称查询list
   *
   * @param customerOrgName
   * @param isLike
   * @param tenantCode
   * @return
   */
  public List<ChannelOrg> findListByName(String customerOrgName, boolean isLike, String tenantCode) {
    Validate.notBlank(customerOrgName, "渠道组织名称不能为空");
    return this.lambdaQuery()
        .eq(ChannelOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ChannelOrg::getTenantCode, tenantCode)
        .eq(!isLike, ChannelOrg::getChannelOrgName, customerOrgName)
        .like(isLike, ChannelOrg::getChannelOrgName, customerOrgName)
        .list();
  }

  /**
   * 根据渠道组织编码查询详情
   *
   * @param customerOrgCode
   * @param tenantCode
   * @return
   */
  public ChannelOrg findDetailsByCode(String customerOrgCode, String tenantCode) {
    Validate.notBlank(customerOrgCode, "渠道组织编码不能为空");
    return this.lambdaQuery()
        .eq(ChannelOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ChannelOrg::getTenantCode, tenantCode)
        .eq(ChannelOrg::getChannelOrgCode, customerOrgCode)
        .one();
  }

  /**
   * 根据渠道组织编码查询list
   *
   * @param customerOrgCode
   * @param tenantCode
   * @return
   */
  public List<ChannelOrg> findListByCode(String customerOrgCode, String tenantCode) {
    Validate.notBlank(customerOrgCode, "渠道组织编码不能为空");
    return this.lambdaQuery()
        .eq(ChannelOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ChannelOrg::getTenantCode, tenantCode)
        .eq(ChannelOrg::getChannelOrgCode, customerOrgCode)
        .list();
  }

  public ChannelOrg findDetailsById(String id, String tenantCode) {
    Validate.notBlank(id, "渠道组织id不能为空");
    return this.lambdaQuery()
        .eq(ChannelOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ChannelOrg::getTenantCode, tenantCode)
        .eq(ChannelOrg::getId, id)
        .one();
  }

  public List<ChannelOrg> findChildrenListByParentCode(String parentCode, String tenantCode) {
    return this.lambdaQuery()
        .eq(ChannelOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ChannelOrg::getTenantCode, tenantCode)
        .isNull(!StringUtils.isNotEmpty(parentCode), ChannelOrg::getParentCode)
        .eq(StringUtils.isNotEmpty(parentCode), ChannelOrg::getParentCode, parentCode).list();
  }

  /**
   * 通过id集合查询渠道组织数据
   *
   * @param ids
   * @param tenantCode
   * @return
   */
  public List<ChannelOrg> findListByIds(List<String> ids, String tenantCode) {
    return this.lambdaQuery()
        .eq(ChannelOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ChannelOrg::getTenantCode, tenantCode)
        .in(ChannelOrg::getId, ids).list();
  }

  /**
   * 通过RuleCode集合查询渠道组织数据
   *
   * @param ruleCodes
   * @param tenantCode
   * @return
   */
  public List<ChannelOrg> findListByRuleCodes(Set<String> ruleCodes, String tenantCode) {
    return this.lambdaQuery()
        .eq(ChannelOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ChannelOrg::getTenantCode, tenantCode)
        .in(ChannelOrg::getRuleCode, ruleCodes).list();
  }

  /**
   * 通过启用状态和RuleCodes模糊查询自身及子集
   *
   * @param ruleCodes
   * @param enableStatus
   * @param tenantCode
   * @return
   */
  public List<ChannelOrg> findCurAndChildrenByRuleCodeList(List<String> ruleCodes, String enableStatus, String tenantCode) {
    return this.baseMapper.findCurAndChildrenByRuleCodeList(ruleCodes, enableStatus, tenantCode, DelFlagStatusEnum.NORMAL);
  }

  /**
   * 根据父级Codes查询所有子集
   *
   * @param parentCodes
   * @param tenantCode
   * @return
   */
  public List<ChannelOrg> findChildrenListByParentCodes(List<String> parentCodes, String tenantCode) {
    return this.lambdaQuery()
        .eq(ChannelOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ChannelOrg::getTenantCode, tenantCode)
        .in(ChannelOrg::getParentCode, parentCodes).list();
  }

  /**
   * 查找parentCode不为空但找不到对应上级的数据,设置parentCode为null
   *
   * @return
   */
  public void updateOrphanParentCodeNull(String tenantCode) {
    this.baseMapper.updateOrphanParentCodeNull(tenantCode, DelFlagStatusEnum.NORMAL);
  }

  /**
   * 查询所有没有父级编码的子集
   *
   * @param tenantCode
   * @return
   */
  public List<ChannelOrg> findListWithoutParentCode(String tenantCode) {
    return this.lambdaQuery()
        .eq(ChannelOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ChannelOrg::getTenantCode, tenantCode)
        .isNull(ChannelOrg::getParentCode)
        .list();
  }

  /**
   * 查询所有渠道组织
   *
   * @param tenantCode
   * @return
   */
  public List<ChannelOrg> findAll(String tenantCode) {
    return this.lambdaQuery()
        .eq(ChannelOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ChannelOrg::getTenantCode, tenantCode)
        .list();
  }

  /**
   * 分页条件查询
   *
   * @param pageable
   * @param channelOrgPaginationDto
   * @return
   */
  public Page<ChannelOrgVo> findByConditions(Pageable pageable, ChannelOrgPaginationDto channelOrgPaginationDto) {
    Page<ChannelOrgVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.baseMapper.findByConditions(page, channelOrgPaginationDto, DelFlagStatusEnum.NORMAL);
  }

  /**
   * 查询渠道组织懒加载数据
   *
   * @param enableStatus    启用状态
   * @param topOnly         传true只查第一层
   * @param parentCode      只查询该编码下一级
   * @param codeList        只查询这些编码
   * @param ruleCodeList    只查询这些降维编码
   * @param excludeRuleCode 排除这个降维编码的下级
   * @param tenantCode
   * @return
   */
  public List<LazyTreeVo> findLazyTreeList(String enableStatus,
      Boolean topOnly,
      String parentCode,
      List<String> codeList,
      List<String> ruleCodeList,
      String excludeRuleCode,
      String tenantCode) {
    return this.baseMapper.findLazyTreeList(enableStatus, topOnly, parentCode, codeList, ruleCodeList, excludeRuleCode, tenantCode, DelFlagStatusEnum.NORMAL);
  }

  /**
   * 通过名称模糊查询渠道组织list
   *
   * @param enableStatus
   * @param name
   * @param tenantCode
   * @param channelOrgLevel
   * @return
   */
  public List<ChannelOrg> findListLikeNameAndChannelOrgLevel(String enableStatus, String name, String tenantCode, String channelOrgLevel) {
    return this.lambdaQuery()
        .eq(ChannelOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ChannelOrg::getTenantCode, tenantCode)
        .eq(StringUtils.isNotEmpty(enableStatus), ChannelOrg::getEnableStatus, enableStatus)
        .like(StringUtils.isNotEmpty(name), ChannelOrg::getChannelOrgName, name)
        .eq(ChannelOrg::getChannelOrgLevel, channelOrgLevel)
        .list();
  }

  /**
   * 通过名称模糊查询渠道组织list
   *
   * @param enableStatus
   * @param name
   * @param tenantCode
   * @return
   */
  public List<ChannelOrg> findListLikeName(String enableStatus, String name, String tenantCode) {
    return this.lambdaQuery()
        .eq(ChannelOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ChannelOrg::getTenantCode, tenantCode)
        .eq(StringUtils.isNotEmpty(enableStatus), ChannelOrg::getEnableStatus, enableStatus)
        .like(StringUtils.isNotEmpty(name), ChannelOrg::getChannelOrgName, name)
        .list();
  }

  /**
   * 通过code集合查询渠道组织数据
   *
   * @param codes
   * @param tenantCode
   * @return
   */
  public List<ChannelOrg> findListByCodes(List<String> codes, String tenantCode) {
    return this.lambdaQuery()
        .eq(ChannelOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ChannelOrg::getTenantCode, tenantCode)
        .in(ChannelOrg::getChannelOrgCode, codes).list();
  }

  /**
   * 通过组织机构集合查询当前组织及其以下组织的信息
   *
   * @param orgCodes
   * @param enableStatus
   * @param tenantCode
   * @return
   */
  public List<ChannelOrg> findCurAndChildrenByOrgCodeList(List<String> orgCodes, String enableStatus, String tenantCode) {
    return this.baseMapper.findCurAndChildrenByOrgCodeList(orgCodes, enableStatus, tenantCode);
  }

  /**
   * 手动设置父级编码为空
   *
   * @param id
   */
  public void setParentCodeNull(String id) {
    this.lambdaUpdate().set(ChannelOrg::getParentCode, null)
        .eq(ChannelOrg::getId, id)
        .eq(ChannelOrg::getTenantCode, TenantUtils.getTenantCode())    //新增租户编号判断条件
        .update();
  }

  /**
   * 手动设置父级编码，规则编码为空
   *
   * @param id
   */
  public void setParentCodeAndRuleCodeNull(String id) {
    this.lambdaUpdate()
        .set(ChannelOrg::getParentCode, null)
        .set(ChannelOrg::getRuleCode, "")
        .eq(ChannelOrg::getId, id)
        .eq(ChannelOrg::getTenantCode, TenantUtils.getTenantCode())    //新增租户编号判断条件
        .update();
  }

  public ChannelOrg findByCustomerOrgCode(String code) {
    return this.lambdaQuery()
        .eq(ChannelOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ChannelOrg::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .eq(ChannelOrg::getChannelOrgCode, code)
        .eq(ChannelOrg::getTenantCode, TenantUtils.getTenantCode())    //新增租户编号判断条件
        .one();
  }

  public List<ChannelOrg> findByCustomerOrgCodes(List<String> code) {
    return this.lambdaQuery()
        .eq(ChannelOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(ChannelOrg::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .eq(ChannelOrg::getTenantCode, TenantUtils.getTenantCode())    //新增租户编号判断条件
        .in(ChannelOrg::getChannelOrgCode, code)
        .list();
  }

  /**
   * 批量删除
   *
   * @param ids ID集合
   */
  public void updateDelFlagByIds(List<String> ids) {
    this.lambdaUpdate()
        .in(ChannelOrg::getId, ids)
        .eq(ChannelOrg::getTenantCode, TenantUtils.getTenantCode())    //新增租户编号判断条件
        .set(ChannelOrg::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
        .update();
  }

  /**
   * 批量更新启用禁用状态
   *
   * @param ids          ID集合
   * @param enableStatus 启用禁用状态
   */
  public void updateEnableStatusByIds(List<String> ids, EnableStatusEnum enableStatus) {
    this.lambdaUpdate()
        .in(ChannelOrg::getId, ids)
        .eq(ChannelOrg::getTenantCode, TenantUtils.getTenantCode())    //新增租户编号判断条件
        .set(ChannelOrg::getEnableStatus, enableStatus.getCode())
        .update();
  }

  /**
   * 根据渠道组织查询规则编码
   */
  public String findRuleCodeByChannelOrgCode(String channelOrgCode, String tenantCode) {
    ChannelOrg org =
        this.lambdaQuery()
            .eq(ChannelOrg::getChannelOrgCode, channelOrgCode)
            .eq(ChannelOrg::getTenantCode, tenantCode)
            .eq(ChannelOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .select(ChannelOrg::getRuleCode)
            .one();
    if(Objects.isNull(org)) {
      return null;
    }
    return org.getRuleCode();
  }

  /**
   * 渠道编码查询
   * @param channelOrgCode
   * @return
   */
  public ChannelOrg findByChannelOrgCode(String channelOrgCode) {
    return this.lambdaQuery()
            .eq(ChannelOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .eq(ChannelOrg::getChannelOrgCode, channelOrgCode)
            .one();
  }

  /**
   * 通过RuleCode集合查询渠道组织数据
   *
   * @param ruleCodes
   * @param tenantCode
   * @return
   */
  public List<ChannelOrg> findListByRuleCodes(Set<String> ruleCodes) {
    return this.lambdaQuery()
            .eq(ChannelOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .in(ChannelOrg::getRuleCode, ruleCodes)
            .list();
  }

  /**
   * 依据规则编码查询下级 组织编码
   * @param: * @param null
   * @return: * @return: null
   * @author: longhuajiang
   * @date: 2023/1/13
   */
  public List<ChannelOrg> findChildrenOrgByRuleCode(String ruleCode) {
    return this.lambdaQuery()
        .eq(ChannelOrg::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .likeRight(ChannelOrg::getRuleCode, ruleCode)
        .list();
  }
}
