package com.biz.crm.mdm.business.channel.org.local.service.helper;
/**
 * Created by Bao Hongbin on 2021-10-28 14:11.
 */

import com.biz.crm.mdm.business.channel.org.local.repository.ChannelOrgRepository;
import com.biz.crm.mdm.business.channel.org.local.entity.ChannelOrg;
import com.biz.crm.mdm.business.channel.org.sdk.dto.ChannelOrgCreateDto;
import com.biz.crm.mdm.business.channel.org.sdk.dto.ChannelOrgUpdateDto;
import com.biz.crm.mdm.business.channel.org.sdk.event.ChannelOrgEventListener;
import com.biz.crm.mdm.business.channel.org.sdk.vo.ChannelOrgVo;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @program: crm
 * @description: 渠道组织服务助手
 * @author: Bao Hongbin
 * @create: 2021-10-28 14:11
 **/
@Component
@Slf4j
public class ChannelOrgServiceHelper {
  @Autowired(required = false)
  private ChannelOrgRepository channelOrgRepository;
  @Autowired(required = false)
  @Lazy
  private List<ChannelOrgEventListener> channelOrgEventListeners;

  /**
   * 验证创建请求参数
   *
   * @param channelOrgCreateDto
   */
  public void createValidation(ChannelOrgCreateDto channelOrgCreateDto) {
    Validate.notNull(channelOrgCreateDto, "进行当前操作时，信息对象必须传入!");
    log.info("新增渠道组织请求dto：{}", JsonUtils.obj2JsonString(channelOrgCreateDto));
    Validate.notBlank(channelOrgCreateDto.getChannelOrgName(), "渠道组织名称不能为空");
    List<ChannelOrg> channelOrgs =
        channelOrgRepository.findListByName(channelOrgCreateDto.getChannelOrgName(), false, TenantUtils.getTenantCode());
    Validate.isTrue(CollectionUtils.isEmpty(channelOrgs), "渠道组织名称已存在");
    if (StringUtils.isNotEmpty(channelOrgCreateDto.getChannelOrgCode())) {
      Validate.isTrue(channelOrgCreateDto.getChannelOrgCode().length() < 64,
          "渠道组织编码信息，在进行新增时填入值超过了限定长度(64)，请检查!");
      ChannelOrg channelOrg =
          channelOrgRepository.findDetailsByCode(channelOrgCreateDto.getChannelOrgCode(), TenantUtils.getTenantCode());
      Validate.isTrue(null == channelOrg,
          "编码[" + channelOrgCreateDto.getChannelOrgCode() + "]已存在");
    }
    if (StringUtils.isNotEmpty(channelOrgCreateDto.getParentCode())) {
      ChannelOrg parent =
          channelOrgRepository.findDetailsByCode(channelOrgCreateDto.getParentCode(), TenantUtils.getTenantCode());
      Validate.notNull(parent, "上级编码无效");
    }
    Validate.notBlank(channelOrgCreateDto.getChannelOrgType(), "新增信息时，渠道组织类型不能为空！");
    Validate.notNull(channelOrgCreateDto.getEnableStatus(), "新增信息时，渠道组织启用状态不能为空！");
    Validate.isTrue(channelOrgCreateDto.getChannelOrgName().length() < 128,
        "渠道组织名称信息，在进行新增时填入值超过了限定长度(128)，请检查!");
  }

  /**
   * 验证更新请求参数
   *
   * @param channelOrgUpdateDto
   */
  public void updateValidation(ChannelOrgUpdateDto channelOrgUpdateDto) {
    Validate.notNull(channelOrgUpdateDto, "进行当前操作时，信息对象必须传入!");
    log.info("编辑渠道组织请求dto：{}", JsonUtils.obj2JsonString(channelOrgUpdateDto));
    Validate.notBlank(channelOrgUpdateDto.getId(), "修改信息时，渠道组织ID不能为空！");
    ChannelOrg channelOrg =
        channelOrgRepository.findDetailsById(channelOrgUpdateDto.getId(), TenantUtils.getTenantCode());
    Validate.notNull(channelOrg, "无效的业务技术编号信息");
    ChannelOrg detailsByCode =
        channelOrgRepository.findDetailsByCode(channelOrgUpdateDto.getChannelOrgCode(), TenantUtils.getTenantCode());
    Validate.isTrue(detailsByCode == null ||
        channelOrgUpdateDto.getId().equals(detailsByCode.getId()), "渠道组织编码已存在");
    List<String> idsByName =
        Optional.ofNullable(
            channelOrgRepository.findListByName(
                channelOrgUpdateDto.getChannelOrgName(), false, TenantUtils.getTenantCode()))
            .orElse(new ArrayList<>())
            .stream().map(ChannelOrg::getId).collect(Collectors.toList());

    Validate.isTrue(CollectionUtils.isEmpty(idsByName) ||
        (idsByName.size() == 1 && idsByName.contains(channelOrgUpdateDto.getId())), "渠道组织名称已存在");
    Validate.notBlank(channelOrg.getRuleCode(), "当前渠道组织ruleCode不能为空");
    Validate.notNull(channelOrgUpdateDto.getChannelOrgCode(), "修改信息时，渠道组织编码不能为空！");
    Validate.isTrue(
        channelOrg.getChannelOrgCode().equals(channelOrgUpdateDto.getChannelOrgCode()),
        "渠道组织编码不能修改");
    if (StringUtils.isNotEmpty(channelOrgUpdateDto.getParentCode())) {
      ChannelOrg parent =
          channelOrgRepository.findDetailsByCode(channelOrgUpdateDto.getParentCode(), TenantUtils.getTenantCode());
      Validate.notNull(parent, "上级渠道组织不存在");
      Validate.isTrue(!parent.getId().equals(channelOrg.getId()), "上级渠道组织不能是自己");
      Validate.isTrue(!parent.getRuleCode().startsWith(channelOrg.getRuleCode()),
          "上级渠道组织不能是当前层级的下级");
      if (!parent.getChannelOrgCode().equals(channelOrg.getParentCode())) {
        Validate.notBlank(parent.getRuleCode(), "上级渠道组织ruleCode不能为空");
      }
    }
    Validate.notBlank(channelOrgUpdateDto.getChannelOrgName(), "修改信息时，渠道组织名称不能为空！");
    Validate.notNull(channelOrgUpdateDto.getChannelOrgType(), "修改信息时，渠道组织类型不能为空！");
    Validate.isTrue(channelOrgUpdateDto.getChannelOrgName().length() < 128,
        "渠道组织名称信息，在进行修改时填入值超过了限定长度(128)，请检查!");
  }

  /**
   * 推送创建事件
   *
   * @param channelOrgVos
   */
  public void sendCreateEvent(List<ChannelOrgVo> channelOrgVos) {
    if (Objects.nonNull(this.channelOrgEventListeners)) {
      for (ChannelOrgEventListener listener : this.channelOrgEventListeners) {
        listener.onBatchCreate(channelOrgVos);
      }
    }
  }

  /**
   * 推送更新事件
   *
   * @param channelOrgVos
   */
  public void sendUpdateEvent(List<ChannelOrgVo> channelOrgVos) {
    if (Objects.nonNull(this.channelOrgEventListeners)) {
      for (ChannelOrgEventListener listener : this.channelOrgEventListeners) {
        listener.onBatchUpdate(channelOrgVos);
      }
    }
  }

  /**
   * 推送启用事件
   *
   * @param channelOrgVos
   */
  public void sendEnableEvent(List<ChannelOrgVo> channelOrgVos) {
    if (Objects.nonNull(this.channelOrgEventListeners)) {
      for (ChannelOrgEventListener listener : this.channelOrgEventListeners) {
        listener.onBatchEnable(channelOrgVos);
      }
    }
  }


  /**
   * 推送禁用事件
   *
   * @param channelOrgVos
   */
  public void sendDisableEvent(List<ChannelOrgVo> channelOrgVos) {
    if (Objects.nonNull(this.channelOrgEventListeners)) {
      for (ChannelOrgEventListener listener : this.channelOrgEventListeners) {
        listener.onBatchDisable(channelOrgVos);
      }
    }
  }


  /**
   * 推送删除事件
   *
   * @param channelOrgVos
   */
  public void sendDeleteEvent(List<ChannelOrgVo> channelOrgVos) {
    if (Objects.nonNull(this.channelOrgEventListeners)) {
      for (ChannelOrgEventListener listener : this.channelOrgEventListeners) {
        listener.onBatchDelete(channelOrgVos);
      }
    }
  }
}
