package com.biz.crm.mdm.business.cost.center.local.service.internal;

import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.cost.center.local.entity.CostCenter;
import com.biz.crm.mdm.business.cost.center.local.repository.CostCenterRepository;
import com.biz.crm.mdm.business.cost.center.sdk.constant.CostCenterConstant;
import com.biz.crm.mdm.business.cost.center.sdk.dto.CostCenterDto;
import com.biz.crm.mdm.business.cost.center.sdk.dto.CostCenterLogEventDto;
import com.biz.crm.mdm.business.cost.center.sdk.dto.CostCenterProfitsSelectDto;
import com.biz.crm.mdm.business.cost.center.sdk.dto.CostCenterSelectDto;
import com.biz.crm.mdm.business.cost.center.sdk.event.log.CostCenterEventListener;
import com.biz.crm.mdm.business.cost.center.sdk.service.CostCenterService;
import com.biz.crm.mdm.business.cost.center.sdk.vo.CostCenterVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.common.base.vo.CommonSelectVo;
import com.biz.crm.mn.third.system.sap.fi.sdk.dto.SapFiCostCenterDto;
import com.biz.crm.mn.third.system.sap.fi.sdk.service.SapFiService;
import com.biz.crm.mn.third.system.sap.fi.sdk.vo.SapFiCostCenterVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author: chenlong
 * @date: 2022/12/3 11:37
 * @description: 成本中心(CostCenter)表服务实现类
 */
@Service("CostCenterService")
@Slf4j
public class CostCenterServiceImpl implements CostCenterService {

    @Autowired(required = false)
    private SapFiService sapFiService;

    @Autowired(required = false)
    private CostCenterRepository costCenterRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private DictToolkitService dictToolkitService;

    /**
     * 分页查询所有数据
     *
     * @param pageable      分页对象
     * @param costCenterDto 查询dto
     * @return 所有数据
     */
    @Override
    public Page<CostCenterVo> findByConditions(Pageable pageable, CostCenterDto costCenterDto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
        if (Objects.isNull(costCenterDto)) {
            costCenterDto = new CostCenterDto();
        }
        costCenterDto.setTenantCode(TenantUtils.getTenantCode());
        //利润中心名称默认的default，需要通过字典转换一下。
        Page<CostCenterVo> page = this.costCenterRepository.findByConditions(pageable, costCenterDto);
        if (!CollectionUtils.isEmpty(page.getRecords())) {
            Map<String, String> dictMap = dictToolkitService.findMapByDictTypeCode(CostCenterConstant.DICT_TPM_PROFIT_CENTER);
            for (CostCenterVo vo : page.getRecords()) {
                if (StringUtils.isNotBlank(vo.getProfitCenterCode()) && dictMap.containsKey(vo.getProfitCenterCode())) {
                    vo.setProfitCenterName(dictMap.get(vo.getProfitCenterCode()));
                }
            }
        }
        return page;
    }

    /**
     * 通过主键查询单条数据
     *
     * @param id 主键
     * @return 单条数据
     */
    @Override
    public CostCenterVo findById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        CostCenter costCenter = this.costCenterRepository.getById(id);
        if (null == costCenter) {
            return null;
        }
        return this.nebulaToolkitService.copyObjectByWhiteList(costCenter, CostCenterVo.class, null, null);
    }

    /**
     * 删除数据
     *
     * @param ids 主键结合
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids), "删除数据时，主键集合不能为空！");
        List<CostCenter> entityList = this.costCenterRepository.listByIds(ids);
        ArrayList<CostCenter> list = new ArrayList<>();
        entityList.forEach(item -> {
            CostCenter entity = new CostCenter();
            entity.setId(item.getId());
            entity.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
            entity.setCostCenterCode(entity.getCostCenterCode() + entity.getId());
            list.add(entity);

            // 删除业务日志
            CostCenterLogEventDto logEventDto = new CostCenterLogEventDto();
            CostCenterVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(item, CostCenterVo.class, null, null);
            logEventDto.setOriginal(oldVo);
            CostCenterDto newDto =
                    this.nebulaToolkitService.copyObjectByWhiteList(item, CostCenterDto.class, null, null);
            newDto.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
            newDto.setCostCenterCode(entity.getCostCenterCode() + entity.getId());
            logEventDto.setNewest(newDto);
            SerializableBiConsumer<CostCenterEventListener, CostCenterLogEventDto> onDelete =
                    CostCenterEventListener::onDelete;
            this.nebulaNetEventClient.publish(logEventDto, CostCenterEventListener.class, onDelete);
        });
        this.costCenterRepository.updateBatchById(list);
    }

    /**
     * 启用
     *
     * @param ids 主键列表
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void enableBatch(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids), "请选择要操作的数据");
        List<CostCenter> entityList = this.costCenterRepository.listByIds(ids);
        Validate.isTrue(!CollectionUtils.isEmpty(entityList), "不存在或已删除!");

        ArrayList<CostCenter> list = new ArrayList<>();
        entityList.forEach(item -> {
            CostCenter entity = new CostCenter();
            entity.setId(item.getId());
            entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            list.add(entity);

            // 启用业务日志
            CostCenterLogEventDto logEventDto = new CostCenterLogEventDto();
            CostCenterVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(item, CostCenterVo.class, null, null);
            logEventDto.setOriginal(oldVo);
            CostCenterDto newDto =
                    this.nebulaToolkitService.copyObjectByWhiteList(item, CostCenterDto.class, null, null);
            newDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            logEventDto.setNewest(newDto);
            SerializableBiConsumer<CostCenterEventListener, CostCenterLogEventDto> onEnable =
                    CostCenterEventListener::onEnable;
            this.nebulaNetEventClient.publish(logEventDto, CostCenterEventListener.class, onEnable);
        });
        this.costCenterRepository.updateBatchById(list);
    }

    /**
     * 禁用
     *
     * @param ids 主键列表
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void disableBatch(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids), "请选择要操作的数据");
        List<CostCenter> entityList = this.costCenterRepository.listByIds(ids);
        Validate.isTrue(!CollectionUtils.isEmpty(entityList), "不存在或已删除!");

        ArrayList<CostCenter> list = new ArrayList<>();
        entityList.forEach(item -> {
            CostCenter entity = new CostCenter();
            entity.setId(item.getId());
            entity.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
            list.add(entity);

            // 启用业务日志
            CostCenterLogEventDto logEventDto = new CostCenterLogEventDto();
            CostCenterVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(item, CostCenterVo.class, null, null);
            logEventDto.setOriginal(oldVo);
            CostCenterDto newDto =
                    this.nebulaToolkitService.copyObjectByWhiteList(item, CostCenterDto.class, null, null);
            newDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            logEventDto.setNewest(newDto);
            SerializableBiConsumer<CostCenterEventListener, CostCenterLogEventDto> onEnable =
                    CostCenterEventListener::onEnable;
            this.nebulaNetEventClient.publish(logEventDto, CostCenterEventListener.class, onEnable);
        });
        this.costCenterRepository.updateBatchById(list);
    }

    /**
     * 成本中心下拉
     *
     * @param dto 成本中心查询参数
     * @return 下拉数据
     */
    @Override
    public List<CommonSelectVo> findFormSelectList(CostCenterSelectDto dto) {
        if (Objects.isNull(dto)) {
            dto = new CostCenterSelectDto();
        }
        dto.setTenantCode(TenantUtils.getTenantCode());
        return costCenterRepository.findFormSelectList(dto);
    }

    /**
     * 利润中心下拉
     *
     * @param dto 利润中心查询参数
     * @return 下拉数据
     */
    @Override
    public List<CommonSelectVo> findProfitsFormSelectList(CostCenterProfitsSelectDto dto) {
        if (Objects.isNull(dto)) {
            dto = new CostCenterProfitsSelectDto();
        }
        dto.setTenantCode(TenantUtils.getTenantCode());
        List<CommonSelectVo> list = costCenterRepository.findProfitsFormSelectList(dto);
        if (!CollectionUtils.isEmpty(list)) {
            Map<String, String> map = dictToolkitService.findMapByDictTypeCode(CostCenterConstant.DICT_TPM_PROFIT_CENTER);
            list.forEach(v ->
                    v.setValue(map.getOrDefault(v.getCode(), v.getValue()))
            );
        }
        return list;
    }

    /**
     * 批量拉取成本中心科目
     *
     * @author huojia
     * @date 2022/12/7 22:03
     **/
    @Override
    public void pullCostCenterList(SapFiCostCenterDto dto) {
        List<SapFiCostCenterVo> sapFiCostCenterVos = sapFiService.pullCostCenterList(dto);
        if (CollectionUtils.isEmpty(sapFiCostCenterVos)) {
            return;
        }
        //数据校验 成本中心编码+公司代码+利润中心编码
        Map<String, CostCenter> costCenterMap = sapCostCenterVoToMap(sapFiCostCenterVos);
        if (CollectionUtils.isEmpty(costCenterMap)) {
            return;
        }
        List<String> costCenterCodeList = costCenterMap.values().stream()
                .map(CostCenter::getCostCenterCode).collect(Collectors.toList());
        List<CostCenter> byCodes = costCenterRepository.findByCostCenterCodes(costCenterCodeList);
        if (CollectionUtils.isEmpty(byCodes)) {
            this.saveOrUpdateBatch(new ArrayList<>(costCenterMap.values()), null);
            return;
        }
        // 区分更新、新增   成本中心编码+公司代码+利润中心编码
        List<CostCenter> saveList = new ArrayList<>();
        List<CostCenter> updateList = new ArrayList<>();
        Map<String, CostCenter> oldMap = byCodes.stream().collect(Collectors.toMap(
                v -> v.getCostCenterCode() + StrUtil.nullToEmpty(v.getCompanyCode()) + StrUtil.nullToEmpty(v.getProfitCenterCode()),
                v -> v, (oldValue, newValue) -> newValue));
        costCenterMap.forEach((key, entity) -> {
            if (oldMap.containsKey(key)) {
                CostCenter oldEntity = oldMap.get(key);
                entity.setId(oldEntity.getId());
                entity.setCreateTime(oldEntity.getCreateTime());
                entity.setCreateName(oldEntity.getCreateName());
                entity.setCreateAccount(oldEntity.getCreateAccount());
                updateList.add(entity);
            } else {
                saveList.add(entity);
            }
        });
        this.saveOrUpdateBatch(saveList, updateList);
    }


    /**
     * 通过编码查询单条数据
     *
     * @param costCenterCode 成本中心编码
     * @return 单条数据
     */
    @Override
    public CostCenterVo findByCode(String costCenterCode) {
        if (StringUtils.isBlank(costCenterCode)) {
            return null;
        }
        CostCenter costCenter = this.costCenterRepository.lambdaQuery()
                .eq(CostCenter::getCostCenterCode, costCenterCode)
                .eq(CostCenter::getTenantCode, TenantUtils.getTenantCode())
                .eq(CostCenter::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .one();
        if (null == costCenter) {
            return null;
        }
        return this.nebulaToolkitService.copyObjectByWhiteList(costCenter, CostCenterVo.class, null, null);
    }

    /**
     * 批量新增数据（MDG用）
     *
     * @param saveList
     * @param updateList
     * @author dutaotao
     * @date 2023/1/3 17:38
     **/
    private void saveOrUpdateBatch(List<CostCenter> saveList, List<CostCenter> updateList) {
        if (!org.springframework.util.CollectionUtils.isEmpty(saveList)) {
            this.costCenterRepository.saveBatch(saveList);
        }
        if (!org.springframework.util.CollectionUtils.isEmpty(updateList)) {
            this.costCenterRepository.updateBatchById(updateList);
        }
    }

    /**
     * SAP成本中心数据构建map
     *
     * @param sapFiCostCenterVos
     * @author huojia
     * @date 2022/12/6 21:37
     **/
    private Map<String, CostCenter> sapCostCenterVoToMap(List<SapFiCostCenterVo> sapFiCostCenterVos) {
        if (CollectionUtils.isEmpty(sapFiCostCenterVos)) {
            return Collections.emptyMap();
        }
        // 去重 成本中心编码+公司代码+利润中心编码
        Map<String, CostCenter> resultMap = new HashMap<>(sapFiCostCenterVos.size());
        sapFiCostCenterVos.forEach(sapVo -> {
            // 成本中心编码
            sapVo.setKOSTL(StrUtil.nullToEmpty(sapVo.getKOSTL()));
            if (StringUtils.isEmpty(sapVo.getKOSTL())) {
                log.error("SAP数据成本中心编码为空[{}]", sapFiCostCenterVos);
                return;
            }
            // 公司代码
            sapVo.setBUKRS(StrUtil.nullToEmpty(sapVo.getBUKRS()));
            // 利润中心编码
            sapVo.setPRCTR(StrUtil.nullToEmpty(sapVo.getPRCTR()));
            //成本中心编码+公司代码+利润中心编码
            String key = sapVo.getKOSTL() + sapVo.getBUKRS() + sapVo.getPRCTR();
            CostCenter newEntity = this.getCostCenterForSapVo(sapVo);
            CostCenter entity = this.compareNewAndOldEntity(newEntity, resultMap.get(key));
            resultMap.put(key, entity);
        });
        return resultMap;
    }

    /**
     * 构建成本中心
     *
     * @return
     */
    private CostCenter getCostCenterForSapVo(SapFiCostCenterVo sapVo) {
        CostCenter entity = new CostCenter();
        entity.setTenantCode(TenantUtils.getTenantCode());
        entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        // 成本中心编码
        entity.setCostCenterCode(sapVo.getKOSTL());
        // 成本中心名称
        entity.setCostCenterName(sapVo.getLTEXT());
        // 公司代码
        entity.setCompanyCode(sapVo.getBUKRS());
        // 利润中心编码
        entity.setProfitCenterCode(sapVo.getPRCTR());
        //起禁用状态  实际初级成本的冻结标志 不为空时冻结
        EnableStatusEnum enableStatusEnum = EnableStatusEnum.ENABLE;
        if (StringUtils.isNotBlank(sapVo.getBKZKP())) {
            enableStatusEnum = EnableStatusEnum.DISABLE;
        }
        // 利润中心名称 默认
        entity.setProfitCenterName(CostCenterConstant.ProfitCenterName);
        //有效生效时间
        if (StringUtils.isNotBlank(sapVo.getDATAB())) {
            try {
                entity.setBeginTime(DateUtil.date_yyyy_MM_dd.parse(sapVo.getDATAB()));
            } catch (Exception e) {
                log.error("", e);
            }
        }
        //有效截止时间
        if (StringUtils.isNotBlank(sapVo.getDATBI())) {
            try {
                entity.setEndTime(DateUtil.date_yyyy_MM_dd.parse(sapVo.getDATBI()));
                //有效截止时间小于当前时间  禁用
                if (Objects.nonNull(entity.getEndTime())) {
                    Date nowDate = new Date();
                    if (entity.getEndTime().compareTo(nowDate) < 0) {
                        enableStatusEnum = EnableStatusEnum.DISABLE;
                    }
                }
            } catch (Exception e) {
                log.error("", e);
            }
        }
        entity.setEnableStatus(enableStatusEnum.getCode());
        return entity;

    }

    /**
     * 对比新和旧的成本中心
     *
     * @param newEntity 新
     * @param oldEntity 旧
     * @return
     */
    private CostCenter compareNewAndOldEntity(CostCenter newEntity, CostCenter oldEntity) {
        if (Objects.isNull(oldEntity)) {
            return newEntity;
        }
        //启用的优先
        if (EnableStatusEnum.ENABLE.getCode().equals(newEntity.getEnableStatus())) {
            return newEntity;
        }
        if (EnableStatusEnum.ENABLE.getCode().equals(oldEntity.getEnableStatus())) {
            return oldEntity;
        }
        //有效结束时间晚的的优先
        if (Objects.nonNull(newEntity.getEndTime())
                && Objects.nonNull(oldEntity.getEndTime())) {
            if (newEntity.getEndTime().compareTo(oldEntity.getEndTime()) > 0) {
                return newEntity;
            } else {
                return oldEntity;
            }
        }
        //有效开始时间晚的的优先
        if (Objects.nonNull(newEntity.getBeginTime())
                && Objects.nonNull(oldEntity.getBeginTime())) {
            if (newEntity.getBeginTime().compareTo(oldEntity.getBeginTime()) > 0) {
                return newEntity;
            } else {
                return oldEntity;
            }
        }
        return newEntity;
    }
}
