package com.biz.crm.mdm.business.customer.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mdm.business.customer.local.entity.CustomerEntity;
import com.biz.crm.mdm.business.customer.local.service.CustomerDockingService;
import com.biz.crm.mdm.business.customer.local.service.CustomerService;
import com.biz.crm.mdm.business.customer.sdk.dto.BindCustomerDockingDto;
import com.biz.crm.mdm.business.customer.sdk.dto.CustomerDto;
import com.biz.crm.mdm.business.customer.sdk.dto.CustomerRebindOrgDto;
import com.biz.crm.mdm.business.customer.sdk.dto.UnbindCustomerDockingDto;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

/**
 * 客户信息: Customer: 客户管理
 *
 * @author ning.zhang
 * @date 2021/10/15
 */
@Slf4j
@RestController
@RequestMapping("/v1/customer/customer")
@Api(tags = "客户信息: Customer: 客户管理")
public class CustomerController {

  @Autowired(required = false)
  private CustomerService customerService;
  @Autowired(required = false)
  private CustomerDockingService customerDockingService;

  @ApiOperation(value = "新增客户")
  @PostMapping("/create")
  public Result create(@RequestBody CustomerDto dto) {
    try {
      return Result.ok(this.customerService.create(dto));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation(value = "更新客户")
  @PatchMapping("/update")
  public Result update(@RequestBody CustomerDto dto) {
    try {
      return Result.ok(this.customerService.update(dto));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 更新客户，网关没有配置patch转post
   *
   * @param dto
   * @return
   */
  @ApiOperation(value = "更新客户")
  @PostMapping("/updateTurnPatch")
  public Result updateTurnPatch(@RequestBody CustomerDto dto) {
    return Result.ok(this.update(dto));
  }

  @ApiOperation(value = "删除")
  @DeleteMapping("/deleteBatch")
  public Result deleteBatch(@RequestParam("ids") List<String> ids) {
    try {
      this.customerService.deleteBatch(ids);
      return Result.ok("删除成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation(value = "启用")
  @PatchMapping("/enableBatch")
  public Result enableBatch(@RequestBody List<String> ids) {
    try {
      this.customerService.enableBatch(ids);
      return Result.ok("启用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation(value = "禁用")
  @PatchMapping("/disableBatch")
  public Result disableBatch(@RequestBody List<String> ids) {
    try {
      this.customerService.disableBatch(ids);
      return Result.ok("禁用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation(value = "冻结")
  @PatchMapping("/freezeBatch")
  public Result freezeBatch(@RequestBody List<String> ids) {
    try {
      this.customerService.freezeBatch(ids);
      return Result.ok("冻结成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation(value = "解冻")
  @PatchMapping("/unfreezeBatch")
  public Result unfreezeBatch(@RequestBody List<String> ids) {
    try {
      this.customerService.unfreezeBatch(ids);
      return Result.ok("解冻成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation(value = "客户更换企业组织")
  @PatchMapping("/rebindOrg")
  public Result rebindOrg(@RequestBody CustomerRebindOrgDto dto) {
    try {
      this.customerService.rebindOrg(dto);
      return Result.ok("更换成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation(value = "客户更换客户组织")
  @PatchMapping("/rebindCustomerOrg")
  public Result rebindCustomerOrg(@RequestBody CustomerRebindOrgDto dto) {
    try {
      this.customerService.rebindCustomerOrg(dto);
      return Result.ok("更换成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation(value = "绑定对接人")
  @PatchMapping("/bindDocking")
  public Result bindDocking(@RequestBody BindCustomerDockingDto dto) {
    try {
      this.customerDockingService.bind(dto);
      return Result.ok("绑定成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation(value = "解绑对接人")
  @PatchMapping("/unbindDocking")
  public Result unbindDocking(@RequestBody UnbindCustomerDockingDto dto) {
    try {
      this.customerDockingService.unbind(dto);
      return Result.ok("解绑成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 上级客户信息下拉框分页列表
   *
   * @param pageable
   * @return
   */
  @ApiOperation(value = "上级客户信息下拉框分页列表")
  @GetMapping("/findByParentCustomerIsNull")
  public Result<Page<CustomerEntity>> findByParentCustomerIsNull(@PageableDefault(50) Pageable pageable) {
    try {
      return Result.ok(customerService.findByParentCustomerIsNull(pageable));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 下级客户信息分页列表
   *
   * @param pageable
   * @return
   */
  @ApiOperation(value = "下级客户信息分页列表")
  @GetMapping("/findChildrenByCustomerCode")
  public Result<Page<CustomerEntity>> findChildrenByCustomerCode(@PageableDefault(50) Pageable pageable,
                                                                 @ApiParam(name = "customerCode", value = "客户编码") @RequestParam(value = "customerCode") String customerCode) {
    try {
      return Result.ok(customerService.findChildrenByCustomerCode(pageable, customerCode));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }


  /**
   * 根据标签id查询
   *
   * @param pageable
   * @return
   */
  @ApiOperation(value = "根据标签id查询")
  @GetMapping("/findByTagId")
  public Result<Page<CustomerEntity>> findByTagId(@PageableDefault(50) Pageable pageable,
                                                  @ApiParam(name = "tagId", value = "标签id") @RequestParam(value = "tagId") String tagId) {
    try {
      return Result.ok(customerService.findByTagId(pageable, tagId));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
  * 根据客户编码或客户名称查询
  *
  * @param
  * @return
  */
  @ApiOperation(value = "根据客户编码或客户名称查询")
  @GetMapping("/findByCustomerCodeLikeOrCustomerNameLike")
  public Result<List<CustomerEntity>> findByCustomerCodeLikeOrCustomerNameLike(@RequestParam(value = "customerCodeLikeOrNameLike") String customerCodeLikeOrNameLike){
    try {
      return Result.ok(customerService.findByCustomerCodeLikeOrCustomerNameLike(customerCodeLikeOrNameLike));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
}
